<?php
/**
 * ZiziCache – Main Plugin Bootstrap File
 *
 * Plugin Name: ZiziCache
 * Plugin URI: https://zizicache.com
 * Description: Set It. Cache It. Done.
 * Version: 0.5.7-beta.5
 * Requires PHP: 8.1
 * Requires at least: 6.8
 * Author: ZiziCache.com
 * Author URI: https://zizicache.com/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * Bootstrap file: initializes performance monitoring, cache layers,
 * object cache, and optimization modules.
 * Includes Redis Object Cache Drop-In (Till Krüss, GPLv3).
 *
 * @package ZiziCache
 */

defined('ABSPATH') or die('No script kiddies please!');

require_once dirname(__FILE__) . '/vendor/autoload.php';

// Register activation hook to set up security
register_activation_hook(__FILE__, function() {
    // Include SysTool class if not already loaded
    if (!class_exists('\\ZiziCache\\SysTool')) {
        require_once __DIR__ . '/src/SysTool.php';
    }
    
    // Ensure cache directories exist with proper security
    !is_dir(ZIZI_CACHE_CACHE_DIR) && wp_mkdir_p(ZIZI_CACHE_CACHE_DIR);
    
    // Create security .htaccess files for all cache directories
    \ZiziCache\SysTool::ensure_cache_security();
    
    // Create Image Converter statistics table
    if (!class_exists('\\ZiziCache\\ImageConverterStats')) {
        require_once __DIR__ . '/src/ImageConverterStats.php';
    }
    \ZiziCache\ImageConverterStats::createTable();
    
    // Log activation security setup
    if (class_exists('\\ZiziCache\\CacheSys')) {
        \ZiziCache\CacheSys::writeLog('ZiziCache activated - Security .htaccess files created, ImageConverter stats table initialized', 'Security');
    }
});

// Register deactivation hook to clean up third-party files
register_deactivation_hook(__FILE__, function() {
    // Include SysTool class directly if not already loaded
    if (!class_exists('\\ZiziCache\\SysTool')) {
        require_once __DIR__ . '/src/SysTool.php';
    }
    // Force cleanup of third-party files
    \ZiziCache\SysTool::cleanup_third_party_files(true);
});

// Start performance timer for plugin bootstrap
\ZiziCache\PerformanceMonitor::bootstrap_start();

define('ZIZI_CACHE_VERSION', '0.5.7-beta.5');
define('ZIZI_CACHE_FILE', __FILE__);
define('ZIZI_CACHE_FILE_NAME', plugin_basename(__FILE__));
define('ZIZI_CACHE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ZIZI_CACHE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ZIZI_CACHE_CACHE_DIR', WP_CONTENT_DIR . '/cache/zizi-cache/');
define('ZIZI_CACHE_CACHE_URL', WP_CONTENT_URL . '/cache/zizi-cache/');
define('ZIZI_CACHE_LOG_FILE', WP_CONTENT_DIR . '/plugins/zizi-cache/zizi-log.log'); // Unified log file path

// Debug mode - can be overridden in wp-config.php
if (!defined('ZIZI_CACHE_DEBUG')) {
    // Default to false for production, can be enabled in wp-config.php with: define('ZIZI_CACHE_DEBUG', true);
    define('ZIZI_CACHE_DEBUG', false);
}

// Local environment detection - can be overridden in wp-config.php
if (!defined('ZIZI_CACHE_LOCAL_ENV')) {
    // Auto-detection of local environment - localhost or specific domain extensions
    $server_name = isset($_SERVER['SERVER_NAME']) ? strtolower($_SERVER['SERVER_NAME']) : '';
    $remote_addr = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
    define('ZIZI_CACHE_LOCAL_ENV', 
        $server_name === 'localhost' || 
        strpos($server_name, '.local') !== false || 
        strpos($server_name, '.test') !== false || 
        strpos($server_name, '.dev') !== false || 
        $remote_addr === '127.0.0.1' || 
        $remote_addr === '::1'
    );
}

// Load text domain for translations
function zizi_cache_load_textdomain() {
    load_plugin_textdomain('zizi-cache', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'zizi_cache_load_textdomain');

// Initialize AJAX handlers
add_action('init', function() {
    \ZiziCache\AjaxHandler::init();
    \ZiziCache\MediaLibraryIndexer::init(); // Automatic FTP upload detection
});

// Initialize Font Statistics collection (frontend script)
add_action('wp_enqueue_scripts', function() {
    // Only load if font optimization is enabled
    $config = get_option('ZIZI_CACHE_CONFIG', []);
    if (empty($config['fonts_optimize_google_fonts']) || is_admin()) {
        return;
    }
    
    // Enqueue font detector script with debug support
    $plugin_url = plugin_dir_url(__FILE__);
    $debug = defined('WP_DEBUG') && WP_DEBUG;
    $script_file = $debug ? 'font-detector.js' : 'font-detector.min.js';
    $script_url = $plugin_url . 'assets/js/' . $script_file;
    
    wp_enqueue_script(
        'zizicache-font-detector',
        $script_url,
        [],
        ZIZI_CACHE_VERSION . '.' . time(),
        true
    );
    
    // Pass configuration to script
    $debug_enabled = !empty($config['fonts_debug']) ? '1' : '0';
    
    wp_localize_script('zizicache-font-detector', 'ziziCacheConfig', [
        'isAdmin' => current_user_can('manage_options') ? '1' : '0',
        'debugMode' => $debug_enabled,
        'showForAll' => '1',
        'aboveFoldHeight' => 1200,
    ]);
    
    // Debug log for admins
    if (current_user_can('manage_options') && !empty($config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog("Font Detector: Enqueueing script for font statistics collection", 'INFO', 'FontStatistics');
    }
});

// Create cache directory if it doesn't exist and ensure security
if (!is_dir(ZIZI_CACHE_CACHE_DIR)) {
    wp_mkdir_p(ZIZI_CACHE_CACHE_DIR);
    
    // Ensure security .htaccess files are created when cache directory is first created
    if (class_exists('\\ZiziCache\\SysTool')) {
        \ZiziCache\SysTool::ensure_cache_security();
    }
}

// Register preload continuation hook
add_action('zizi_cache_preload_next', ['ZiziCache\Preload', 'preload_available']);

// --- ZiziCache: PreloadSqlite worker ---
require_once __DIR__ . '/src/PreloadSqlite.php';

// Include REST endpoint for system info
require_once __DIR__ . '/vendor/includes/rest-system-info.php';

// --- License System Initialization ---
// Initialize license management only after all core components
require_once __DIR__ . '/src/LicenseManager.php';
require_once __DIR__ . '/src/RestApi/License.php';
require_once __DIR__ . '/src/LemonSqueezyUpdater.php';

// Initialize Media Deletion Handler
require_once __DIR__ . '/src/MediaDeletionHandler.php';

// Initialize REST API for license management
add_action('rest_api_init', function() {
    if (class_exists('ZiziCache\RestApi\License')) {
        ZiziCache\RestApi\License::register();
    }
});

// Initialize auto-updater for licensed users
add_action('init', function() {
    // Early return for non-admin users to avoid unnecessary license checks
    if (!current_user_can('manage_options') && !is_admin()) {
        return;
    }
    
    if (class_exists('ZiziCache\LicenseManager') && 
        class_exists('ZiziCache\LemonSqueezyUpdater') && 
        ZiziCache\LicenseManager::is_active()) {
        
        // Initialize ZiziCache API proxy updater
        new ZiziCache\LemonSqueezyUpdater(
            __FILE__, 
            ZIZI_CACHE_VERSION
        );
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('Auto-updater initialized with ZiziCache API proxy', 'License');
        }
    }
    
    // Auto daily license validation (only for admin users)
    if (class_exists('ZiziCache\LicenseManager') && current_user_can('manage_options')) {
        ZiziCache\LicenseManager::auto_daily_validation();
    }
}, 15); // Later priority to ensure license check works

// Register background license validation cron task
add_action('zizi_cache_background_license_validation', 'ZiziCache\LicenseManager::background_license_validation');

// Secure license table creation on activation
register_activation_hook(__FILE__, 'zizi_cache_create_license_table');

function zizi_cache_create_license_table() {
    global $wpdb;
    $table = $wpdb->prefix . 'zizi_cache_license';
    
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS {$table} (
        id int(11) NOT NULL AUTO_INCREMENT,
        license_key_hash varchar(64) NOT NULL,
        license_key_encrypted longtext NOT NULL,
        instance_id varchar(36) DEFAULT NULL,
        instance_name varchar(255) DEFAULT NULL,
        status enum('active','inactive','expired','disabled') DEFAULT 'inactive',
        activation_limit int(11) DEFAULT 1,
        activation_usage int(11) DEFAULT 0,
        expires_at datetime DEFAULT NULL,
        customer_email varchar(255) DEFAULT NULL,
        customer_name varchar(255) DEFAULT NULL,
        product_name varchar(255) DEFAULT 'ZiziCache',
        product_id varchar(255) DEFAULT NULL,
        variant_id varchar(255) DEFAULT NULL,
        last_validated datetime DEFAULT NULL,
        site_url varchar(255) DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY idx_license_hash (license_key_hash),
        KEY idx_status (status),
        KEY idx_instance_id (instance_id),
        KEY idx_product_id (product_id)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    
    // Add initial options for license system
    add_option('zizi_cache_license_table_version', '2.0');
    
    // Log license table creation
    if (class_exists('\\ZiziCache\\CacheSys')) {
        \ZiziCache\CacheSys::writeLog('License table created/updated successfully', 'License');
    }
}

// Check for license table updates on plugin load
add_action('plugins_loaded', function() {
    $current_version = get_option('zizi_cache_license_table_version', '0');
    if (version_compare($current_version, '2.0', '<')) {
        zizi_cache_create_license_table();
        update_option('zizi_cache_license_table_version', '2.0');
        
        // Log migration
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('License table migrated to version 2.0 (added product_id, variant_id columns)', 'License');
        }
    }
}, 5); // Early hook priority

// Check for plugin version updates and regenerate advanced-cache.php if needed
add_action('plugins_loaded', function() {
    $current_version = get_option('zizi_cache_plugin_version', '0');
    if (version_compare($current_version, ZIZI_CACHE_VERSION, '<')) {
        // Plugin version updated - regenerate advanced-cache.php with new ignore queries
        if (class_exists('\\ZiziCache\\AdvancedCache')) {
            \ZiziCache\AdvancedCache::addAdvancedCache();
            
            // Log regeneration
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('Plugin updated to ' . ZIZI_CACHE_VERSION . ' - Advanced cache regenerated with updated ignore queries (zizi_warmed added)', 'Cache');
            }
        }
        
        // Update stored version
        update_option('zizi_cache_plugin_version', ZIZI_CACHE_VERSION);
    }
}, 6); // After license check

// Core Caching Mechanisms
ZiziCache\WPCache::init();
ZiziCache\AdvancedCache::init();
ZiziCache\ObjectCacheHandler::init();
ZiziCache\CacheSys::init(); // General cache system operations

// Server and Environment Configuration
ZiziCache\Htaccess::init();
ZiziCache\SysConfig::init(); // System configuration and checks
ZiziCache\OPcache::init();   // Register OPcache REST routes and CLI commands
ZiziCache\HostProviders\HostProviders::init(); // Hosting provider specific adjustments

// Database Management
ZiziCache\Database::init();
ZiziCache\DatabaseStatus::init();
ZiziCache\DatabaseMonitor::init(); // Initialize database monitor
ZiziCache\ActionSchedulerCleanup::init();

// Integrations
ZiziCache\Plugins::init(); // General plugin hooks - delegates to PluginManager
ZiziCache\CompatLayer::init(); // Compatibility layer for themes/plugins
ZiziCache\WooCommerceCompatibility::init(); // WooCommerce compatibility layer

// Front-end Optimizations & Features
// Front-end Optimizations & Features
// If the current request is in WP admin and the current user can manage options,
// skip initializing front-end optimization modules to avoid interfering with
// other admin plugins (for example WooCommerce onboarding). This creates a
// safe admin-bypass for optimization features.
// Avoid calling current_user_can() at plugin load time (pluggable functions may not be available yet).
// Use is_admin() as a safe proxy during bootstrap. This prevents fatal errors during early load.

// Detect specific WooCommerce onboarding wizard page and set a permanent bypass
// so the plugin fully avoids initializing optimizations on that page.
// Matches: /wp-admin/admin.php?page=wc-admin&path=/setup-wizard (and variants)
if (!defined('ZIZI_CACHE_ADMIN_BYPASS')) {
    $is_wc_onboarding = false;
    if (isset($_GET['page']) && $_GET['page'] === 'wc-admin' && isset($_GET['path'])) {
        // path may be percent-encoded; normalize
        $path = urldecode($_GET['path']);
        if (strpos($path, '/setup-wizard') === 0 || strpos($path, 'setup-wizard') !== false) {
            $is_wc_onboarding = true;
        }
    }
    if ($is_wc_onboarding) {
        define('ZIZI_CACHE_ADMIN_BYPASS', true);
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('Admin bypass enabled: WooCommerce onboarding detected, skipping front-end optimizations', 'INFO', 'Bootstrap');
        }
    } else {
        define('ZIZI_CACHE_ADMIN_BYPASS', false);
    }
}

$zizi_admin_bypass = is_admin() || (defined('ZIZI_CACHE_ADMIN_BYPASS') && ZIZI_CACHE_ADMIN_BYPASS);

if ( $zizi_admin_bypass ) {
    if ( class_exists('ZiziCache\CacheSys') ) {
        \ZiziCache\CacheSys::writeLog('Admin bypass: Skipping front-end optimization modules for current admin user', 'INFO', 'Bootstrap');
    }
} else {
    // Front-end-only modules. These should NOT run on the WooCommerce onboarding wizard.
    ZiziCache\LcpHandler::init(); // Initialize handler for LCP detection
    ZiziCache\SpeculativeLoading::init();
    ZiziCache\QuicklinkIntegration::init(); // Initialize Quicklink integration
    ZiziCache\AutomaticallyClear::init(); // Automatic cache clearing rules

    // Early Hints HTTP 103 Performance Optimization (Enhanced from competitive analysis)
    if (class_exists('\\ZiziCache\\Core\\EarlyHints')) {
        ZiziCache\Core\EarlyHints::init();
    }

    // Image Converter Module (AVIF/WebP optimization)
    if (class_exists('\\ZiziCache\\ImageConverter')) {
        ZiziCache\ImageConverter::init();
    }
    if (class_exists('\\ZiziCache\\ImageConverterBulk')) {
        ZiziCache\ImageConverterBulk::init();
    }
    if (class_exists('\\ZiziCache\\ImageConverterFrontend')) {
        ZiziCache\ImageConverterFrontend::init();
    }
    if (class_exists('\\ZiziCache\\ImageConverterStats')) {
        ZiziCache\ImageConverterStats::init();
    }

    // Thumbnail Management Module (Prevention & Cleanup)
    if (class_exists('\\ZiziCache\\ImageThumbnailManager')) {
        ZiziCache\ImageThumbnailManager::init();
    }
    if (class_exists('\\ZiziCache\\ImageThumbnailAPI')) {
        ZiziCache\ImageThumbnailAPI::init();
    }

    // Advanced Thumbnail Management Components
    if (class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
        ZiziCache\WooCommerceThumbnailManager::init();
    }
    if (class_exists('\\ZiziCache\\PageBuilderThumbnailDetector')) {
        ZiziCache\PageBuilderThumbnailDetector::init();
    }
    if (class_exists('\\ZiziCache\\AdvancedUsageAnalyzer')) {
        ZiziCache\AdvancedUsageAnalyzer::init();
    }

    // Media Library Management
    if (class_exists('\\ZiziCache\\MediaLibraryIndexer')) {
        ZiziCache\MediaLibraryIndexer::init();
    }
    if (class_exists('\\ZiziCache\\MediaDeletionHandler')) {
        ZiziCache\MediaDeletionHandler::init();
    }
}

// (Front-end-only modules are initialized above inside the !$zizi_admin_bypass branch.)

// Initialize OptiCore always (methods need to be available)
// but output buffering will only start on frontend
ZiziCache\OptiCore::init();

// Admin Area, UI, and API
ZiziCache\AdminBar::init();
ZiziCache\Dashboard::init();
ZiziCache\RestApi::init();

// Developer Tools & Monitoring
ZiziCache\WpCLI::init();
if (is_admin()) {
    ZiziCache\PerformanceMonitor::init(); // Unified Log System & Performance monitoring
}

// Add custom plugin action links
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    // Add Docs link at the beginning
    $docs_link = '<a href="https://zizicache.com/docs/" target="_blank" rel="noopener">' . __('Docs', 'zizi-cache') . '</a>';
    array_unshift($links, $docs_link);
    
    return $links;
});