<?php
/**
 * ZiziCache Uninstall Script
 * 
 * This file is executed when the plugin is deleted via the WordPress admin.
 * It cleans up all plugin data including .htaccess rules.
 * 
 * SECURITY: This file includes multiple security checks to ensure it can only
 * be executed by WordPress administrators during legitimate plugin deletion.
 */

// SECURITY CHECK 1: Exit if called directly or not from WordPress uninstall process
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit('Direct access denied. This file can only be executed by WordPress during plugin uninstall.');
}

// SECURITY CHECK 2: Verify WordPress is loaded and user has admin capabilities
if (!function_exists('current_user_can') || !current_user_can('activate_plugins')) {
    exit('Insufficient permissions. Only administrators can uninstall plugins.');
}

// SECURITY CHECK 3: Verify this is actually our plugin being uninstalled
if (WP_UNINSTALL_PLUGIN !== 'zizi-cache/zizi-cache.php') {
    exit('Invalid plugin context. Uninstall aborted.');
}

// SECURITY CHECK 4: Verify we're in admin context
if (!is_admin()) {
    exit('Uninstall can only be performed from WordPress admin area.');
}

// Log the uninstall attempt for security audit
if (function_exists('error_log')) {
    error_log(sprintf(
        'ZiziCache: Uninstall initiated by user ID %d (%s) at %s',
        get_current_user_id(),
        wp_get_current_user()->user_login ?? 'unknown',
        current_time('mysql')
    ));
}

// SAFE CLEANUP OPERATIONS START HERE
try {
    // Remove .htaccess rules (with safety checks)
    $htaccess_file = ABSPATH . '.htaccess';
    if (file_exists($htaccess_file) && is_writable($htaccess_file)) {
        $htaccess_content = file_get_contents($htaccess_file);
        
        // Only proceed if we actually have ZiziCache rules to remove
        if (preg_match('/# BEGIN ZiziCache.*# END ZiziCache/s', $htaccess_content)) {
            // Create backup before modification
            $backup_content = $htaccess_content;
            
            // Remove ZiziCache section using the same regex as in the main plugin
            $htaccess_content = preg_replace('/# BEGIN ZiziCache.*# END ZiziCache/s', '', $htaccess_content);
            
            // Clean up any extra newlines
            $htaccess_content = preg_replace('/\n{3,}/', "\n\n", $htaccess_content);
            
            // Verify the modification didn't break the file structure
            if (strlen($htaccess_content) > 0 && strlen($htaccess_content) < strlen($backup_content)) {
                file_put_contents($htaccess_file, $htaccess_content);
            } else {
                error_log('ZiziCache: .htaccess modification failed - file restored from backup');
            }
        }    }

    // Remove advanced-cache.php if it was created by our plugin (with safety checks)
    $advanced_cache_file = WP_CONTENT_DIR . '/advanced-cache.php';
    if (file_exists($advanced_cache_file) && is_writable($advanced_cache_file)) {
        $content = file_get_contents($advanced_cache_file);
        // Only remove if it contains our signature AND is small enough to be ours
        if (strpos($content, 'ZiziCache') !== false && strlen($content) < 50000) {
            unlink($advanced_cache_file);
        }
    }

    // Remove cache directory and all cached files (with safety checks)
    $cache_dir = WP_CONTENT_DIR . '/cache/zizi-cache/';
    if (is_dir($cache_dir) && strpos(realpath($cache_dir), realpath(WP_CONTENT_DIR)) === 0) {
        function zizi_cache_remove_directory($dir) {
            // Safety check: ensure we're only deleting within wp-content
            if (strpos(realpath($dir), realpath(WP_CONTENT_DIR)) !== 0) {
                return false;
            }
            
            if (!is_dir($dir)) return false;
            
            $files = array_diff(scandir($dir), array('.', '..'));
            foreach ($files as $file) {
                $path = $dir . '/' . $file;
                if (is_dir($path)) {
                    zizi_cache_remove_directory($path);
                } else {
                    // Additional safety: only delete cache-related files
                    if (pathinfo($file, PATHINFO_EXTENSION) === 'gz' || 
                        pathinfo($file, PATHINFO_EXTENSION) === 'html' ||
                        $file === '.htaccess' || 
                        $file === 'index.php') {
                        unlink($path);
                    }
                }
            }
            return rmdir($dir);
        }
        
        zizi_cache_remove_directory($cache_dir);
    }

    // Remove database options (with safety checks)
    if (function_exists('delete_option')) {
        delete_option('zizi_cache_config');
        delete_option('zizi_cache_license_data');
    }    // Remove license table if it exists (with safety checks)
    global $wpdb;
    if (isset($wpdb) && is_object($wpdb)) {
        $table_name = $wpdb->prefix . 'zizi_cache_licenses';
        // Verify table name contains our prefix and is safe before dropping
        if (strpos($table_name, 'zizi_cache_licenses') !== false && 
            preg_match('/^[a-zA-Z0-9_]+$/', $table_name)) {
            $wpdb->query("DROP TABLE IF EXISTS `{$table_name}`");
        }
    }

} catch (Exception $e) {
    // Log any errors during uninstall
    if (function_exists('error_log')) {
        error_log('ZiziCache uninstall error: ' . $e->getMessage());
    }
} catch (Error $e) {
    // Log fatal errors during uninstall
    if (function_exists('error_log')) {
        error_log('ZiziCache uninstall fatal error: ' . $e->getMessage());
    }
}

// Final security log
if (function_exists('error_log')) {
    error_log('ZiziCache: Uninstall completed successfully');
}
