/**
 * Translation helper functions
 * 
 * This file provides translation utilities for the ZiziCache admin interface.
 * When FORCE_ENGLISH is true (set in webpack.config.js), all text will be in English
 * regardless of the WordPress site language.
 */

import { __ as wpI18n } from '@wordpress/i18n';

/**
 * Translate text, honoring the FORCE_ENGLISH setting
 * 
 * @param {string} text - Text to translate
 * @param {string} domain - Text domain (optional)
 * @return {string} Translated text or original if FORCE_ENGLISH is true
 */
export const __ = (text, domain) => {
  console.log('Translation requested for:', text);
  console.log('FORCE_ENGLISH value:', typeof FORCE_ENGLISH !== 'undefined' ? FORCE_ENGLISH : 'undefined');
  
  // FORCE_ENGLISH is always true - hardcoded for now to guarantee English
  return text;
  
  // Original dynamic logic:
  /*
  if (typeof FORCE_ENGLISH !== 'undefined' && FORCE_ENGLISH === true) {
    return text;
  }
  return wpI18n(text, domain);
  */
};

/**
 * English to Czech translation mapping for static content
 * Only used when hard-coded Czech text is present in components
 */
const translations = {
  // Navigation
  'Pokročilá nastavení': 'Advanced Settings',
  'Rozšířené možnosti a experimentální funkce': 'Advanced options and experimental features',
  'Vypnuto': 'Off',
  'Speculative Loading (přednačítání/prerender)': 'Speculative Loading',
  'Režim:': 'Mode:',
  'Prefetch (přednačtení)': 'Prefetch',
  'Prerender (předvykreslení)': 'Prerender',
  'Zvolte, zda chcete stránky pouze přednačítat (prefetch), předvykreslovat (prerender), nebo funkci vypnout.': 
    'Choose whether to prefetch pages, prerender them, or turn off this feature.',
  'Eagerness (agresivita):': 'Eagerness:',
  'Auto (dle WP)': 'Auto (WP default)',
  'Conservative (nejméně agresivní)': 'Conservative (least aggressive)',
  'Moderate (doporučeno)': 'Moderate (recommended)',
  'Eager (nejagresivnější)': 'Eager (most aggressive)',
  'Určuje, jak agresivně bude prohlížeč přednačítat/prerenderovat stránky. Doporučujeme "Moderate".': 
    'Determines how aggressively the browser will prefetch/prerender pages. We recommend "Moderate".',
  'Vyloučit URL patterny:': 'Exclude URL patterns:',
  'Zadané patterny budou vyloučeny z prefetch/prerender. Podporuje hvězdičku (*).': 
    'The specified patterns will be excluded from prefetch/prerender. Supports wildcards (*).',
  'Podmíněné spouštění': 'Conditional execution',
  'Aktivovat spekulativní načítání pouze na vybraných typech stránek': 
    'Activate speculative loading only on selected page types',
  // Add more translations as needed
};

/**
 * Force Czech text to English regardless of WordPress language
 * Use this for components with hardcoded Czech text
 * 
 * @param {string} czechText - The Czech text to convert
 * @return {string} English text if available, or the original Czech text
 */
export const forceEnglish = (czechText) => {
  if (!FORCE_ENGLISH) return czechText;
  
  return translations[czechText] || czechText;
};

export default {
  __,
  forceEnglish
};
