import { ref } from 'vue'

/**
 * Composable pro optimalizovanou API komunikaci
 * Debouncing, error handling a change tracking integration
 */
export function useOptimizedApi(apiCallFunction, debounceMs = 500) {
  const isLoading = ref(false)
  const error = ref(null)
  const lastSaveTime = ref(null)
  
  let debounceTimer = null

  /**
   * Optimalizované uložení - pouze změněné hodnoty s debouncing
   */
  const saveChanges = async (getChangedValues, markAsSaved) => {
    return new Promise((resolve, reject) => {
      // Clear previous timer
      if (debounceTimer) {
        clearTimeout(debounceTimer)
      }

      debounceTimer = setTimeout(async () => {
        const changedValues = getChangedValues()
        
        // Nic se nezměnilo
        if (Object.keys(changedValues).length === 0) {
          resolve({ success: true, message: 'No changes to save' })
          return
        }

        isLoading.value = true
        error.value = null

        try {
          console.log('Saving only changed values:', changedValues)
          const result = await apiCallFunction(changedValues)
          markAsSaved(result)
          lastSaveTime.value = new Date()
          resolve(result)
        } catch (err) {
          error.value = err.message || 'Save failed'
          reject(err)
        } finally {
          isLoading.value = false
        }
      }, debounceMs)
    })
  }

  /**
   * Okamžité uložení bez debouncing (pro tlačítka) - optimalizovaná verze
   */
  const saveImmediately = async (getChangedValues, markAsSaved) => {
    if (debounceTimer) {
      clearTimeout(debounceTimer)
    }
    
    const changedValues = getChangedValues()
    if (Object.keys(changedValues).length === 0) {
      console.log('useOptimizedApi: No changes to save');
      return { success: true, message: 'No changes to save' }
    }

    isLoading.value = true
    error.value = null

    try {
      console.log('useOptimizedApi: Immediate save - changed values:', changedValues);
      
      // Volání API funkce
      const result = await apiCallFunction(changedValues);
      console.log('useOptimizedApi: API response received:', result);
      
      // Zpracování výsledku - normalizace různých formátů odpovědi
      let processedResult;
      if (result && result.optimized) {
        // Nový optimalizovaný formát odpovědi
        processedResult = result.data;
        console.log('useOptimizedApi: Using optimized result format', processedResult);
      } else {
        // Starý formát odpovědi nebo jiný formát
        processedResult = result;
        console.log('useOptimizedApi: Using standard result format', processedResult);
      }
      
      // Označení změn jako uložených
      markAsSaved(processedResult);
      lastSaveTime.value = new Date();
      
      // Vrácení původní odpovědi pro další zpracování
      return result;
    } catch (err) {
      console.error('useOptimizedApi: Save failed', err);
      error.value = err.message || 'Save failed';
      throw err;
    } finally {
      isLoading.value = false;
    }
  }

  /**
   * Zruší pending save operaci
   */
  const cancelPendingSave = () => {
    if (debounceTimer) {
      clearTimeout(debounceTimer)
      debounceTimer = null
    }
  }

  return {
    isLoading,
    error,
    lastSaveTime,
    saveChanges,
    saveImmediately,
    cancelPendingSave
  }
}
