import { ref, reactive, watch } from 'vue'

/**
 * Composable pro sledování změn v konfiguraci
 * Umožňuje posílat na server pouze změněné hodnoty místo celé konfigurace
 */
export function useChangeTracking(initialConfig = {}) {
  const originalConfig = reactive({ ...initialConfig })
  const currentConfig = reactive({ ...initialConfig })
  const changedFields = ref(new Set())

  // Sleduje změny a označuje pole
  watch(currentConfig, (newVal, oldVal) => {
    Object.keys(newVal).forEach(key => {
      if (newVal[key] !== originalConfig[key]) {
        changedFields.value.add(key)
      } else {
        changedFields.value.delete(key)
      }
    })
  }, { deep: true })

  /**
   * Vrátí pouze změněné hodnoty
   */
  const getChangedValues = () => {
    const changed = {}
    changedFields.value.forEach(field => {
      changed[field] = currentConfig[field]
    })
    return changed
  }

  /**
   * Označí konfiguraci jako uloženou (resetuje tracking)
   */
  const markAsSaved = (newConfig = null) => {
    if (newConfig) {
      Object.assign(originalConfig, newConfig)
      Object.assign(currentConfig, newConfig)
    } else {
      Object.assign(originalConfig, currentConfig)
    }
    changedFields.value.clear()
  }

  /**
   * Aktualizuje konfiguraci
   */
  const updateConfig = (newConfig) => {
    Object.assign(currentConfig, newConfig)
  }

  /**
   * Má komponenta nějaké neuložené změny?
   */
  const hasChanges = () => changedFields.value.size > 0

  return {
    currentConfig,
    getChangedValues,
    markAsSaved,
    updateConfig,
    hasChanges,
    changedFields: changedFields.value
  }
}
