/**
 * Composable for cached API calls with memoization
 * Implements caching strategy for expensive computations
 * According to Phase 4.2 of optimization plan
 */
import { ref, computed, reactive } from 'vue'

// Global cache for API responses
const apiCache = reactive(new Map())
const cacheTimestamps = reactive(new Map())

// Cache TTL in milliseconds (5 minutes default)
const CACHE_TTL = 5 * 60 * 1000

export function useCachedApi(endpoint, options = {}) {
  const {
    ttl = CACHE_TTL,
    cacheKey = endpoint,
    transform = (data) => data
  } = options
  
  const data = ref(null)
  const isLoading = ref(false)
  const error = ref(null)
  
  // Memoized cache check - expensive computation
  const isCacheValid = computed(() => {
    if (!apiCache.has(cacheKey)) return false
    
    const timestamp = cacheTimestamps.get(cacheKey)
    const now = Date.now()
    
    return (now - timestamp) < ttl
  })
  
  // Cached data getter with memoization
  const cachedData = computed(() => {
    if (isCacheValid.value) {
      return transform(apiCache.get(cacheKey))
    }
    return null
  })
  
  // Fetch data with caching
  const fetchData = async (payload = null, method = 'GET') => {
    // Return cached data if valid
    if (isCacheValid.value && method === 'GET') {
      data.value = cachedData.value
      return data.value
    }
    
    isLoading.value = true
    error.value = null
    
    try {
      const url = endpoint.startsWith('/') ? endpoint : `/wp-json/zizi-cache/v1/${endpoint}`
      
      const fetchOptions = {
        method,
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': window.ziziCacheData?.nonce || ''
        }
      }
      
      if (payload && method !== 'GET') {
        fetchOptions.body = JSON.stringify(payload)
      }
      
      const response = await fetch(url, fetchOptions)
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`)
      }
      
      const result = await response.json()
      const transformedData = transform(result)
      
      // Cache successful responses for GET requests
      if (method === 'GET') {
        apiCache.set(cacheKey, result)
        cacheTimestamps.set(cacheKey, Date.now())
      }
      
      data.value = transformedData
      return transformedData
      
    } catch (err) {
      error.value = err
      console.error(`API Error for ${endpoint}:`, err)
      throw err
    } finally {
      isLoading.value = false
    }
  }
  
  // Invalidate cache for specific key
  const invalidateCache = (key = cacheKey) => {
    apiCache.delete(key)
    cacheTimestamps.delete(key)
  }
  
  // Clear all cache
  const clearAllCache = () => {
    apiCache.clear()
    cacheTimestamps.clear()
  }
  
  // Preload data if cache is empty
  const preloadData = async () => {
    if (!isCacheValid.value) {
      await fetchData()
    }
  }
  
  return {
    data,
    isLoading,
    error,
    fetchData,
    invalidateCache,
    clearAllCache,
    preloadData,
    isCacheValid,
    cachedData
  }
}

// Composable for memoized expensive computations
export function useMemoizedComputation(computation, dependencies = []) {
  const memoCache = reactive(new Map())
  
  const memoizedResult = computed(() => {
    // Create cache key from dependencies
    const cacheKey = JSON.stringify(dependencies.map(dep => 
      typeof dep === 'function' ? dep() : dep
    ))
    
    // Return cached result if available
    if (memoCache.has(cacheKey)) {
      return memoCache.get(cacheKey)
    }
    
    // Compute and cache result
    const result = computation()
    memoCache.set(cacheKey, result)
    
    // Limit cache size to prevent memory leaks
    if (memoCache.size > 100) {
      const firstKey = memoCache.keys().next().value
      memoCache.delete(firstKey)
    }
    
    return result
  })
  
  const clearMemoCache = () => {
    memoCache.clear()
  }
  
  return {
    memoizedResult,
    clearMemoCache
  }
}
