<?php
/**
 * AJAX endpoint to retrieve font statistics
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Verifice AJAX nonce
check_ajax_referer('zizi-cache-ajax-nonce', 'security');

// Check user permissions
if (!current_user_can('manage_options')) {
    wp_send_json_error(['message' => 'Permission denied.'], 403);
}

// Get font statistics
$font_stats = get_font_statistics();

// Return the statistics
wp_send_json_success(['data' => $font_stats]);

/**
 * Get statistics about fonts across the site
 *
 * @return array Statistics about fonts
 */
function get_font_statistics() {
    global $wpdb;
    
    $options = $wpdb->get_results("
        SELECT option_name, option_value
        FROM {$wpdb->options} 
        WHERE option_name LIKE 'zizi_cache_font_data_%'
    ");
    
    if (empty($options)) {
        return [];
    }
    
    // Agregace dat
    $font_stats = [];
    
    foreach ($options as $option) {
        $data = maybe_unserialize($option->option_value);
        if (!$data || empty($data['fonts']) || (isset($data['updated']) && (time() - $data['updated']) > 604800)) {
            continue; // Skip expired or invalid data (7 days)
        }
        
        foreach ($data['fonts'] as $font) {
            if (empty($font['family'])) continue;
            
            $family = $font['family'];
            
            if (!isset($font_stats[$family])) {
                $font_stats[$family] = [
                    'family' => $family,
                    'pages_count' => 1,
                    'weights' => isset($font['weight']) ? [$font['weight']] : [],
                    'isHeading' => isset($font['isHeading']) ? $font['isHeading'] : false,
                    'isBody' => isset($font['isHeading']) ? !$font['isHeading'] : true
                ];
            } else {
                $font_stats[$family]['pages_count']++;
                if (isset($font['weight']) && !in_array($font['weight'], $font_stats[$family]['weights'])) {
                    $font_stats[$family]['weights'][] = $font['weight'];
                }
                if (isset($font['isHeading'])) {
                    $font_stats[$family]['isHeading'] = $font_stats[$family]['isHeading'] || $font['isHeading'];
                    $font_stats[$family]['isBody'] = $font_stats[$family]['isBody'] || !$font['isHeading'];
                }
            }
        }
    }
    
    // Seřazení podle počtu stránek
    usort($font_stats, function($a, $b) {
        return $b['pages_count'] - $a['pages_count'];
    });
    
    // Convert to indexed array and limit to top 10
    return array_values(array_slice($font_stats, 0, 10));
}
