<?php

namespace ZiziCache;

/**
 * WordPress CLI Commands for ZiziCache
 * 
 * This class provides a CLI interface for managing ZiziCache using WP-CLI.
 */
class WpCLI
{
    /**
     * Definition of available commands
     */
    const CMD_PRELOAD_CACHE = 'preload-cache';
    const CMD_PURGE_PAGES = 'purge-pages';
    const CMD_PURGE_PAGES_AND_PRELOAD = 'purge-pages-and-preload';
    const CMD_PURGE_EVERYTHING = 'purge-everything';
    const CMD_PURGE_EVERYTHING_AND_PRELOAD = 'purge-everything-and-preload';
    const CMD_ACTIVATE_LICENSE = 'activate-license';
    const CMD_REDIS_FLUSH = 'redis-flush';
    const CMD_OBJECT_CACHE_FLUSH = 'object-cache-flush';
    const CMD_OPCACHE_FLUSH = 'opcache-flush';
    const CMD_OPCACHE_FLUSH_ALT = 'opcache/flush';
    const CMD_QUICKLINK_ENABLE = 'quicklink-enable';
    const CMD_QUICKLINK_DISABLE = 'quicklink-disable';
    const CMD_QUICKLINK_STATUS = 'quicklink-status';
    const CMD_QUICKLINK_CONFIGURE = 'quicklink-configure';

    /**
     * Array of all available commands for easier management and documentation
     */
    const AVAILABLE_COMMANDS = [
        self::CMD_PRELOAD_CACHE,
        self::CMD_PURGE_PAGES,
        self::CMD_PURGE_PAGES_AND_PRELOAD,
        self::CMD_PURGE_EVERYTHING,
        self::CMD_PURGE_EVERYTHING_AND_PRELOAD,
        self::CMD_ACTIVATE_LICENSE,
        self::CMD_REDIS_FLUSH,
        self::CMD_OBJECT_CACHE_FLUSH,
        self::CMD_OPCACHE_FLUSH,
        self::CMD_OPCACHE_FLUSH_ALT,
        self::CMD_QUICKLINK_ENABLE,
        self::CMD_QUICKLINK_DISABLE,
        self::CMD_QUICKLINK_STATUS,
        self::CMD_QUICKLINK_CONFIGURE
    ];

    /**
     * Initialize WP-CLI commands
     *
     * @return void
     */
    public static function init()
    {
        if (defined('WP_CLI') && WP_CLI) {
            \WP_CLI::add_command('zizi-cache', [__CLASS__, 'sub_commands']);
        }
    }

    /**
     * Process WP-CLI subcommands
     *
     * @param array $args        Array of command line arguments
     * @param array $assoc_args  Array of associative arguments from command line
     * @return void
     */
    public static function sub_commands($args, $assoc_args)
    {
        if (empty($args)) {
            self::showUsageError();
            return;
        }

        $subcommand = $args[0];

        // Check if the command is valid
        if (!in_array($subcommand, self::AVAILABLE_COMMANDS)) {
            self::showInvalidCommandError();
            return;
        }

        // Check for all commands except license activation
        if ($subcommand !== self::CMD_ACTIVATE_LICENSE) {
            self::checkCacheDirectoryPermissions();
        }

        // Process specific command
        try {
            switch ($subcommand) {
                case self::CMD_PRELOAD_CACHE:
                    self::handlePreloadCache();
                    break;

                case self::CMD_PURGE_PAGES:
                    self::handlePurgePages();
                    break;

                case self::CMD_PURGE_PAGES_AND_PRELOAD:
                    self::handlePurgePagesAndPreload();
                    break;

                case self::CMD_PURGE_EVERYTHING:
                    self::handlePurgeEverything();
                    break;

                case self::CMD_PURGE_EVERYTHING_AND_PRELOAD:
                    self::handlePurgeEverythingAndPreload();
                    break;

                case self::CMD_ACTIVATE_LICENSE:
                    if (empty($args[1])) {
                        \WP_CLI::error('Please enter a license key.');
                        return;
                    }
                    self::handleActivateLicense($args[1]);
                    break;
                    
                case self::CMD_REDIS_FLUSH:
                case self::CMD_OBJECT_CACHE_FLUSH:
                    self::handleObjectCacheFlush();
                    break;
                    
                case self::CMD_OPCACHE_FLUSH:
                case self::CMD_OPCACHE_FLUSH_ALT:
                    self::handleOpcacheFlush();
                    break;
                
                case self::CMD_QUICKLINK_ENABLE:
                    self::handleQuicklinkEnable($assoc_args);
                    break;
                
                case self::CMD_QUICKLINK_DISABLE:
                    self::handleQuicklinkDisable();
                    break;
                
                case self::CMD_QUICKLINK_STATUS:
                    self::handleQuicklinkStatus();
                    break;
                
                case self::CMD_QUICKLINK_CONFIGURE:
                    self::handleQuicklinkConfigure($assoc_args);
                    break;
            }
        } catch (\Exception $e) {
            \WP_CLI::error(
                sprintf('Error executing command %s: %s', $subcommand, $e->getMessage())
            );
        }
    }

    /**
     * Show missing subcommand error and help
     *
     * @return void
     */
    /**
     * Show usage error with available commands
     *
     * @return void
     */
    private static function showUsageError()
    {
        \WP_CLI::log('ZiziCache CLI Commands:');
        \WP_CLI::log('');
        
        \WP_CLI::log('=== Cache Management ===');
        \WP_CLI::log('  wp zizi-cache preload-cache               Preload the page cache');
        \WP_CLI::log('  wp zizi-cache purge-pages                 Purge specific pages from cache');
        \WP_CLI::log('  wp zizi-cache purge-pages-and-preload     Purge specific pages and preload them');
        \WP_CLI::log('  wp zizi-cache purge-everything            Purge all caches');
        \WP_CLI::log('  wp zizi-cache purge-everything-and-preload Purge all caches and preload');
        
        \WP_CLI::log('');
        \WP_CLI::log('=== Object Cache Management ===');
        \WP_CLI::log('  wp zizi-cache redis-flush                 Flush Redis cache');
        \WP_CLI::log('  wp zizi-cache object-cache-flush          Flush object cache');
        
        \WP_CLI::log('');
        \WP_CLI::log('=== OPcache Management ===');
        \WP_CLI::log('  wp zizi-cache opcache-flush               Flush OPcache');
        \WP_CLI::log('  wp zizi-cache opcache/flush               Alternative command for OPcache flush');
        
        \WP_CLI::log('');
        \WP_CLI::log('=== Quicklink Management ===');
        \WP_CLI::log('  wp zizi-cache quicklink-enable            Enable Quicklink functionality');
        \WP_CLI::log('    Options:');
        \WP_CLI::log('      --mode=<prefetch|prerender|both>      Set Quicklink mode');
        \WP_CLI::log('      --conditional=<true|false>            Enable conditional execution');
        \WP_CLI::log('      --page-types=<types>                  Comma-separated page types');
        \WP_CLI::log('');
        \WP_CLI::log('  wp zizi-cache quicklink-disable           Disable Quicklink functionality');
        \WP_CLI::log('');
        \WP_CLI::log('  wp zizi-cache quicklink-status            Show Quicklink status and configuration');
        \WP_CLI::log('');
        \WP_CLI::log('  wp zizi-cache quicklink-configure         Configure Quicklink settings');
        \WP_CLI::log('    Options:');
        \WP_CLI::log('      --mode=<mode>                         Set mode (prefetch, prerender, both)');
        \WP_CLI::log('      --throttle=<number>                   Set concurrent requests limit');
        \WP_CLI::log('      --limit=<number>                      Set total requests limit');
        \WP_CLI::log('      --delay=<ms>                          Set delay before prefetch');
        \WP_CLI::log('      --timeout=<ms>                        Set idle timeout');
        \WP_CLI::log('      --threshold=<0-100>                   Set viewport threshold percentage');
        \WP_CLI::log('      --priority=<true|false>               Enable/disable high priority');
        \WP_CLI::log('      --respect-data-saver=<true|false>     Enable/disable data saver respect');
        \WP_CLI::log('      --min-connection=<type>               Set minimum connection type');
        \WP_CLI::log('      --block-ads=<true|false>              Enable/disable ad blocking');
        \WP_CLI::log('      --conditional=<true|false>            Enable/disable conditional execution');
        \WP_CLI::log('      --page-types=<types>                  Set page types (comma-separated)');
        \WP_CLI::log('      --origins=<domains>                   Set allowed origins (comma-separated)');
        \WP_CLI::log('      --ignores=<patterns>                  Set ignore patterns (comma-separated)');
        
        \WP_CLI::log('');
        \WP_CLI::log('=== License Management ===');
        \WP_CLI::log('  wp zizi-cache activate-license            Activate license');
        
        \WP_CLI::error('Please enter a valid subcommand');
    }

    /**
     * Show invalid command error
     *
     * @return void
     */
    private static function showInvalidCommandError()
    {
        $available_commands = implode(', ', self::AVAILABLE_COMMANDS);
        \WP_CLI::error(
            sprintf('Invalid subcommand. Please use one of: %s', $available_commands)
        );
    }

    /**
     * Check write permissions for cache directory
     *
     * @return void
     * @throws \Exception If cache directory is not writable
     */
    private static function checkCacheDirectoryPermissions()
    {
        if (!is_writable(ZIZI_CACHE_CACHE_DIR)) {
            throw new \Exception(
                'Cannot write to wp-content/cache directory. Please check directory permissions and try again.'
            );
        }
    }

    /**
     * Process preload-cache command
     *
     * @return void
     * @throws \Exception If there's an error loading cache
     */
    private static function handlePreloadCache()
    {
        \WP_CLI::log('Starting cache preloading...');
        Preload::preload_cache();
        \WP_CLI::success('Cache has been successfully preloaded.');
    }

    /**
     * Process purge-pages command
     *
     * @return void
     * @throws \Exception If there's an error purging pages
     */
    private static function handlePurgePages()
    {
        \WP_CLI::log('Purging page cache...');
        Purge::purge_pages();
        \WP_CLI::success('Pages have been successfully purged from cache.');
    }

    /**
     * Process purge-pages-and-preload command
     *
     * @return void
     * @throws \Exception If there's an error purging or loading
     */
    private static function handlePurgePagesAndPreload()
    {
        \WP_CLI::log('Purging page cache...');
        Purge::purge_pages();
        \WP_CLI::log('Starting cache preloading...');
        Preload::preload_cache();
        \WP_CLI::success('Pages have been purged and cache has been successfully reloaded.');
    }

    /**
     * Process purge-everything command
     *
     * @return void
     * @throws \Exception If there's an error during complete cleanup
     */
    private static function handlePurgeEverything()
    {
        \WP_CLI::log('Purging all cache...');
        Purge::purge_everything();
        \WP_CLI::success('All cache has been successfully purged.');
    }

    /**
     * Process purge-everything-and-preload command
     *
     * @return void
     * @throws \Exception If there's an error during cleanup or loading
     */
    private static function handlePurgeEverythingAndPreload()
    {
        \WP_CLI::log('Purging all cache...');
        Purge::purge_everything();
        \WP_CLI::log('Starting cache preloading...');
        Preload::preload_cache();
        \WP_CLI::success('All cache has been purged and successfully reloaded.');
    }

    /**
     * Process activate-license command
     *
     * @param string $license_key License key to activate
     * @return void
     * @throws \Exception If there's an error activating the license
     */
    private static function handleActivateLicense($license_key)
    {
        \WP_CLI::log('Activating ZiziCache license...');
        License::activate_license($license_key);
        \WP_CLI::success('ZiziCache license has been successfully activated.');
    }

    /**
     * Process redis-flush / object-cache-flush command
     * Clears object cache (Redis or standard WordPress object cache)
     *
     * @return void
     * @throws \Exception If there's an error clearing object cache
     */
    private static function handleObjectCacheFlush()
    {
        \WP_CLI::log('Flushing object cache...');
        
        if (class_exists('ZiziCache\\ObjectCacheHandler')) {
            \ZiziCache\ObjectCacheHandler::flush();
        } elseif (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
        
        \WP_CLI::success('Object cache has been successfully flushed.');
    }

    /**
     * Process opcache-flush / opcache/flush command
     * Flushes OPcache
     *
     * @return void
     * @throws \Exception If there is an error flushing OPcache
     */
    private static function handleOpcacheFlush()
    {
        \WP_CLI::log('Flushing OPcache...');
        
        if (class_exists('ZiziCache\\OPcache')) {
            \ZiziCache\OPcache::flush();
        } elseif (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        \WP_CLI::success('OPcache has been successfully flushed.');
    }
    
    /**
     * Enable Quicklink functionality
     * 
     * @param array $assoc_args CLI arguments
     * @return void
     */
    private static function handleQuicklinkEnable($assoc_args)
    {
        $config = SysConfig::$config;
        $config['quicklink_enabled'] = true;
        
        // Handle mode parameter if provided
        if (isset($assoc_args['mode'])) {
            $mode = $assoc_args['mode'];
            if (in_array($mode, ['prefetch', 'prerender', 'both'])) {
                $config['quicklink_mode'] = $mode;
                \WP_CLI::log(sprintf("Quicklink mode set to '%s'", $mode));
            } else {
                \WP_CLI::warning("Invalid mode. Using default mode 'prefetch'");
                $config['quicklink_mode'] = 'prefetch';
            }
        }
        
        // Check for conditional parameter
        if (isset($assoc_args['conditional']) && $assoc_args['conditional'] === 'true') {
            $config['quicklink_conditional'] = true;
            
            // Set page types if provided
            if (isset($assoc_args['page-types'])) {
                $page_types = explode(',', $assoc_args['page-types']);
                $valid_types = ['home', 'front_page', 'singular', 'archive', 'search'];
                $config['quicklink_page_types'] = array_intersect($page_types, $valid_types);
                
                if (empty($config['quicklink_page_types'])) {
                    \WP_CLI::warning("No valid page types provided. Quicklink will run on all pages.");
                    $config['quicklink_conditional'] = false;
                } else {
                    \WP_CLI::log("Quicklink will run only on: " . implode(', ', $config['quicklink_page_types']));
                }
            } else {
                \WP_CLI::warning("No page types specified for conditional mode. Quicklink will run on all pages.");
                $config['quicklink_conditional'] = false;
            }
        }
        
        // Update configuration
        SysConfig::update_config($config);
        
        // Verify the update
        $updated_config = SysConfig::$config;
        if ($updated_config['quicklink_enabled']) {
            \WP_CLI::success("Quicklink has been enabled successfully.");
            
            // Check if Speculation Rules API is also enabled
            if (!empty($updated_config['speculative_loading_mode']) && $updated_config['speculative_loading_mode'] !== 'off') {
                \WP_CLI::warning("Both Quicklink and Speculation Rules API are now active. This may cause duplicate prefetching.");
                \WP_CLI::log("Consider using different modes or page types for each system to optimize performance.");
            }
        } else {
            \WP_CLI::error("Failed to enable Quicklink.");
        }
    }
    
    /**
     * Disable Quicklink functionality
     * 
     * @return void
     */
    private static function handleQuicklinkDisable()
    {
        $config = SysConfig::$config;
        $config['quicklink_enabled'] = false;
        
        // Update configuration
        SysConfig::update_config($config);
        
        // Verify the update
        $updated_config = SysConfig::$config;
        if (!$updated_config['quicklink_enabled']) {
            \WP_CLI::success("Quicklink has been disabled successfully.");
        } else {
            \WP_CLI::error("Failed to disable Quicklink.");
        }
    }
    
    /**
     * Display Quicklink status and configuration
     * 
     * @return void
     */
    private static function handleQuicklinkStatus()
    {
        $config = SysConfig::$config;
        
        \WP_CLI::log("=== Quicklink Status ===");
        \WP_CLI::log("Enabled: " . ($config['quicklink_enabled'] ? 'Yes' : 'No'));
        
        if ($config['quicklink_enabled']) {
            \WP_CLI::log("Mode: " . $config['quicklink_mode']);
            \WP_CLI::log("Concurrent requests limit: " . $config['quicklink_throttle']);
            \WP_CLI::log("Total requests limit: " . $config['quicklink_limit']);
            \WP_CLI::log("Delay: " . $config['quicklink_delay'] . "ms");
            \WP_CLI::log("Timeout: " . $config['quicklink_timeout'] . "ms");
            \WP_CLI::log("Priority mode: " . ($config['quicklink_priority'] ? 'Enabled' : 'Disabled'));
            \WP_CLI::log("Respect data saver: " . ($config['quicklink_respect_data_saver'] ? 'Yes' : 'No'));
            \WP_CLI::log("Min connection type: " . ($config['quicklink_min_connection'] ?? 'Any'));
            \WP_CLI::log("Block ads: " . ($config['quicklink_block_ads'] ? 'Yes' : 'No'));
            
            // Check conditional execution
            \WP_CLI::log("Conditional execution: " . ($config['quicklink_conditional'] ? 'Yes' : 'No'));
            if ($config['quicklink_conditional'] && !empty($config['quicklink_page_types'])) {
                \WP_CLI::log("Page types: " . implode(', ', $config['quicklink_page_types']));
            }
            
            // Check coexistence with Speculation Rules
            $speculation_active = !empty($config['speculative_loading_mode']) && $config['speculative_loading_mode'] !== 'off';
            \WP_CLI::log("Coexistence with Speculation Rules: " . ($speculation_active ? 'Active' : 'Not active'));
            
            if ($speculation_active) {
                \WP_CLI::log("Speculation Rules mode: " . $config['speculative_loading_mode']);
                
                // Check for potential conflicts
                if ($config['speculative_loading_mode'] === $config['quicklink_mode']) {
                    \WP_CLI::warning("Both systems are using the same mode '{$config['quicklink_mode']}'. Consider using different modes for optimal performance.");
                }
                
                // Check page type overlap
                if ($config['quicklink_conditional'] && !empty($config['speculative_loading_conditional'])) {
                    $quicklink_pages = $config['quicklink_page_types'] ?? [];
                    $speculation_pages = $config['speculative_loading_page_types'] ?? [];
                    $overlap = array_intersect($quicklink_pages, $speculation_pages);
                    
                    if (!empty($overlap)) {
                        \WP_CLI::warning("Page type overlap detected: " . implode(', ', $overlap) . ". This may cause duplicate prefetching.");
                    }
                }
            }
        }
    }
    
    /**
     * Configure Quicklink settings
     * 
     * @param array $assoc_args CLI arguments
     * @return void
     */
    private static function handleQuicklinkConfigure($assoc_args)
    {
        $config = SysConfig::$config;
        $updated = false;
        
        // Map CLI arguments to config keys
        $mapping = [
            'mode' => 'quicklink_mode',
            'throttle' => 'quicklink_throttle',
            'limit' => 'quicklink_limit',
            'delay' => 'quicklink_delay',
            'timeout' => 'quicklink_timeout',
            'threshold' => 'quicklink_threshold',
            'priority' => 'quicklink_priority',
            'respect-data-saver' => 'quicklink_respect_data_saver',
            'min-connection' => 'quicklink_min_connection',
            'block-ads' => 'quicklink_block_ads',
            'conditional' => 'quicklink_conditional',
        ];
        
        // Special handling for boolean values
        $boolean_keys = ['priority', 'respect-data-saver', 'block-ads', 'conditional'];
        
        foreach ($mapping as $cli_key => $config_key) {
            if (isset($assoc_args[$cli_key])) {
                $value = $assoc_args[$cli_key];
                
                // Convert boolean strings
                if (in_array($cli_key, $boolean_keys)) {
                    $value = ($value === 'true' || $value === '1');
                }
                
                // Convert numeric values
                if (in_array($cli_key, ['throttle', 'limit', 'delay', 'timeout', 'threshold']) && is_numeric($value)) {
                    $value = (int) $value;
                }
                
                // Validate mode
                if ($cli_key === 'mode' && !in_array($value, ['prefetch', 'prerender', 'both'])) {
                    \WP_CLI::warning("Invalid mode '{$value}'. Using default mode 'prefetch'.");
                    $value = 'prefetch';
                }
                
                // Validate min-connection
                if ($cli_key === 'min-connection' && !in_array($value, ['5g', '4g', '3g', 'any'])) {
                    \WP_CLI::warning("Invalid min-connection value '{$value}'. Using default '3g'.");
                    $value = '3g';
                }
                
                // Update config
                $config[$config_key] = $value;
                $updated = true;
                
                \WP_CLI::log("Updated {$cli_key} to: " . (is_bool($value) ? ($value ? 'true' : 'false') : $value));
            }
        }
        
        // Handle page types separately
        if (isset($assoc_args['page-types'])) {
            $page_types = explode(',', $assoc_args['page-types']);
            $valid_types = ['home', 'front_page', 'singular', 'archive', 'search'];
            $config['quicklink_page_types'] = array_intersect($page_types, $valid_types);
            
            if (empty($config['quicklink_page_types']) && $config['quicklink_conditional']) {
                \WP_CLI::warning("No valid page types provided but conditional mode is enabled. Disabling conditional mode.");
                $config['quicklink_conditional'] = false;
            } else {
                \WP_CLI::log("Updated page types to: " . implode(', ', $config['quicklink_page_types']));
            }
            
            $updated = true;
        }
        
        // Handle origins (allowed domains)
        if (isset($assoc_args['origins'])) {
            $origins = explode(',', $assoc_args['origins']);
            $valid_origins = [];
            
            foreach ($origins as $origin) {
                $origin = trim($origin);
                if (filter_var($origin, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME)) {
                    $valid_origins[] = $origin;
                } else {
                    \WP_CLI::warning("Invalid domain: {$origin}");
                }
            }
            
            $config['quicklink_origins'] = $valid_origins;
            \WP_CLI::log("Updated origins to: " . (empty($valid_origins) ? "same-origin only" : implode(', ', $valid_origins)));
            $updated = true;
        }
        
        // Handle ignores (patterns to exclude)
        if (isset($assoc_args['ignores'])) {
            $ignores = explode(',', $assoc_args['ignores']);
            $config['quicklink_ignores'] = array_map('trim', $ignores);
            \WP_CLI::log("Updated ignore patterns to: " . implode(', ', $config['quicklink_ignores']));
            $updated = true;
        }
        
        if ($updated) {
            // Update configuration
            SysConfig::update_config($config);
            \WP_CLI::success("Quicklink configuration has been updated successfully.");
        } else {
            \WP_CLI::log("No changes were made to the Quicklink configuration.");
            \WP_CLI::log("Available options:");
            \WP_CLI::log("  --mode=[prefetch|prerender|both]");
            \WP_CLI::log("  --throttle=<number>            (Concurrent requests limit)");
            \WP_CLI::log("  --limit=<number>               (Total requests limit)");
            \WP_CLI::log("  --delay=<milliseconds>         (Delay before prefetch)");
            \WP_CLI::log("  --timeout=<milliseconds>       (Idle timeout)");
            \WP_CLI::log("  --threshold=<0-100>            (Viewport threshold %)");
            \WP_CLI::log("  --priority=[true|false]        (Use high priority fetching)");
            \WP_CLI::log("  --respect-data-saver=[true|false]");
            \WP_CLI::log("  --min-connection=[5g|4g|3g|any]");
            \WP_CLI::log("  --block-ads=[true|false]       (Block ad/affiliate links)");
            \WP_CLI::log("  --conditional=[true|false]     (Enable conditional execution)");
            \WP_CLI::log("  --page-types=home,front_page,singular,archive,search");
            \WP_CLI::log("  --origins=example.com,example.org");
            \WP_CLI::log("  --ignores=/api/*,*.pdf,/logout");
        }
    }
}