<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * WooCommerce Thumbnail Manager - Specialized thumbnail management for WooCommerce
 * 
 * Provides advanced WooCommerce-specific thumbnail management including:
 * - WooCommerce thumbnail size detection and management
 * - Product image cleanup for deleted products
 * - Variation image management
 * - Category and tag image management
 * - Optimized processing for large product catalogs
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.5
 */
class WooCommerceThumbnailManager
{
    /**
     * WooCommerce default thumbnail sizes
     */
    private const WC_DEFAULT_SIZES = [
        'woocommerce_thumbnail' => 'WooCommerce Thumbnail',
        'woocommerce_single' => 'WooCommerce Single Product',
        'woocommerce_gallery_thumbnail' => 'WooCommerce Gallery Thumbnail'
    ];

    /**
     * Initialize WooCommerce Thumbnail Manager
     */
    public static function init(): void
    {
        if (!self::is_woocommerce_active()) {
            return;
        }

        // Hook into product deletion
        add_action('before_delete_post', [__CLASS__, 'cleanup_deleted_product_images'], 10, 1);
        add_action('wp_trash_post', [__CLASS__, 'cleanup_trashed_product_images'], 10, 1);
        
        // Hook into variation deletion
        add_action('woocommerce_delete_product_variation', [__CLASS__, 'cleanup_deleted_variation_images'], 10, 1);
        
        // Register WooCommerce specific settings
        add_action('init', [__CLASS__, 'register_wc_settings']);
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('WooCommerceThumbnailManager initialized', 'WCThumbnailManager');
        }
    }

    /**
     * Check if WooCommerce is active
     */
    public static function is_woocommerce_active(): bool
    {
        return class_exists('WooCommerce') && function_exists('wc_get_product');
    }

    /**
     * Get all WooCommerce thumbnail sizes
     */
    public static function get_woocommerce_thumbnail_sizes(): array
    {
        if (!self::is_woocommerce_active()) {
            return [];
        }

        $sizes = [];
        
        // Default WooCommerce sizes
        foreach (self::WC_DEFAULT_SIZES as $size => $label) {
            $size_data = wc_get_image_size($size);
            
            $sizes[$size] = [
                'label' => $label,
                'width' => (int)($size_data['width'] ?? 0),
                'height' => (int)($size_data['height'] ?? 0),
                'crop' => (bool)($size_data['crop'] ?? false),
                'source' => 'woocommerce',
                'active' => true,
                'usage_count' => self::count_size_usage($size)
            ];
        }

        // Theme-specific WooCommerce sizes
        $theme_sizes = self::detect_theme_wc_sizes();
        $sizes = array_merge($sizes, $theme_sizes);

        return $sizes;
    }

    /**
     * Detect theme-specific WooCommerce thumbnail sizes
     */
    private static function detect_theme_wc_sizes(): array
    {
        global $_wp_additional_image_sizes;
        $sizes = [];
        
        if (empty($_wp_additional_image_sizes)) {
            return $sizes;
        }

        $wc_patterns = [
            'woocommerce', 'wc_', 'shop_', 'product_', 'cart_', 'checkout_',
            'single_product', 'archive_product', 'category_', 'gallery_'
        ];

        foreach ($_wp_additional_image_sizes as $size => $data) {
            $is_wc_size = false;
            
            foreach ($wc_patterns as $pattern) {
                if (strpos(strtolower($size), $pattern) !== false) {
                    $is_wc_size = true;
                    break;
                }
            }

            if ($is_wc_size) {
                $sizes[$size] = [
                    'label' => self::generate_wc_size_label($size, $data),
                    'width' => (int)$data['width'],
                    'height' => (int)$data['height'],
                    'crop' => (bool)$data['crop'],
                    'source' => self::detect_wc_size_source($size),
                    'active' => true,
                    'usage_count' => self::count_size_usage($size)
                ];
            }
        }

        return $sizes;
    }

    /**
     * Generate label for WooCommerce size
     */
    private static function generate_wc_size_label(string $size_name, array $data): string
    {
        $width = $data['width'] ?? 0;
        $height = $data['height'] ?? 0;
        $crop = $data['crop'] ?? false;
        
        $clean_name = ucwords(str_replace(['_', '-'], ' ', $size_name));
        $dimensions = $crop ? "{$width}x{$height} (cropped)" : "{$width}x{$height}";
        
        return "{$clean_name} ({$dimensions})";
    }

    /**
     * Detect source of WooCommerce size
     */
    private static function detect_wc_size_source(string $size_name): string
    {
        $current_theme = wp_get_theme();
        $theme_name = strtolower($current_theme->get('Name'));
        
        if (strpos($size_name, $theme_name) !== false) {
            return $current_theme->get('Name') . ' Theme (WC)';
        }
        
        // Common WooCommerce theme patterns
        $wc_themes = [
            'storefront' => 'Storefront Theme',
            'astra' => 'Astra Theme (WC)',
            'oceanwp' => 'OceanWP Theme (WC)',
            'divi' => 'Divi Theme (WC)',
            'avada' => 'Avada Theme (WC)'
        ];
        
        foreach ($wc_themes as $pattern => $name) {
            if (strpos($size_name, $pattern) !== false) {
                return $name;
            }
        }
        
        return 'WooCommerce Plugin/Theme';
    }

    /**
     * Count usage of specific thumbnail size in WooCommerce context
     */
    private static function count_size_usage(string $size_name): int
    {
        if (!self::is_woocommerce_active()) {
            return 0;
        }

        global $wpdb;
        
        // Count in product attachments
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE p.post_type = 'attachment'
             AND p.post_mime_type LIKE 'image/%'
             AND pm.meta_key = '_wp_attachment_metadata'
             AND pm.meta_value LIKE %s",
            '%' . $size_name . '%'
        ));

        return (int)$count;
    }

    /**
     * Cleanup images when product is deleted
     */
    public static function cleanup_deleted_product_images(int $post_id): void
    {
        if (!self::is_woocommerce_active()) {
            return;
        }

        if (!current_user_can('delete_products')) {
            return;
        }

        $product = wc_get_product($post_id);
        if (!$product) {
            return;
        }

        // Get all product images
        $image_ids = [];
        
        // Featured image
        $featured_image_id = $product->get_image_id();
        if ($featured_image_id) {
            $image_ids[] = $featured_image_id;
        }
        
        // Gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        if (!empty($gallery_ids)) {
            $image_ids = array_merge($image_ids, $gallery_ids);
        }

        // Process each image
        foreach ($image_ids as $image_id) {
            self::process_orphaned_image($image_id, $post_id);
        }

        if (class_exists('\\ZiziCache\\CacheSys') && !empty($image_ids)) {
            \ZiziCache\CacheSys::writeLog(
                "Processed " . count($image_ids) . " images for deleted product ID: {$post_id}",
                'WCThumbnailManager'
            );
        }
    }

    /**
     * Cleanup images when product is trashed
     */
    public static function cleanup_trashed_product_images(int $post_id): void
    {
        $config = SysConfig::$config;
        $cleanup_trashed = $config['wc_cleanup_trashed_images'] ?? false;
        
        if ($cleanup_trashed) {
            self::cleanup_deleted_product_images($post_id);
        }
    }

    /**
     * Cleanup images when variation is deleted
     */
    public static function cleanup_deleted_variation_images(int $variation_id): void
    {
        if (!self::is_woocommerce_active()) {
            return;
        }

        $variation = wc_get_product($variation_id);
        if (!$variation) {
            return;
        }

        $image_id = $variation->get_image_id();
        if ($image_id) {
            self::process_orphaned_image($image_id, $variation_id);
        }
    }

    /**
     * Process potentially orphaned image
     */
    private static function process_orphaned_image(int $image_id, int $current_product_id): void
    {
        if (!self::is_image_used_elsewhere($image_id, $current_product_id)) {
            self::delete_image_with_thumbnails($image_id);
        }
    }

    /**
     * Check if image is used elsewhere
     */
    /**
     * Check if image is used elsewhere (public interface for other components)
     * 
     * @param int $image_id Image/attachment ID
     * @param int $current_product_id Current product ID to exclude from search
     * @return bool True if image is used elsewhere
     */
    public static function is_image_used_elsewhere(int $image_id, int $current_product_id): bool
    {
        global $wpdb;

        // Check if used as featured image in other products
        $featured_usage = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE p.post_type IN ('product', 'product_variation')
             AND p.post_status != 'trash'
             AND pm.meta_key = '_thumbnail_id'
             AND pm.meta_value = %d
             AND pm.post_id != %d",
            $image_id,
            $current_product_id
        ));

        if ($featured_usage > 0) {
            return true;
        }

        // Check if used in product galleries
        $gallery_usage = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE p.post_type = 'product'
             AND p.post_status != 'trash'
             AND pm.meta_key = '_product_image_gallery'
             AND pm.meta_value LIKE %s
             AND pm.post_id != %d",
            '%' . $image_id . '%',
            $current_product_id
        ));

        if ($gallery_usage > 0) {
            return true;
        }

        // Check if used in content
        $content_usage = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts}
             WHERE post_status != 'trash'
             AND post_content LIKE %s",
            '%wp-image-' . $image_id . '%'
        ));

        return $content_usage > 0;
    }

    /**
     * Delete image with all its thumbnails
     */
    private static function delete_image_with_thumbnails(int $image_id): void
    {
        $config = SysConfig::$config;
        $force_delete = $config['wc_force_delete_images'] ?? true;
        
        // Get image metadata for logging
        $image_path = get_attached_file($image_id);
        $metadata = wp_get_attachment_metadata($image_id);
        
        $deleted_files = 0;
        $total_size = 0;

        // Calculate total size before deletion
        if ($image_path && file_exists($image_path)) {
            $total_size += filesize($image_path);
        }

        if (!empty($metadata['sizes'])) {
            $upload_dir = wp_upload_dir();
            $base_dir = dirname($image_path);
            
            foreach ($metadata['sizes'] as $size_data) {
                $thumb_path = $base_dir . '/' . $size_data['file'];
                if (file_exists($thumb_path)) {
                    $total_size += filesize($thumb_path);
                }
            }
        }

        // Delete the attachment
        $deleted = wp_delete_attachment($image_id, $force_delete);
        
        if ($deleted && class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Deleted orphaned image ID: {$image_id}, saved: " . size_format($total_size),
                'WCThumbnailManager'
            );
        }
    }

    /**
     * Analyze WooCommerce thumbnail usage
     */
    public static function analyze_woocommerce_usage(int $batch_size = 100): array
    {
        if (!self::is_woocommerce_active()) {
            return [
                'error' => 'WooCommerce not active'
            ];
        }

        $results = [
            'total_products' => 0,
            'total_variations' => 0,
            'total_images' => 0,
            'orphaned_images' => 0,
            'wc_size_usage' => [],
            'potential_savings' => 0,
            'recommended_actions' => []
        ];

        // Get product count
        $results['total_products'] = wp_count_posts('product')->publish ?? 0;
        $results['total_variations'] = wp_count_posts('product_variation')->publish ?? 0;

        // Analyze in batches
        $offset = 0;
        $wc_sizes = self::get_woocommerce_thumbnail_sizes();
        
        do {
            $products = wc_get_products([
                'limit' => $batch_size,
                'offset' => $offset,
                'status' => 'publish'
            ]);

            foreach ($products as $product) {
                $results = self::analyze_product_images($product, $results, $wc_sizes);
            }

            $offset += $batch_size;
            
        } while (count($products) === $batch_size);

        // Generate recommendations
        $results['recommended_actions'] = self::generate_wc_recommendations($results);

        return $results;
    }

    /**
     * Analyze individual product images
     */
    private static function analyze_product_images($product, array $results, array $wc_sizes): array
    {
        $image_ids = [];
        
        // Featured image
        $featured_id = $product->get_image_id();
        if ($featured_id) {
            $image_ids[] = $featured_id;
        }
        
        // Gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        $image_ids = array_merge($image_ids, $gallery_ids);

        foreach ($image_ids as $image_id) {
            $results['total_images']++;
            
            // Check if orphaned
            if (!self::is_image_used_elsewhere($image_id, $product->get_id())) {
                $results['orphaned_images']++;
                
                // Calculate potential savings
                $metadata = wp_get_attachment_metadata($image_id);
                if ($metadata && !empty($metadata['sizes'])) {
                    foreach ($metadata['sizes'] as $size_name => $size_data) {
                        if (isset($wc_sizes[$size_name])) {
                            $file_path = self::get_thumbnail_file_path($image_id, $size_name);
                            if ($file_path && file_exists($file_path)) {
                                $results['potential_savings'] += filesize($file_path);
                            }
                        }
                    }
                }
            }

            // Analyze thumbnail usage
            $metadata = wp_get_attachment_metadata($image_id);
            if ($metadata && !empty($metadata['sizes'])) {
                foreach ($metadata['sizes'] as $size_name => $size_data) {
                    if (isset($wc_sizes[$size_name])) {
                        if (!isset($results['wc_size_usage'][$size_name])) {
                            $results['wc_size_usage'][$size_name] = 0;
                        }
                        $results['wc_size_usage'][$size_name]++;
                    }
                }
            }
        }

        return $results;
    }

    /**
     * Generate WooCommerce-specific recommendations
     */
    private static function generate_wc_recommendations(array $results): array
    {
        $recommendations = [];

        if ($results['orphaned_images'] > 0) {
            $recommendations[] = [
                'type' => 'cleanup',
                'priority' => 'high',
                'title' => 'Orphaned Product Images',
                'description' => "Found {$results['orphaned_images']} orphaned product images",
                'action' => 'cleanup_orphaned_images',
                'potential_savings' => $results['potential_savings']
            ];
        }

        // Analyze size usage efficiency
        $total_thumbnails = array_sum($results['wc_size_usage']);
        foreach ($results['wc_size_usage'] as $size_name => $count) {
            $usage_rate = $total_thumbnails > 0 ? ($count / $total_thumbnails) * 100 : 0;
            
            if ($usage_rate < 5) {
                $recommendations[] = [
                    'type' => 'optimization',
                    'priority' => 'medium',
                    'title' => "Low Usage Thumbnail Size: {$size_name}",
                    'description' => "Size '{$size_name}' is used in only {$usage_rate}% of images",
                    'action' => 'consider_disabling',
                    'size_name' => $size_name
                ];
            }
        }

        return $recommendations;
    }

    /**
     * Get thumbnail file path
     */
    private static function get_thumbnail_file_path(int $attachment_id, string $size_name): ?string
    {
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        if (!$metadata || empty($metadata['sizes'][$size_name])) {
            return null;
        }
        
        $upload_dir = wp_upload_dir();
        $file_dir = dirname($metadata['file']);
        $filename = $metadata['sizes'][$size_name]['file'];
        
        return $upload_dir['basedir'] . '/' . $file_dir . '/' . $filename;
    }

    /**
     * Register WooCommerce-specific settings
     */
    public static function register_wc_settings(): void
    {
        $config = SysConfig::$config;
        
        // Set default WooCommerce settings if not exist
        if (!isset($config['wc_cleanup_deleted_images'])) {
            $config['wc_cleanup_deleted_images'] = true;
            SysConfig::updateConfig($config);
        }
        
        if (!isset($config['wc_cleanup_trashed_images'])) {
            $config['wc_cleanup_trashed_images'] = false;
            SysConfig::updateConfig($config);
        }
        
        if (!isset($config['wc_force_delete_images'])) {
            $config['wc_force_delete_images'] = true;
            SysConfig::updateConfig($config);
        }
    }

    /**
     * Get WooCommerce thumbnail statistics
     */
    public static function get_wc_statistics(): array
    {
        if (!self::is_woocommerce_active()) {
            return [
                'woocommerce_active' => false
            ];
        }

        $wc_sizes = self::get_woocommerce_thumbnail_sizes();
        
        return [
            'woocommerce_active' => true,
            'wc_thumbnail_sizes' => count($wc_sizes),
            'total_products' => wp_count_posts('product')->publish ?? 0,
            'total_variations' => wp_count_posts('product_variation')->publish ?? 0,
            'wc_sizes' => $wc_sizes
        ];
    }
}
