<?php

namespace ZiziCache;

/**
 * WooCommerce Compatibility Layer
 * 
 * Handles compatibility issues and conflicts between ZiziCache and WooCommerce
 * 
 * @package ZiziCache
 * @since 0.5.3-beta.5
 */
class WooCommerceCompatibility
{
    /**
     * Initialize WooCommerce compatibility layer
     */
    public static function init()
    {
        // Add early detection for WooCommerce admin pages
        add_action('admin_init', [__CLASS__, 'detect_woocommerce_admin'], 1);
        
        // Add filter to prevent ZiziCache admin scripts on WooCommerce pages
        add_filter('zizi_cache_should_load_admin_scripts', [__CLASS__, 'should_load_admin_scripts'], 10, 1);
        
        // Log initialization
        if (defined('ZIZI_CACHE_DEBUG') && ZIZI_CACHE_DEBUG) {
            CacheSys::writeLog('WooCommerce compatibility layer initialized', 'INFO', 'WooCommerceCompat');
        }
    }
    
    /**
     * Detect if we're on WooCommerce admin pages
     * Sets constants and flags for other components to use
     */
    public static function detect_woocommerce_admin()
    {
        if (!is_admin()) {
            return;
        }
        
        $is_wc_admin = false;
        
        // Check for WooCommerce admin pages
        if (isset($_GET['page']) && $_GET['page'] === 'wc-admin') {
            $is_wc_admin = true;
        }
        
        // Check for specific WooCommerce setup wizard
        if (isset($_GET['page']) && $_GET['page'] === 'wc-admin' && isset($_GET['path'])) {
            $path = urldecode($_GET['path']);
            if (strpos($path, '/setup-wizard') === 0 || strpos($path, 'setup-wizard') !== false) {
                $is_wc_admin = true;
            }
        }
        
        // Set global flag
        if ($is_wc_admin && !defined('ZIZI_CACHE_WC_ADMIN_PAGE')) {
            define('ZIZI_CACHE_WC_ADMIN_PAGE', true);
            
            if (defined('ZIZI_CACHE_DEBUG') && ZIZI_CACHE_DEBUG) {
                CacheSys::writeLog('WooCommerce admin page detected: ' . $_SERVER['REQUEST_URI'], 'INFO', 'WooCommerceCompat');
            }
        }
    }
    
    /**
     * Determine if admin scripts should be loaded
     * 
     * @param bool $should_load Current decision
     * @return bool Modified decision
     */
    public static function should_load_admin_scripts($should_load)
    {
        // Don't load admin scripts on WooCommerce pages
        if (defined('ZIZI_CACHE_WC_ADMIN_PAGE') && ZIZI_CACHE_WC_ADMIN_PAGE) {
            return false;
        }
        
        // Respect admin bypass setting
        if (defined('ZIZI_CACHE_ADMIN_BYPASS') && ZIZI_CACHE_ADMIN_BYPASS) {
            return false;
        }
        
        return $should_load;
    }
    
    /**
     * Check if current page is a WooCommerce admin page
     * 
     * @return bool
     */
    public static function is_woocommerce_admin_page()
    {
        if (!is_admin()) {
            return false;
        }
        
        // Check constant set by early detection
        if (defined('ZIZI_CACHE_WC_ADMIN_PAGE') && ZIZI_CACHE_WC_ADMIN_PAGE) {
            return true;
        }
        
        // Runtime check as fallback
        if (isset($_GET['page']) && $_GET['page'] === 'wc-admin') {
            return true;
        }
        
        return false;
    }
    
    /**
     * Get WooCommerce admin page context
     * 
     * @return array Context information
     */
    public static function get_woocommerce_context()
    {
        if (!self::is_woocommerce_admin_page()) {
            return ['is_wc_admin' => false];
        }
        
        $context = [
            'is_wc_admin' => true,
            'page' => $_GET['page'] ?? '',
            'path' => isset($_GET['path']) ? urldecode($_GET['path']) : '',
            'is_setup_wizard' => false,
            'is_analytics' => false,
            'is_marketing' => false,
        ];
        
        // Analyze path for specific contexts
        if (!empty($context['path'])) {
            $context['is_setup_wizard'] = strpos($context['path'], 'setup-wizard') !== false;
            $context['is_analytics'] = strpos($context['path'], 'analytics') !== false;
            $context['is_marketing'] = strpos($context['path'], 'marketing') !== false;
        }
        
        return $context;
    }
}
