<?php

namespace ZiziCache;

/**
 * Class WPCache
 *
 * Manages the WP_CACHE constant in wp-config.php:
 * - Ensures define('WP_CACHE', true) is present on init.
 * - Removes the constant on plugin deactivation.
 */
class WPCache
{
    /**
     * Hooks into plugin lifecycle to add or remove WP_CACHE constant.
     */
    public static function init(): void
    {
        // Ensure WP_CACHE constant is defined
        self::add_constant();
        // Remove WP_CACHE on plugin deactivation
        register_deactivation_hook(ZIZI_CACHE_FILE_NAME, [__CLASS__, 'remove_constant']);
    }

    /**
     * Locate wp-config.php file (root or parent directory).
     *
     * @return string|null Path or null if not found.
     */
    private static function get_wp_config_path(): ?string
    {
        $paths = [ABSPATH . 'wp-config.php', dirname(ABSPATH) . '/wp-config.php'];
        foreach ($paths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }
        return null;
    }

    /**
     * Modify wp-config.php content via callback, write only if changed.
     *
     * @param callable(string): string $modifier
     */
    private static function update_wp_config(callable $modifier): void
    {
        $path = self::get_wp_config_path();
        if (!$path || !is_writable($path)) {
            return;
        }
        $content = file_get_contents($path);
        $new_content = $modifier($content);
        // Write only if content differs
        if ($new_content !== $content) {
            file_put_contents($path, $new_content);
        }
    }

    /**
     * Add or ensure WP_CACHE constant in wp-config.php.
     */
    public static function add_constant(): void
    {
        if (defined('WP_CACHE') && WP_CACHE) {
            return;
        }
        self::update_wp_config(function ($content) {
            // Remove old WP_CACHE definitions and plugin comments
            $content = preg_replace('/^[ \t]*(?:define\s*\(\s*[\'\"]WP_CACHE[\'\"].*?;|\/\/\s*Added by ZiziCache).*?[\r\n]+/mi', '', $content);
            // Insert constant after <?php if not present
            if (!preg_match("/define\(\s*'WP_CACHE'/i", $content) && preg_match('/(<\?php\s*\r?\n)/', $content, $matches)) {
                $content = preg_replace('/(<\?php\s*\r?\n)/', "$1define('WP_CACHE', true); // Added by ZiziCache\r\n", $content, 1);
            }
            return $content;
        });
    }

    /**
     * Remove WP_CACHE constant from wp-config.php.
     */
    public static function remove_constant(): void
    {
        self::update_wp_config(function ($content) {
            // Remove WP_CACHE definitions and plugin comments
            return preg_replace('/^[ \t]*(?:define\s*\(\s*[\'\"]WP_CACHE[\'\"].*?;|\/\/\s*Added by ZiziCache).*?[\r\n]+/mi', '', $content);
        });
    }
}
