<?php

namespace ZiziCache;

/**
 * Class Utils
 * Performance and security optimized utility functions for ZiziCache plugin
 */
class SysTool
{
  /**
   * List of external domains for asset downloads.
   * @var string[]
   */
  public const EXTERNAL_DOMAINS = [
    'cdn.jsdelivr.net',
    'cdnjs.cloudflare.com',
    'unpkg.com',
    'code.jquery.com',
    'ajax.googleapis.com',
    'use.fontawesome.com',
    'ajax.aspnetcdn.com',
    'cdn.polyfill.io',
    'stackpath.bootstrapcdn.com',
    'bootstrapcdn.com',
    'cdn.tailwindcss.com',
    'typekit.net',
    'use.typekit.net',
    'fonts.googleapis.com',
    'fonts.gstatic.com',
  ];

  /**
   * Static variable for external file downloads (browser-like user agent)
   * @var string
   */
  public static string $user_agent = 'Mozilla/5.0 (Macintosh; Intel Mac OS X 14_7_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/134.0.0.0 Safari/537.36';

  /**
   * Cache for keyword matching to avoid redundant processing within request
   * @var array<string,bool>
   */
  private static array $keyword_match_cache = [];

  /**
   * Download cache to avoid duplicate downloads within same request
   * @var array<string,string|null>
   */
  private static array $download_cache = [];

  /**
   * Precompiled regex patterns for better performance
   * @var array<string,string>
   */
  private static array $regex_patterns = [
    'protocol' => '/^https?:\/\//',
    'css_js_extension' => '/\.(css|js)$/',
    'resource_hints' => '/<link[^>]*(?:prefetch|preconnect|preload)[^>]*%s[^>]*>/i',
  ];

  /**
   * Returns a unique User-Agent for preloading, based on the current domain.
   * @return string
   */
  public static function get_user_agent(): string
  {
    static $cached_user_agent = null;
    
    if ($cached_user_agent !== null) {
      return $cached_user_agent;
    }
    
    $host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'unknown-host';
    $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https://' : 'http://') . $host . '/';
    $cached_user_agent = 'ZiziCachePreloadBot/1.0 (+'.$url.')';
    
    return $cached_user_agent;
  }

  /**
   * Checks if any of the keywords are present in the given string (case-insensitive).
   * Uses in-memory caching for performance.
   * 
   * @param string[] $keywords
   * @param string $string
   * @return bool
   */
  public static function any_keywords_match_string(array $keywords, string $string): bool
  {
    // Generate cache key based on inputs
    $cache_key = md5(json_encode($keywords) . '|' . $string);
    
    // Return cached result if available
    if (isset(self::$keyword_match_cache[$cache_key])) {
      return self::$keyword_match_cache[$cache_key];
    }
    
    // Filter out empty elements
    $keywords = array_filter($keywords);

    foreach ($keywords as $keyword) {
      if (stripos($string, $keyword) !== false) {
        // Cache result before returning
        self::$keyword_match_cache[$cache_key] = true;
        return true;
      }
    }

    // Cache result before returning
    self::$keyword_match_cache[$cache_key] = false;
    return false;
  }

  /**
   * Replace the first occurrence of a string with another string.
   * Optimized for performance with empty check.
   * 
   * @param string $search
   * @param string $replace
   * @param string $subject
   * @return string
   */
  public static function str_replace_first(string $search, string $replace, string $subject): string
  {
    // Quick path for empty inputs
    if ($search === '' || $subject === '') {
      return $subject;
    }
    
    $pos = strpos($subject, $search);
    if ($pos !== false) {
      return substr_replace($subject, $replace, $pos, strlen($search));
    }
    return $subject;
  }

  /**
   * Checks if a URL is external (from a different domain than the site).
   * 
   * @param string $url URL to check
   * @param string $site_url Base site URL to compare against
   * @return bool True if URL is external, false otherwise
   */
  public static function is_external_url(string $url, string $site_url): bool
  {
    // If no protocol is specified, assume the same protocol as the site
    if (strpos($url, '//') === 0) {
      $url = 'https:' . $url;
    }

    // Parse URLs
    $url_parts = parse_url($url);
    $site_parts = parse_url($site_url);

    // If we couldn't parse either URL, consider it not external
    if (!$url_parts || !isset($url_parts['host']) || !$site_parts || !isset($site_parts['host'])) {
      return false;
    }

    // Compare hosts
    return strtolower($url_parts['host']) !== strtolower($site_parts['host']);
  }

  /**
   * Downloads an external file if it matches allowed domains and saves it locally.
   * Includes caching, security validations, and performance optimizations.
   * 
   * @param string $url URL to download
   * @param string $type Optional file type ('css' or 'js')
   * @return string|null URL to the cached file or null on failure
   */
  public static function download_external_file(string $url, string $type = ''): ?string
  {
    // Security check - validate URL format
    if (!filter_var($url, FILTER_VALIDATE_URL) && strpos($url, '//') !== 0) {
      \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache: Invalid URL format: ' . $url);
      return null;
    }
    
    // Use cached result if available
    if (isset(self::$download_cache[$url])) {
      return self::$download_cache[$url];
    }
    
    $external_domains = apply_filters(
      'zizi_cache_selfhost_external_domains',
      self::EXTERNAL_DOMAINS
    );

    // Check if the base domain is present in the external domains array
    if (!self::any_keywords_match_string($external_domains, $url)) {
      \ZiziCache\CacheSys::writeWarning('URL domain not in allowed domains list: ' . $url, 'Security');
      self::$download_cache[$url] = null;
      return null;
    }

    // Check if src has protocol if not prepend https
    $url_new = preg_match(self::$regex_patterns['protocol'], $url) ? $url : 'https:' . $url;

    // Security: SSRF protection - validate URL doesn't point to private IP ranges
    $parsed_url = parse_url($url_new);
    if (isset($parsed_url['host'])) {
      $ip = gethostbyname($parsed_url['host']);
      if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        \ZiziCache\CacheSys::writeWarning('SSRF protection - blocked private/reserved IP: ' . $ip . ' for URL: ' . $url_new, 'Security');
        self::$download_cache[$url] = null;
        return null;
      }
    }

    // Security: Sanitize file name to prevent directory traversal
    $raw_filename = basename(parse_url($url_new, PHP_URL_PATH));
    $file_name = self::sanitize_external_filename($raw_filename, $url_new);
    $file_name = strtok($file_name, '?');
    
    // Ensure filename isn't empty after sanitization (handled in sanitize_external_filename)
    // Note: sanitize_external_filename now provides secure fallback naming
    
    // Create directory path for third party files
    $third_party_dir = ZIZI_CACHE_CACHE_DIR . '3rd-css-js/';
    $cache_file_path = $third_party_dir . $file_name;
    $cache_file_url = ZIZI_CACHE_CACHE_URL . '3rd-css-js/' . $file_name;
    
    // Check if file already exists and is valid (not empty)
    if (is_file($cache_file_path) && filesize($cache_file_path) > 0) {
      \ZiziCache\CacheSys::writeLog('Using cached external file for: ' . $url, 'INFO');
      self::$download_cache[$url] = $cache_file_url;
      return self::$download_cache[$url];
    }

    // Ensure third-party cache directory exists with proper permissions
    if (!is_dir($third_party_dir)) {
      if (!wp_mkdir_p($third_party_dir)) {
        \ZiziCache\CacheSys::writeError('Failed to create third-party cache directory: ' . $third_party_dir);
        self::$download_cache[$url] = null;
        return null;
      }
      // Ensure proper permissions - same as page cache
      @chmod($third_party_dir, 0755);
      \ZiziCache\CacheSys::writeLog('Created third-party cache directory: ' . $third_party_dir, 'INFO');
      
      // SECURITY: Create .htaccess protection for cache directories
      self::ensure_cache_security();
    }

    // Get the cache lifespan from htaccess settings (or use default)
    $cache_lifespan = self::get_cache_lifespan_days();
    
    // Get the content from the file with optimized settings
    $response = wp_remote_get($url_new, [
      'user-agent' => self::$user_agent,
      'httpversion' => '2.0',
      'timeout' => 15,
      'sslverify' => true,
      'redirection' => 5,
      'headers' => [
        'Accept-Encoding' => 'gzip, deflate'
      ]
    ]);

    // Return if the file response is invalid
    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
      $error = is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response);
      \ZiziCache\CacheSys::writeError('Failed to download file: ' . $url_new . ' - ' . $error);
      self::$download_cache[$url] = null;
      return null;
    }

    // Content type of the response
    $content_type = wp_remote_retrieve_header($response, 'content-type');

    // Determine the extension based on the content type
    $extension = self::get_extension_from_content_type($content_type, $file_name);
    
    // If the specific type is provided and the extension doesn't match, use the provided type
    if (!empty($type) && !preg_match('/\.' . $type . '$/i', $file_name)) {
      $extension = $type;
    }

    // If the file name does not have an extension then append the type as extension
    if (!preg_match(self::$regex_patterns['css_js_extension'], $file_name)) {
      $file_name = md5($url_new) . '.' . $extension;
      $cache_file_path = $third_party_dir . $file_name;
      $cache_file_url = ZIZI_CACHE_CACHE_URL . '3rd-css-js/' . $file_name;
    }

    $content = wp_remote_retrieve_body($response);

    // Security: Validate content length
    if (strlen($content) <= 0 || strlen($content) > 10 * 1024 * 1024) { // Max 10MB
      \ZiziCache\CacheSys::writeWarning('Invalid content length: ' . strlen($content) . ' bytes for ' . $url_new);
      self::$download_cache[$url] = null;
      return null;
    }

    // Security: Validate file content for malicious code before saving
    if (!self::validate_external_file_content($content, $extension, $url_new)) {
      \ZiziCache\CacheSys::writeWarning('Blocked malicious content in external file: ' . $url_new, 'Security');
      self::$download_cache[$url] = null;
      return null;
    }

    // Filter hook to modify file contents before saving
    $content = apply_filters(
      'zizi_cache_download_external_file:before',
      $content,
      $url_new,
      $extension
    );

    // Add metadata comment for tracking
    $metadata = "/* ZiziCache hosted file: {$url}\n * Downloaded: " . date('Y-m-d H:i:s') . "\n * Cache expires: " . date('Y-m-d H:i:s', strtotime("+{$cache_lifespan} days")) . "\n */\n";
    if ($extension === 'js' || $extension === 'css') {
      $content = $metadata . $content;
    }

    // Write file with exclusive lock for thread safety
    $result = file_put_contents($cache_file_path, $content, LOCK_EX);
    
    if ($result === false) {
      \ZiziCache\CacheSys::writeError('Failed to write file: ' . $cache_file_path . ' - Error: ' . error_get_last()['message']);
      self::$download_cache[$url] = null;
      return null;
    }

    // Security: Set proper file permissions
    @chmod($cache_file_path, 0644);
    
    // Log detailed success info
    \ZiziCache\CacheSys::writeLog(sprintf(
      'Successfully downloaded and cached external file: %s -> %s (Size: %s bytes, Type: %s, Lifespan: %d days)',
      $url_new,
      $cache_file_path,
      strlen($content),
      $extension,
      $cache_lifespan
    ), 'INFO');
    
    // Cache and return result
    self::$download_cache[$url] = $cache_file_url;
    return self::$download_cache[$url];
  }

  /**
   * Determine the file extension based on content-type header
   * 
   * @param string $content_type The content-type header
   * @param string $file_name Original file name
   * @return string The appropriate file extension
   */
  private static function get_extension_from_content_type(string $content_type, string $file_name): string
  {
    $content_type = strtolower($content_type);
    
    // Map of common content types to extensions
    $content_type_map = [
      'text/css' => 'css',
      'text/javascript' => 'js',
      'application/javascript' => 'js',
      'application/x-javascript' => 'js',
      'font/woff' => 'woff',
      'font/woff2' => 'woff2',
      'font/ttf' => 'ttf',
      'font/otf' => 'otf',
      'image/svg+xml' => 'svg',
    ];
    
    // First try exact match
    foreach ($content_type_map as $type => $ext) {
      if (strpos($content_type, $type) !== false) {
        return $ext;
      }
    }
    
    // If we have a valid extension in the filename, use that
    if (preg_match('/\.(js|css|woff|woff2|ttf|eot|svg)$/i', $file_name, $matches)) {
      return strtolower($matches[1]);
    }
    
    // Default fallbacks
    if (strpos($content_type, 'text/') === 0) {
      return 'txt';
    }
    
    if (strpos($content_type, 'image/') === 0) {
      return 'img';
    }
    
    // Last resort
    return 'js';
  }

  /**
   * Retrieves the cache lifespan in days for external files.
   * Based on htaccess cache settings or SysConfig.
   *
   * @return int Cache lifespan in days (default: 30)
   */
  public static function get_cache_lifespan_days(): int
  {
    // Default cache lifespan is 30 days
    $default_lifespan = 30;
    
    // Get cache_lifespan from SysConfig if available
    if (class_exists('\\ZiziCache\\SysConfig') && isset(SysConfig::$config['cache_lifespan'])) {
      $config_lifespan = SysConfig::$config['cache_lifespan'];
      
      // Convert config lifespan to days
      switch ($config_lifespan) {
        case '1hour':
          return 1; // Use minimum 1 day for hourly settings
        case '1day':
          return 1;
        case '3days':
          return 3;
        case '1week':
          return 7;
        case '2weeks':
          return 14;
        case '1month':
          return 30;
        case '6months':
          return 180;
        case '1year':
          return 365;
        case 'never':
          return 365; // For 'never' expire, use 1 year for external files
        default:
          // Try to parse numeric value (days)
          if (is_numeric($config_lifespan)) {
            return (int) $config_lifespan;
          }
          break;
      }
    }
    
    // Check htaccess settings for CSS/JS file expiration
    $htaccess_file = ABSPATH . '.htaccess';
    if (file_exists($htaccess_file) && is_readable($htaccess_file)) {
      $htaccess_content = file_get_contents($htaccess_file);
      
      // Look for CSS/JS expiration settings in htaccess
      if (preg_match('/ExpiresByType\s+(?:text\/css|application\/javascript)\s+"access\s+plus\s+(\d+)\s+(second|minute|hour|day|month|year)s?"/', 
        $htaccess_content, $matches)) {
        
        $duration = (int) $matches[1];
        $unit = strtolower($matches[2]);
        
        // Convert to days
        switch ($unit) {
          case 'second':
            return max(1, ceil($duration / 86400)); // At least 1 day
          case 'minute':
            return max(1, ceil($duration / 1440)); // At least 1 day
          case 'hour':
            return max(1, ceil($duration / 24)); // At least 1 day
          case 'day':
            return $duration;
          case 'month':
            return $duration * 30; // Approximate
          case 'year':
            return $duration * 365; // Approximate
          default:
            break;
        }
      }
    }
    
    return $default_lifespan;
  }

  /**
   * Clean up cached third-party CSS/JS files.
   * Called when the feature is disabled or a cleanup is requested.
   *
   * @param bool $force Whether to force cleanup regardless of feature status
   * @return int Number of files removed
   */
  public static function cleanup_third_party_files(bool $force = false): int
  {
    // Only clean up if self-hosting is disabled or forced
    if (!$force && !empty(SysConfig::$config['self_host_third_party_css_js'])) {
      return 0;
    }
    
    $third_party_dir = ZIZI_CACHE_CACHE_DIR . '3rd-css-js/';
    
    // If directory doesn't exist, nothing to clean
    if (!is_dir($third_party_dir)) {
      return 0;
    }
    
    $files_removed = 0;
    $files = glob($third_party_dir . '*');
    
    foreach ($files as $file) {
      if (is_file($file) && unlink($file)) {
        $files_removed++;
      }
    }
    
    // Log the cleanup
    \ZiziCache\CacheSys::writeLog("Cleaned up {$files_removed} third-party hosted files from {$third_party_dir}", 'INFO');
    
    return $files_removed;
  }
  
  /**
   * Removes resource hints from HTML for a given URL.
   * Uses precompiled regex patterns for better performance.
   * 
   * @param string $url
   * @param string $html
   * @return string
   */
  public static function remove_resource_hints(string $url, string $html): string
  {
    $url_host = parse_url($url, PHP_URL_HOST);

    if (!$url_host) {
      return $html;
    }

    // Create regex with proper escaping for the hostname
    $pattern = sprintf(self::$regex_patterns['resource_hints'], preg_quote($url_host, '/'));
    
    // Use a more efficient preg_replace with a compiled pattern
    $html = preg_replace($pattern, '', $html);
    
    return $html;
  }
  
  /**
   * Safely creates a directory with proper error handling.
   * 
   * @param string $path The directory path to create
   * @param int $permissions The directory permissions (octal)
   * @return bool True on success, false on failure
   */
  public static function ensure_directory(string $path, int $permissions = 0755): bool
  {
    if (is_dir($path)) {
      return true;
    }
    
    $result = wp_mkdir_p($path);
    
    if ($result) {
      @chmod($path, $permissions);
    }
    
    return $result;
  }
  
  /**
   * Sanitizes a filename to be safe for filesystem operations.
   * 
   * @param string $filename The filename to sanitize
   * @return string Sanitized filename
   */
  public static function sanitize_filename(string $filename): string
  {
    // Remove any path components
    $filename = basename($filename);
    
    // Remove any potentially dangerous characters
    $filename = preg_replace('/[^\w\s\d\.\-_]/u', '', $filename);
    
    // Ensure filename isn't empty
    if (empty($filename)) {
      return 'file_' . substr(md5(random_bytes(16)), 0, 8);
    }
    
    return $filename;
  }

  /**
   * Verify that the "Hostovat externí CSS a JS lokálně" feature is working correctly.
   * This is a test function that can be run via WP-CLI or debug endpoints.
   *
   * @return array Test results
   */
  public static function test_third_party_hosting(): array
  {
    $results = [
      'feature_enabled' => false,
      'cache_dir_exists' => false,
      'cache_dir_writable' => false,
      'test_files' => [],
      'cleanup_result' => null
    ];
    
    // Check if feature is enabled
    if (class_exists('\\ZiziCache\\SysConfig')) {
      $results['feature_enabled'] = !empty(SysConfig::$config['self_host_third_party_css_js']);
    }
    
    // Check if cache directory exists and is writable
    $third_party_dir = ZIZI_CACHE_CACHE_DIR . '3rd-css-js/';
    $results['cache_dir_exists'] = is_dir($third_party_dir);
    $results['cache_dir_writable'] = is_dir($third_party_dir) && is_writable($third_party_dir);
    
    // Test downloading some common external files
    $test_urls = [
      'css' => 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css',
      'js' => 'https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js'
    ];
    
    foreach ($test_urls as $type => $url) {
      $start_time = microtime(true);
      $downloaded_url = self::download_external_file($url, $type);
      $time_taken = microtime(true) - $start_time;
      
      $results['test_files'][$url] = [
        'success' => !empty($downloaded_url),
        'local_url' => $downloaded_url,
        'time_taken' => round($time_taken, 4) . ' seconds',
        'file_exists' => !empty($downloaded_url) && file_exists(str_replace(ZIZI_CACHE_CACHE_URL, ZIZI_CACHE_CACHE_DIR, $downloaded_url))
      ];
    }
    
    // Test cleanup function
    if ($results['feature_enabled']) {
      // Don't actually clean up if feature is enabled
      $results['cleanup_result'] = 'Skipped (feature is enabled)';
    } else {
      // Count files before
      $files_before = is_dir($third_party_dir) ? count(glob($third_party_dir . '*')) : 0;
      
      // Run cleanup
      $files_cleaned = self::cleanup_third_party_files(true);
      
      // Count files after
      $files_after = is_dir($third_party_dir) ? count(glob($third_party_dir . '*')) : 0;
      
      $results['cleanup_result'] = [
        'files_before' => $files_before,
        'files_cleaned' => $files_cleaned,
        'files_after' => $files_after,
      ];
    }
    
    return $results;
  }
  
  /**
   * Enhanced cache security with server type detection
   * Creates appropriate security files based on web server type:
   * - Apache: .htaccess files with mod_rewrite rules
   * - Nginx: Creates warning files and suggests server configuration
   * - IIS: Creates web.config files
   * 
   * @return bool Success status
   */
  public static function ensure_cache_security(): bool
  {
    $server_type = self::detect_server_type();
    
    $security_configs = [
      // Main cache directory - selective protection
      ZIZI_CACHE_CACHE_DIR => [
        'type' => 'main_cache',
        'protection_level' => 'selective'
      ],
      
      // External files directory - CRITICAL protection (only if enabled)
      ZIZI_CACHE_CACHE_DIR . '3rd-css-js/' => [
        'type' => 'external_files', 
        'protection_level' => 'critical',
        'conditional' => true // Only create if directory exists
      ],
      
      // Plugin root directory - LOG FILE protection
      dirname(ZIZI_CACHE_LOG_FILE) . '/' => [
        'type' => 'plugin_root',
        'protection_level' => 'log_files'
      ]
    ];
    
    $success = true;
    
    foreach ($security_configs as $dir => $config) {
      // Skip conditional directories that don't exist
      if (isset($config['conditional']) && $config['conditional'] && !is_dir($dir)) {
        continue;
      }
      
      // Ensure directory exists
      if (!is_dir($dir)) {
        if (!wp_mkdir_p($dir)) {
          \ZiziCache\CacheSys::writeError("Failed to create secure directory: $dir", 'Security');
          $success = false;
          continue;
        }
      }
      
      // Apply security based on server type
      switch ($server_type) {
        case 'apache':
          $result = self::create_apache_security($dir, $config);
          break;
          
        case 'nginx':
          $result = self::create_nginx_security_notice($dir, $config);
          break;
          
        case 'iis':
          $result = self::create_iis_security($dir, $config);
          break;
          
        default:
          $result = self::create_generic_security($dir, $config);
      }
      
      if (!$result) {
        $success = false;
      }
    }
    
    return $success;
  }

  /**
   * Detect web server type from environment
   * 
   * @return string Server type: 'apache', 'nginx', 'iis', or 'unknown'
   */
  private static function detect_server_type(): string
  {
    $server_software = $_SERVER['SERVER_SOFTWARE'] ?? '';
    
    if (stripos($server_software, 'nginx') !== false) {
      return 'nginx';
    }
    
    if (stripos($server_software, 'apache') !== false) {
      return 'apache';
    }
    
    if (stripos($server_software, 'microsoft-iis') !== false || 
        stripos($server_software, 'iis') !== false) {
      return 'iis';
    }
    
    // Additional detection methods
    global $is_nginx, $is_apache, $is_iis;
    
    if (isset($is_nginx) && $is_nginx) {
      return 'nginx';
    }
    
    if (isset($is_apache) && $is_apache) {
      return 'apache';
    }
    
    if (isset($is_iis) && $is_iis) {
      return 'iis';
    }
    
    // Check for server-specific environment variables
    if (isset($_SERVER['SERVER_ADMIN']) || function_exists('apache_get_version')) {
      return 'apache';
    }
    
    return 'unknown';
  }

  /**
   * Create Apache .htaccess security
   * 
   * @param string $dir Directory path
   * @param array $config Security configuration
   * @return bool Success status
   */
  private static function create_apache_security(string $dir, array $config): bool
  {
    $htaccess_content = '';
    
    if ($config['type'] === 'external_files') {
      $htaccess_content = SecurityConfig::get_external_files_htaccess();
    } elseif ($config['type'] === 'plugin_root') {
      $htaccess_content = SecurityConfig::get_plugin_root_htaccess();
    } else {
      $htaccess_content = SecurityConfig::get_main_cache_htaccess();
    }
    
    $htaccess_path = $dir . '.htaccess';
    
    // Write .htaccess if doesn't exist or content differs
    if (!file_exists($htaccess_path) || file_get_contents($htaccess_path) !== $htaccess_content) {
      
      if (file_put_contents($htaccess_path, $htaccess_content, LOCK_EX) === false) {
        \ZiziCache\CacheSys::writeError("Failed to create Apache .htaccess: $htaccess_path", 'Security');
        return false;
      } else {
        @chmod($htaccess_path, 0644);
        \ZiziCache\CacheSys::writeLog("Created Apache .htaccess: $htaccess_path", 'Security');
        return true;
      }
    }
    
    return true;
  }

  /**
   * Create NGINX security notice and sample configuration
   * 
   * @param string $dir Directory path
   * @param array $config Security configuration
   * @return bool Success status
   */
  private static function create_nginx_security_notice(string $dir, array $config): bool
  {
    $nginx_config = SecurityConfig::get_nginx_config($config);
    $notice_path = $dir . 'NGINX_SECURITY_README.txt';
    
    $notice_content = "# NGINX SECURITY CONFIGURATION REQUIRED
# Generated: " . date('Y-m-d H:i:s') . "
# 
# This directory contains sensitive cache files that need protection.
# Add the following configuration to your NGINX server block:

" . $nginx_config . "

# IMPORTANT: Restart NGINX after applying these rules!
# Test configuration: sudo nginx -t
# Reload NGINX: sudo systemctl reload nginx

# For more information, contact your hosting provider or system administrator.
";
    
    if (file_put_contents($notice_path, $notice_content, LOCK_EX) === false) {
      \ZiziCache\CacheSys::writeError("Failed to create NGINX security notice: $notice_path", 'Security');
      return false;
    } else {
      @chmod($notice_path, 0644);
      \ZiziCache\CacheSys::writeWarning("Created NGINX security notice: $notice_path - Manual configuration required!", 'Security');
      return true;
    }
  }

  /**
   * Create IIS web.config security
   * 
   * @param string $dir Directory path
   * @param array $config Security configuration
   * @return bool Success status
   */
  private static function create_iis_security(string $dir, array $config): bool
  {
    $webconfig_content = SecurityConfig::get_web_config($config['type']);
    $webconfig_path = $dir . 'web.config';
    
    if (!file_exists($webconfig_path) || file_get_contents($webconfig_path) !== $webconfig_content) {
      
      if (file_put_contents($webconfig_path, $webconfig_content, LOCK_EX) === false) {
        \ZiziCache\CacheSys::writeError("Failed to create IIS web.config: $webconfig_path", 'Security');
        return false;
      } else {
        @chmod($webconfig_path, 0644);
        \ZiziCache\CacheSys::writeLog("Created IIS web.config: $webconfig_path", 'Security');
        return true;
      }
    }
    
    return true;
  }

  /**
   * Create generic security (fallback)
   * 
   * @param string $dir Directory path
   * @param array $config Security configuration
   * @return bool Success status
   */
  private static function create_generic_security(string $dir, array $config): bool
  {
    // Create index.php to prevent directory browsing
    $index_path = $dir . 'index.php';
    $index_content = "<?php\n// Silence is golden.\n";
    
    if (!file_exists($index_path)) {
      if (file_put_contents($index_path, $index_content, LOCK_EX) === false) {
        \ZiziCache\CacheSys::writeError("Failed to create security index: $index_path", 'Security');
        return false;
      } else {
        @chmod($index_path, 0644);
        \ZiziCache\CacheSys::writeLog("Created security index: $index_path", 'Security');
      }
    }
    
    // Create security notice
    $notice_path = $dir . 'SECURITY_README.txt';
    $notice_content = "# SECURITY NOTICE
# Generated: " . date('Y-m-d H:i:s') . "
# 
# This directory contains sensitive cache files.
# Please ensure your web server is configured to:
# 1. Block direct access to .php, .db, .log files
# 2. Prevent directory browsing
# 3. Set appropriate file permissions (644 for files, 755 for directories)
# 
# Contact your hosting provider for server-specific security configuration.
";
    
    if (file_put_contents($notice_path, $notice_content, LOCK_EX) === false) {
      \ZiziCache\CacheSys::writeError("Failed to create security notice: $notice_path", 'Security');
      return false;
    } else {
      @chmod($notice_path, 0644);
      \ZiziCache\CacheSys::writeWarning("Created generic security notice: $notice_path", 'Security');
    }
    
    return true;
  }

  /**
   * Get NGINX security configuration
  /**
   * Enhanced filename sanitization to prevent path traversal and security issues
   * 
   * @param string $filename Raw filename from URL
   * @param string $url_source Source URL for fallback naming
   * @return string Sanitized filename
   */
  private static function sanitize_external_filename(string $filename, string $url_source): string
  {
    // Remove any path traversal attempts completely
    $filename = basename($filename);
    $filename = str_replace(['..', '/', '\\', "\0"], '', $filename);
    
    // Remove leading dots to prevent hidden files
    $filename = ltrim($filename, '.');
    
    // Whitelist only safe characters (remove dots from regex to prevent ..)
    $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
    
    // Prevent dangerous extensions that could bypass server protection
    $dangerous_exts = ['php', 'phtml', 'php3', 'php4', 'php5', 'phar', 'asp', 'aspx', 'jsp', 'exe', 'bat', 'cmd', 'sh'];
    $pathinfo = pathinfo($filename);
    
    if (isset($pathinfo['extension']) && in_array(strtolower($pathinfo['extension']), $dangerous_exts)) {
      $filename = $pathinfo['filename'] . '.txt';
    }
    
    // Ensure filename isn't empty after sanitization and create secure fallback
    if (empty($filename) || strlen($filename) < 3) {
      // Use timestamp + random to prevent collision attacks
      $filename = 'external_' . time() . '_' . substr(md5($url_source . wp_salt('secure_auth')), 0, 8) . '.cache';
    }
    
    // Limit filename length for filesystem compatibility
    if (strlen($filename) > 100) {
      $pathinfo = pathinfo($filename);
      $extension = isset($pathinfo['extension']) ? '.' . $pathinfo['extension'] : '';
      $filename = substr($pathinfo['filename'], 0, 100 - strlen($extension)) . $extension;
    }
    
    return $filename;
  }

  /**
   * Validate external file content for malicious code and security threats
   * 
   * @param string $content File content to validate
   * @param string $extension Expected file extension
   * @param string $url_source Source URL for logging
   * @return bool True if content is safe, false if malicious
   */
  private static function validate_external_file_content(string $content, string $extension, string $url_source): bool
  {
    // Block empty content
    if (empty($content) || strlen($content) < 10) {
      \ZiziCache\CacheSys::writeWarning('External file content too short: ' . $url_source, 'Security');
      return false;
    }
    
    // Block extremely large files (additional check)
    if (strlen($content) > 15 * 1024 * 1024) { // 15MB limit
      \ZiziCache\CacheSys::writeWarning('External file too large: ' . strlen($content) . ' bytes from ' . $url_source, 'Security');
      return false;
    }
    
    // Critical: Block PHP code injection patterns
    $php_patterns = [
      '/<\?php/',           // PHP opening tag
      '/<\?=/',             // PHP short echo tag
      '/<\?(?![xml])/i',    // PHP opening (not XML)
      '/\beval\s*\(/i',     // eval() function
      '/\bexec\s*\(/i',     // exec() function
      '/\bsystem\s*\(/i',   // system() function
      '/\bpassthru\s*\(/i', // passthru() function
      '/\bshell_exec\s*\(/i', // shell_exec() function
      '/\b`[^`]*`/',        // backtick execution
      '/\$_(?:GET|POST|REQUEST|COOKIE|SESSION|FILES|SERVER)\[/', // Superglobals
    ];
    
    foreach ($php_patterns as $pattern) {
      if (preg_match($pattern, $content)) {
        \ZiziCache\CacheSys::writeError('BLOCKED: PHP code injection detected in external file: ' . $url_source, 'Security');
        return false;
      }
    }
    
    // Block JavaScript injection in CSS files  
    if ($extension === 'css') {
      $js_patterns = [
        '/<script[^>]*>.*?<\/script>/is',
        '/javascript:/i',
        '/expression\s*\(/i',
        '/\bon\w+\s*=/i', // onclick, onload, etc.
        '/url\s*\(\s*[\'"]?javascript:/i'
      ];
      
      foreach ($js_patterns as $pattern) {
        if (preg_match($pattern, $content)) {
          \ZiziCache\CacheSys::writeError('BLOCKED: JavaScript injection detected in CSS file: ' . $url_source, 'Security');
          return false;
        }
      }
    }
    
    // Block common malicious patterns
    $malicious_patterns = [
      '/\bwindow\.location\b/i',
      '/\bdocument\.write\b/i',
      '/\bdocument\.cookie\b/i',
      '/XMLHttpRequest/i',
      '/fetch\s*\(/i',
      '/\bimport\s+/i',
      '/\brequire\s*\(/i'
    ];
    
    foreach ($malicious_patterns as $pattern) {
      if (preg_match($pattern, $content)) {
        \ZiziCache\CacheSys::writeWarning('BLOCKED: Suspicious pattern detected in external file: ' . $url_source, 'Security');
        return false;
      }
    }
    
    // Additional validation for specific file types
    if ($extension === 'css' && !self::validate_css_syntax($content)) {
      \ZiziCache\CacheSys::writeWarning('BLOCKED: Invalid CSS syntax in external file: ' . $url_source, 'Security');
      return false;
    }
    
    return true;
  }

  /**
   * Basic CSS syntax validation
   * 
   * @param string $content CSS content to validate
   * @return bool True if valid CSS syntax
   */
  private static function validate_css_syntax(string $content): bool
  {
    // Remove comments and whitespace for analysis
    $clean_content = preg_replace('/\/\*.*?\*\//s', '', $content);
    $clean_content = preg_replace('/\s+/', ' ', trim($clean_content));
    
    // Basic CSS structure validation - must contain valid CSS selectors and properties
    if (empty($clean_content)) {
      return false;
    }
    
    // Check for basic CSS structure (selector { property: value; })
    if (!preg_match('/[a-zA-Z0-9\-\.\#\:,\s\[\]=\"\']+\s*\{[^}]*\}/', $clean_content)) {
      // Allow @import, @charset, @media rules
      if (!preg_match('/@(import|charset|media|font-face|keyframes)/', $clean_content)) {
        return false;
      }
    }
    
    // Check for balanced braces
    $open_braces = substr_count($clean_content, '{');
    $close_braces = substr_count($clean_content, '}');
    
    return $open_braces === $close_braces;
  }

  /**
   * Parse HTML attributes string into associative array
   * 
   * @param string $attributes_string HTML attributes string
   * @return array Associative array of attributes
   */
  public static function get_atts_array(string $attributes_string): array
  {
    $attributes = [];
    
    // Match attribute="value" or attribute='value' or attribute=value or standalone attribute
    preg_match_all('/(\w+)(?:\s*=\s*(["\']?)([^"\'>\s]*)\2)?/i', $attributes_string, $matches, PREG_SET_ORDER);
    
    foreach ($matches as $match) {
      $attr_name = strtolower($match[1]);
      $attr_value = isset($match[3]) ? $match[3] : $attr_name; // For boolean attributes like 'disabled'
      $attributes[$attr_name] = $attr_value;
    }
    
    return $attributes;
  }

  /**
   * Convert associative array of attributes back to HTML string
   * 
   * @param array $attributes Associative array of attributes
   * @return string HTML attributes string
   */
  public static function get_atts_string(array $attributes): string
  {
    $atts_string = '';
    
    foreach ($attributes as $name => $value) {
      if (is_bool($value)) {
        if ($value) {
          $atts_string .= ' ' . esc_attr($name);
        }
      } else {
        $atts_string .= ' ' . esc_attr($name) . '="' . esc_attr($value) . '"';
      }
    }
    
    return trim($atts_string);
  }
}