<?php

namespace ZiziCache;

use ZiziCache\Security;

/**
 * Configuration Management Class
 * 
 * Handles configuration loading, storage, and updates for Zizi Cache plugin.
 * Provides centralized access to configuration settings across the plugin.
 *
 * @package ZiziCache
 */
class SysConfig
{
  /**
   * Current configuration storage variable
   * 
   * @var array Contains all configuration settings
   */
  public static $config;

  /**
   * Default configuration settings
   * 
   * Used as a base for initialization and migrations
   * 
   * @var array
   */
  protected static $initial_config = [
    // Redis Object Cache defaults
    'redis_enabled'        => false,
    'redis_host'           => '',
    'redis_password'       => '',
    'redis_port'           => 6379,
    'redis_key_salt'       => '',
    'redis_timeout'        => 1,
    'redis_read_timeout'   => 1,
    'redis_maxttl'         => 300,
    
    // Database and Redis configuration
    'redis_database'       => 1,
    'redis_global_groups'  => [],
    'cache_lifespan' => 'never',
    'cache_ignore_queries' => [           // Enhanced with modern tracking parameters
      'utm_source',
      'utm_medium', 
      'utm_campaign',
      'utm_term',
      'utm_content',
      'utm_id',
      'utm_source_platform',
      'utm_creative_format',
      'utm_marketing_tactic',
      'fbclid',
      'gclid',
      'gclsrc',
      'dclid',
      'msclkid',
      'gad_source',
      'gad_campaignid',
      'wbraid', 
      'gbraid',
      '_ga',
      '_gl',
      'mc_cid',
      'mc_eid',
      'pk_source',
      'pk_medium',
      'pk_campaign',
      'piwik_source',
      'piwik_medium',
      'piwik_campaign',
      'mtm_source',
      'mtm_medium',
      'mtm_campaign',
      'ref',
      'affiliate_id',
      'referrer',
      'age-verified',
      'ao_noptimize',
      'usqp',
      'cn-reloaded',
      '_ke',
      'redirect_log_mongo_id',
      'redirect_mongo_id',
      'sb_referer_host',
      'mkwid',
      'pcrid',
      'ef_id',
      's_kwcid',
      'dm_i',
      'igshid',
      'epik',
      'twclid',
      'ttclid', 
      'li_fat_id',
    ],
    'cache_include_queries' => [],
    'cache_logged_in' => false,
    'cache_bypass_urls' => [],
    'cache_bypass_cookies' => [],
    'cache_preload' => true,
    'opcache_auto_flush_after_upgrade' => false,
    'css_minify' => true,
    'css_rucss' => false,
    'css_rucss_method' => 'async',
    'css_rucss_exclude_stylesheets' => [],
    'css_rucss_include_selectors' => [],
    'css_critical' => false,
    'css_critical_method' => 'inline', // inline, file
    'css_critical_exclude_google_fonts' => true,
    'css_critical_exclude_stylesheets' => [],
    'css_critical_viewport_height' => 1200,
    'js_minify' => true,
    'js_preload_links' => true,
    'js_defer' => false,
    'js_defer_excludes' => [],
    'js_debug_level' => 'basic', // 'off', 'basic', 'verbose' - reduced default logging
    'js_delay' => false,
    'js_delay_method' => 'selected',
    'js_delay_all_excludes' => [],
    'js_delay_selected' => [
      'googletagmanager.com',
      'google-analytics.com',
      'googleoptimize.com',
      'adsbygoogle.js',
      'xfbml.customerchat.js',
      'fbevents.js',
      'widget.manychat.com',
      'cookie-law-info',
      'grecaptcha.execute',
      'static.hotjar.com',
      'hs-scripts.com',
      'embed.tawk.to',
      'disqus.com/embed.js',
      'client.crisp.chat',
      'matomo.js',
      'usefathom.com',
      'code.tidio.co',
      'metomic.io',
      'js.driftt.com',
      'cdn.onesignal.com',
      'clarity.ms',
    ],
    'js_delay_excludes' => [
      'jquery',
      'wp-includes',
      'elementor',
      'divi-builder', 
      'beaver-builder',
      'bricks',                          // Bricks Builder support
      'themes/bricks/',                  // Bricks Builder theme files
      'bricks.min.js',                   // Bricks Builder main script
      'bricks-scripts-js-extra',         // Bricks Builder inline scripts
      'splide',                          // Bricks Builder splide
      'swiper',                          // Bricks Builder swiper
      'vc_',
      'fusion-',
      'avada',
      'cookieyes',                       // CookieYes compatibility
      'cookieyes-banner',                // Enhanced CookieYes support
      'thrive-quiz-builder',             // Thrive Quiz Builder
      'etch-',                           // Etch page builder
    ],
    // JavaScript Debug Logging Level
    'js_debug_level' => 'basic', // 'off', 'basic', 'verbose'
    // User Interaction Delay - Advanced script delay until user interaction
    'js_user_interaction_delay' => false,
    'js_user_interaction_method' => 'smart', // smart, interaction, timeout
    'js_user_interaction_timeout' => 3000,   // Fallback timeout in milliseconds
    'js_user_interaction_inline_limit' => 50000, // Max size for inline scripts (bytes, 0 = no limit)
    'js_user_interaction_selected' => [      // Scripts to delay (when using selective mode)
      'embed.tawk.to',
      'client.crisp.chat',
      'code.tidio.co',
      'cdn.onesignal.com',
      'disqus.com/embed.js',
      'widget.manychat.com',
      'intercom.io',
      'zendesk.com',
      'livechatinc.com',
      'olark.com',
      'freshchat.com',
      'zopim.com',
      'paypal.com/sdk',
      'stripe.com',
      'youtube.com/iframe_api',
      'player.vimeo.com',
      'platform.twitter.com/widgets.js',
      'addthis.com',
      'sharethis.com',
    ],
    'js_user_interaction_excludes' => [      // Never delay these scripts
      'jquery',
      'wp-includes',
      'wp-admin', 
      'wp-content/themes',
      'wp-content/plugins',
      'core.min.js',
      'quicklink',
      'speculation',
      'zizi-cache',
      'wp-rocket',
      'w3tc',
      'wp-optimize',
      'litespeed',
      'autoptimize',
      'elementor',
      'divi-builder',
      'beaver-builder',
      'vc_',
      'fusion-',
      'avada',
      'bricks',                          // Bricks Builder support
      'themes/bricks/',                  // Bricks Builder theme files
      'bricks.min.js',                   // Bricks Builder main script
      'bricks-scripts-js-extra',         // Bricks Builder inline scripts
      'splide',                          // Bricks Builder splide
      'swiper',                          // Bricks Builder swiper
      'cookieyes',                       // CookieYes compatibility
      'cookieyes-banner',                // Enhanced CookieYes support
      'thrive-quiz-builder',             // Thrive Quiz Builder
      'etch-',                           // Etch page builder
      'typekit.net',                     // Adobe Fonts
      'woocommerce',
      'contact-form-7',
      'wpforms',
      'gravity',
      'ninja-forms',
      'recaptcha/api.js',
      'hcaptcha.com',
      'cmplz_',
    ],
    'js_lazy_render_selectors' => [],
    'js_lazy_render' => true,
    'js_lazy_render_method' => 'hybrid',           // hybrid, css, javascript
    'js_lazy_render_auto_detect' => true,         // Auto-detect common WordPress elements
    'js_lazy_render_height_cache' => true,        // Cache element heights for stability
    'js_lazy_render_root_margin' => '200px 0px',  // Viewport offset for triggering
    'self_host_third_party_css_js' => true,
    
    // Font Settings - with master switch
    'fonts_enabled' => true,              // NEW: Master switch for all font optimizations
    'fonts_optimize_google_fonts' => true,
    'fonts_display_swap' => true,
    'fonts_preload_urls' => [],
    'fonts_allow_cdn_domains' => true,
    'fonts_debug' => false,
    'fonts_language_priority' => 'auto', // NEW: Language priority for unicode-range deduplication (auto, latin, cyrillic, arabic, vietnamese, greek, hebrew, thai, cjk, preserve-all)
    'fonts_woff2_fallback' => true,      // NEW: Use woff2 fallback when font type unavailable
    // Hidden settings with defaults (not exposed in admin)
    'fonts_atf_height' => 1200,       // Height for above-the-fold content detection (pixels)
    'fonts_cache_ttl' => 5184000,     // Cache time-to-live for font resources (60 days in seconds)
    
    // Early Hints Support - NEW FEATURE
    'early_hints_enabled' => false,      // NEW: Enable Early Hints for faster resource loading
    'early_hints_types' => [            // NEW: Resource types to send Early Hints for
      'css' => true,
      'font' => true,
      'image' => false,                  // Conservative default - can cause issues
      'script' => false,                 // Conservative default
    ],
    'early_hints_crossorigin' => true,   // NEW: Add crossorigin attribute to Early Hints
    'early_hints_fetchpriority' => true, // NEW: Add fetchpriority attribute to Early Hints
    
    'img_lazyload' => true,
    'img_lazyload_exclude_count' => 2,
    'img_lazyload_excludes' => [],
    'img_width_height' => true,
    'img_localhost_gravatar' => false,
    'img_preload' => true,
    'img_responsive' => true,
    'img_priority_hints' => true,     // New: Priority Hints with combination of importance attribute
    'img_auto_lcp' => true,           // New: Automatic LCP detection using Layout Instability API
    'img_auto_lcp_admin_debug' => true, // New: Show LCP detection for administrators
    'img_auto_lcp_public_debug' => false, // New: Show LCP detection for non-logged-in users
    'iframe_lazyload' => false,
    'iframe_lazyload_excludes' => [], // New: CSS selectors to exclude from lazy loading
    'iframe_youtube_placeholder' => false,
    'iframe_youtube_quality' => 'auto', // New: YouTube thumbnail quality (auto, high, medium)
    'iframe_vimeo_placeholder' => false, // New: Replace Vimeo iframe with thumbnail image
    'iframe_optimize_attributes' => true, // New: Optimize attributes for better performance
    'redundancy_remove_google_fonts' => false,
    'redundancy_remove_dashicons' => false,
    'redundancy_remove_restapi' => false,
    'redundancy_post_revisions_control' => false,
    'redundancy_post_revisions_limit' => 5, // Limit of revisions per post
    'redundancy_heartbeat_control' => false,
    'redundancy_heartbeat_behaviour' => 'default',
    'redundancy_heartbeat_frequency' => 15,
    
    // Database settings
    'db_post_revisions' => false,
    'db_post_auto_drafts' => false,
    'db_post_trashed' => false,
    'db_comments_spam' => false,
    'db_comments_trashed' => false,
    'db_transients_expired' => false,
    'db_transients_all' => false,
    'db_optimize_tables' => false,
    'db_schedule_clean' => 'daily',
    'db_batch_size' => 500, // Batch size for batch processing
    'db_safe_mode' => true, // Safe mode for database operations
    'db_actionscheduler_clean' => true,
    'db_actionscheduler_logs_days' => 2,
    'db_actionscheduler_actions_days' => 7,
    'db_growth_monitoring' => true, // Database growth monitoring
    'db_growth_alert_threshold' => 30, // Threshold for growth alert (%)
    'db_growth_alert_emails' => '', // Alert emails, comma-separated
    'db_adaptive_cleaning' => false, // Adaptive cleaning scheduling
    
    // Speculative Loading
    'speculative_loading_mode' => 'off', // 'off', 'prefetch', 'prerender'
    'speculative_loading_eagerness' => 'moderate', // 'conservative', 'moderate', 'eager'
    'speculative_loading_exclusions' => [],
    'speculative_loading_analytics_guard' => true,
    'speculative_loading_ga_id' => '',
    'speculative_loading_google_ads_id' => '',
    'speculative_loading_facebook_pixel_id' => '',
    'speculative_loading_ms_clarity_id' => '',
    'speculative_loading_custom_guard' => '',
    'speculative_loading_conditional' => false,
    'speculative_loading_page_types' => [],
    'speculative_loading_same_origin' => true,
    'speculative_loading_require_https' => false,
    'speculative_loading_anonymous_ip' => true,
    
    // Quicklink settings
    'quicklink_enabled' => false,
    'quicklink_mode' => 'prefetch',
    'quicklink_throttle' => 5,
    'quicklink_limit' => 20,
    'quicklink_delay' => 0,
    'quicklink_timeout' => 2000,
    'quicklink_threshold' => 0,
    'quicklink_priority' => false,
    'quicklink_origins' => [],
    'quicklink_ignores' => [],
    'quicklink_respect_data_saver' => true,
    'quicklink_min_connection' => '3g',
    'quicklink_block_ads' => true,
    'quicklink_ad_patterns' => [],
    'quicklink_conditional' => false,
    'quicklink_page_types' => [],
    
    // Enhanced Quicklink Features - NEW
    'quicklink_loading_strategy' => 'intersection',  // NEW: intersection, hover, instant
    'quicklink_visual_indicators' => false,         // NEW: Show visual indicators for prefetched links
    'quicklink_analytics' => false,                 // NEW: Send events to analytics
    'quicklink_debug' => false,                     // NEW: Debug mode for development
    
    // Early Hints Debug Setting - NEW
    'early_hints_debug' => false,                   // NEW: Debug mode for Early Hints
    
    // Image Converter settings
    'image_converter_enabled' => true,
    'image_converter_formats' => ['webp'], // Default to WebP only for safety
    'image_converter_quality' => 80,
    'image_converter_avif_quality' => 75,
    'image_converter_max_width' => 1920,
    'image_converter_max_height' => 1080,
    'image_converter_keep_originals' => false, // Default to deleting originals
    'image_converter_remove_larger' => true,
    'image_converter_backup_before_delete' => false,
    'image_converter_allowed_types' => ['image/jpeg', 'image/png', 'image/gif'],
    'image_converter_auto_alt' => false,
    'image_converter_frontend_serve' => true,
    'image_converter_htaccess_rules' => true,
    
    // Logging and Debug settings
    'log_level' => 'error', // 'none', 'error', 'warning', 'info', 'debug', 'verbose'
    'log_throttle_enabled' => true, // Enable log throttling to prevent spam
    'log_throttle_threshold' => 10, // Log every Nth occurrence of repeated messages
    'frontend_debug_logging' => false, // Enable verbose frontend debugging logs
    
    // Bulk Optimization Performance settings
    'bulk_performance_mode' => 'auto', // auto, shared, managed, vps, dedicated
    'bulk_max_batch_size' => 0, // 0 = auto-detect, override with specific number
    'bulk_batch_delay' => 0, // 0 = auto-detect, override with specific seconds
    'bulk_memory_threshold' => 80, // Memory usage threshold percentage
    'bulk_concurrent_limit' => 0, // 0 = auto-detect, override with specific limit
    'bulk_adaptive_tuning' => true, // Enable adaptive performance tuning
    'bulk_performance_logging' => false, // Log performance metrics for debugging
    
    // Thumbnail Management settings
    'thumbnail_disabled_sizes' => [],
    'thumbnail_cleanup_enabled' => false,
    'thumbnail_prevention_enabled' => true,
    'thumbnail_disable_big_image_threshold' => false,
    'thumbnail_disable_exif_rotate' => false,
  ];

  public static function init()
  {
    // Retrieve saved configuration from the database
    self::$config = get_option('ZIZI_CACHE_CONFIG', []);

    // If the saved version differs from the current one, run migration/update
    $saved_version = get_option('ZIZI_CACHE_VERSION');
    $current_version = ZIZI_CACHE_VERSION;

    if ($saved_version !== $current_version || empty(self::$config)) {
      update_option('ZIZI_CACHE_VERSION', $current_version);
      self::migrate_config();
    }

    // Create directory for third-party files if the feature is enabled
    if (!empty(self::$config['self_host_third_party_css_js']) && defined('ZIZI_CACHE_CACHE_DIR')) {
      $third_party_dir = ZIZI_CACHE_CACHE_DIR . '3rd-css-js/';
      if (!is_dir($third_party_dir)) {
        wp_mkdir_p($third_party_dir);
        @chmod($third_party_dir, 0755);
        
        // Ensure security files are created for new cache directories
        \ZiziCache\SysTool::ensure_cache_security();
      }
    }

    // Remove configuration on plugin uninstall
    register_uninstall_hook(ZIZI_CACHE_FILE_NAME, [__CLASS__, 'on_uninstall']);
  }

  public static function migrate_config()
  {
    // Add new fields from the default configuration if they do not exist in the saved config
    self::$config = array_merge(self::$initial_config, self::$config);

    update_option('ZIZI_CACHE_CONFIG', self::$config);
    do_action('zizi_cache_update_config:after', self::$config);
    do_action('zizi_cache_upgraded');
  }

  // Function to update configuration
  public static function update_config($new_config = [])
  {
    $old_config = self::$config;
    
    // Make sure boolean values are properly converted and validated
    foreach ($new_config as $key => $value) {
      // Handle boolean values that might be strings from form submissions
      if (is_string($value) && ($value === 'true' || $value === '1')) {
        $new_config[$key] = true;
      } elseif (is_string($value) && ($value === 'false' || $value === '0' || $value === '')) {
        $new_config[$key] = false;
      }
      
      // Special handling for js_defer and other critical JavaScript settings
      if ($key === 'js_defer' || $key === 'js_delay' || $key === 'js_minify' || $key === 'js_user_interaction_delay') {
        $new_config[$key] = (bool)$value;
      }
      
      // Security: Validate and sanitize User Interaction Script Delay settings
      if ($key === 'js_user_interaction_timeout') {
        $new_config[$key] = max(1000, min(10000, intval($value))); // 1-10 seconds max
      }
      
      if ($key === 'js_user_interaction_inline_limit') {
        $new_config[$key] = max(0, min(500000, intval($value))); // 0-500KB max (0 = no limit)
      }
      
      if ($key === 'js_user_interaction_method') {
        $allowed_methods = ['smart', 'all', 'selected'];
        if (!in_array($value, $allowed_methods, true)) {
          $new_config[$key] = 'smart'; // Default to safe option
        }
      }
      
      // Sanitize array inputs for security
      if (in_array($key, ['js_user_interaction_selected', 'js_user_interaction_excludes'])) {
        if (is_array($value)) {
          $new_config[$key] = array_map('sanitize_text_field', $value);
          $new_config[$key] = array_filter($new_config[$key]); // Remove empty values
        }
      }
    }

    // Pokud je v novém configu redis_password, zašifrujeme ho
    if (isset($new_config['redis_password']) && !empty($new_config['redis_password'])) {
      $new_config['redis_password'] = Security::encrypt_password($new_config['redis_password']);
    }
    
    // REMOVED: lemon_squeezy_api_key encryption - no longer used with api.zizicache.com
    
    self::$config = array_merge(self::$config, $new_config);

    // Check if self-hosting feature was enabled and create directory
    if (isset($new_config['self_host_third_party_css_js']) &&
        $new_config['self_host_third_party_css_js'] === true && 
        (!isset($old_config['self_host_third_party_css_js']) || $old_config['self_host_third_party_css_js'] === false)) {
      // Create the directory for third-party files if it doesn't exist
      $third_party_dir = ZIZI_CACHE_CACHE_DIR . '3rd-css-js/';
      if (!is_dir($third_party_dir)) {
        wp_mkdir_p($third_party_dir);
        @chmod($third_party_dir, 0755);
        error_log('ZiziCache: Created directory for third-party files: ' . $third_party_dir);
        
        // Ensure security protection for newly created cache directories
        \ZiziCache\SysTool::ensure_cache_security();
      }
    }

    // Check if self-hosting feature was disabled and trigger cleanup
    if (isset($new_config['self_host_third_party_css_js']) &&
        $new_config['self_host_third_party_css_js'] === false && 
        (!isset($old_config['self_host_third_party_css_js']) || $old_config['self_host_third_party_css_js'] === true)) {
      // Clean up the third-party files
      if (class_exists('\\ZiziCache\\SysTool')) {
        SysTool::cleanup_third_party_files();
      }
    }
    
    update_option('ZIZI_CACHE_CONFIG', self::$config);
    do_action('zizi_cache_update_config:after', self::$config);
  }

  /**
   * Get configuration value by key
   * 
   * @param string $key Configuration key
   * @param mixed $default Default value if key not found
   * @return mixed Configuration value
   */
  public static function get(string $key, $default = null)
  {
    return self::$config[$key] ?? $default;
  }

  /**
   * Set configuration value by key
   * 
   * @param string $key Configuration key
   * @param mixed $value Configuration value
   * @return bool Success status
   */
  public static function set(string $key, $value): bool
  {
    try {
      self::$config[$key] = $value;
      // Update the option in the database
      return update_option('zizi_cache_config', self::$config);
    } catch (\Exception $e) {
      return false;
    }
  }

  /**
   * Alias for update_config method for thumbnail management compatibility
   * 
   * @param array $config Configuration array to update
   * @return bool Success status
   */
  public static function updateConfig(array $config): bool
  {
    try {
      self::update_config($config);
      return true;
    } catch (\Exception $e) {
      if (class_exists('\\ZiziCache\\CacheSys')) {
        \ZiziCache\CacheSys::writeError("Config update failed: " . $e->getMessage(), 'SysConfig');
      }
      return false;
    }
  }

  public static function on_uninstall()
  {
    delete_option('ZIZI_CACHE_CONFIG');
    delete_option('ZIZI_CACHE_VERSION');
  }
}
