<?php

namespace ZiziCache;

/**
 * Speculative Loading implementation for ZiziCache
 *
 * Implements prefetch and prerender functionality using the Speculation Rules API
 * with enhanced security features, validation, analytics protection, and conditional
 * execution based on page types. Supports advanced configuration attributes and
 * browser capability detection.
 *
 * @package ZiziCache
 * @link https://developer.chrome.com/blog/speculationrules/ Speculation Rules API
 */
class SpeculativeLoading
{
    /**
     * Initializes the Speculative Loading functionality
     *
     * Hooks into wp_head to add speculation rules and sets up
     * conditional triggering filter if enabled in configuration.
     *
     * @return void
     */
    public static function init()
    {
        add_action('wp_head', [__CLASS__, 'output_speculation_rules'], 99);

        // Conditional execution - check page type
        if (!empty(SysConfig::$config['speculative_loading_conditional'])) {
            add_filter('zizi_cache_should_apply_speculation', [__CLASS__, 'check_conditional'], 10, 1);
        }
    }

    /**
     * Outputs speculation rules to the page head based on configuration
     *
     * Generates and outputs the appropriate <script type="speculationrules"> tag
     * with prefetch or prerender rules. Applies advanced attributes, handles exclusions,
     * performs browser capability detection, and implements analytics protection.
     *
     * @return void
     */
    public static function output_speculation_rules()
    {
        $config = SysConfig::$config;
        $mode = isset($config['speculative_loading_mode']) ? $config['speculative_loading_mode'] : 'off';
        $eagerness = isset($config['speculative_loading_eagerness']) ? $config['speculative_loading_eagerness'] : 'auto';
        $exclusions = isset($config['speculative_loading_exclusions']) ? $config['speculative_loading_exclusions'] : [];

        if ($mode === 'off') return;

        // Conditional execution - applies a filter that can prevent rendering based on page type
        if (apply_filters('zizi_cache_should_apply_speculation', true) === false) {
            return;
        }

        $rule_type = ($mode === 'prerender') ? 'prerender' : 'prefetch';

        // More specific rules - support for advanced attributes
        $rules = [
            $rule_type => [
                [
                    'source' => 'document',
                    'eagerness' => $eagerness,
                ]
            ]
        ];

        // Add advanced attributes
        $rules = self::add_advanced_attributes($rules, $rule_type, $config);

        // Sanitize exclusions
        $exclusions = self::sanitize_exclusions((array)$exclusions, 50);
        if (!empty($exclusions)) {
            $rules['exclude'] = $exclusions;
        }

        // Browser support detection - first render detection, then rules
        echo '<script>' . "\n";
        echo "// Browser support detection for speculation rules\n";
        echo "let ziziSpeculationSupported = 'supportsSpeculationRules' in document;\n";
        echo "if (ziziSpeculationSupported) {\n";
        echo "  let ziziSpecRules = document.createElement('script');\n";
        echo "  ziziSpecRules.type = 'speculationrules';\n";
        echo "  ziziSpecRules.textContent = " . json_encode(json_encode($rules, JSON_UNESCAPED_SLASHES)) . ";\n";
        echo "  document.head.appendChild(ziziSpecRules);\n";
        echo "}\n";
        echo '</script>' . "\n";

        // Unified analytics guard during prerender (GA, Google Ads, FB Pixel, MS Clarity, custom)
        $gaId        = self::sanitize_analytics_id($config['speculative_loading_ga_id'] ?? '', 'ga');
        $adsId       = self::sanitize_analytics_id($config['speculative_loading_google_ads_id'] ?? '', 'ads');
        $fbPixelId   = self::sanitize_analytics_id($config['speculative_loading_facebook_pixel_id'] ?? '', 'fb');
        $clarityId   = self::sanitize_analytics_id($config['speculative_loading_ms_clarity_id'] ?? '', 'clarity');
        $customGuard = self::validate_custom_js($config['speculative_loading_custom_guard'] ?? '');

        if (!empty($config['speculative_loading_analytics_guard']) && ($gaId || $adsId || $fbPixelId || $clarityId || $customGuard)) {
            echo '<script>' . "\n";
            echo "if (document.visibilityState === 'prerender') {" . "\n";
            if ($gaId) {
                echo "  window['ga-disable-{$gaId}'] = true;" . "\n";
            }
            if ($adsId) {
                echo "  window['ga-disable-{$adsId}'] = true;" . "\n";
            }
            if ($fbPixelId) {
                echo "  window.fbq = function(){};" . "\n";
            }
            if ($clarityId) {
                echo "  window.clarity = function(){};" . "\n";
            }
            if ($customGuard) {
                echo "  {$customGuard}" . "\n";
            }
            echo "}" . "\n";
            echo '</script>' . "\n";
        }
    }

    /**
     * Sanitizes and limits the number of exclusion patterns.
     *
     * Processes an array of URL exclusion patterns, sanitizes each entry,
     * removes empty values, and limits to a maximum number of items.
     *
     * @param array $exclusions Array of URL patterns to sanitize
     * @param int   $maxItems   Maximum number of patterns to keep (default: 150)
     * @return array            Sanitized array of URL patterns
     */
    private static function sanitize_exclusions(array $exclusions, $maxItems = 150)
    {
        $cleaned = [];
        foreach ($exclusions as $exc) {
            $exc = sanitize_text_field($exc);
            if (!empty($exc)) {
                $cleaned[] = $exc;
            }
            if (count($cleaned) >= $maxItems) {
                break;
            }
        }
        return $cleaned;
    }

    /**
     * Sanitizes and validates IDs for analytics platforms.
     *
     * Performs type-specific validation for various analytics platforms
     * to ensure IDs match expected formats. Falls back to basic
     * sanitization for non-matching values.
     *
     * @param string $id   Analytics ID to sanitize
     * @param string $type Type of analytics ('ga', 'ads', 'fb', 'clarity')
     * @return string      Sanitized analytics ID
     */
    private static function sanitize_analytics_id($id, $type)
    {
        $id = trim($id);

        switch ($type) {
            case 'ga':
                // Google Analytics - typically UA-XXXXX-Y or G-XXXXXXX
                if (preg_match('/^(UA-\d{4,10}-\d{1,4}|G-[A-Z0-9]{4,10})$/i', $id)) {
                    return $id;
                }
                break;

            case 'ads':
                // Google Ads - typically AW-XXXXXXXXXX
                if (preg_match('/^(AW-\d{9,12}|AW-\d{9,12}\/\d{9,12})$/i', $id)) {
                    return $id;
                }
                break;

            case 'fb':
                // Facebook Pixel - digits only
                if (preg_match('/^\d{10,16}$/', $id)) {
                    return $id;
                }
                break;

            case 'clarity':
                // Microsoft Clarity - alphanumeric characters
                if (preg_match('/^[a-z0-9]{10,32}$/i', $id)) {
                    return $id;
                }
                break;
        }

        // If it does not meet specific requirements, use general sanitization
        return preg_replace('/[^a-z0-9\-_\.]/i', '', $id);
    }

    /**
     * Validates and sanitizes custom JavaScript code.
     *
     * Checks for dangerous patterns, limits code length,
     * and logs suspicious code. Returns empty string if
     * the code fails security checks.
     *
     * @param string $js JavaScript code to validate
     * @return string    Validated code or empty string if invalid
     */
    private static function validate_custom_js($js)
    {
        $js = trim($js);

        // Basic security checks
        $dangerous_patterns = [
            'document.write',
            'eval(',
            'setTimeout(',
            'setInterval(',
            'createElement("script")',
            'innerHTML',
            'fetch(',
            'XMLHttpRequest'
        ];

        foreach ($dangerous_patterns as $pattern) {
            if (stripos($js, $pattern) !== false) {
                // If it contains dangerous patterns, log and return an empty string
                CacheSys::writeLog('[SpeculativeLoading] Dangerous JS pattern detected in custom guard: ' . $pattern);
                return '';
            }
        }

        // Length limitation
        if (strlen($js) > 1000) {
            CacheSys::writeLog('[SpeculativeLoading] Custom JS guard too long: ' . strlen($js) . ' chars');
            return '';
        }

        return $js;
    }

    /**
     * Determines whether to apply speculation rules based on the current page type.
     *
     * Checks if the current page matches one of the enabled page types
     * in the configuration. Used as a callback for the zizi_cache_should_apply_speculation filter.
     *
     * @param bool $apply Default filter value
     * @return bool       True if speculation should be applied to the current page
     */
    public static function check_conditional($apply)
    {
        $config = SysConfig::$config;
        $page_types = isset($config['speculative_loading_page_types']) ? (array)$config['speculative_loading_page_types'] : [];

        // If nothing is set, apply everywhere
        if (empty($page_types)) {
            return $apply;
        }

        // Check various page types
        if (in_array('home', $page_types) && is_home()) {
            return true;
        }

        if (in_array('front_page', $page_types) && is_front_page()) {
            return true;
        }

        if (in_array('singular', $page_types) && is_singular()) {
            return true;
        }

        if (in_array('archive', $page_types) && is_archive()) {
            return true;
        }

        if (in_array('search', $page_types) && is_search()) {
            return true;
        }

        // For other page types (not included), use the default value
        return false;
    }

    /**
     * Adds advanced attributes to speculation rules based on configuration.
     *
     * Enhances rule definitions with attributes like 'where' (same-origin)
     * and 'requires' (https, anonymous-client-ip) based on settings.
     *
     * @param array  $rules     Original rules array
     * @param string $rule_type Type of rule ('prerender' or 'prefetch')
     * @param array  $config    Plugin configuration
     * @return array            Enhanced rules with additional attributes
     */
    private static function add_advanced_attributes($rules, $rule_type, $config)
    {
        // Add 'where' attribute - e.g., restrict to same domain
        if (!empty($config['speculative_loading_same_origin'])) {
            $rules[$rule_type][0]['where'] = 'same-origin';
        }

        // Add 'requires' attribute - e.g., require HTTPS
        if (!empty($config['speculative_loading_require_https'])) {
            $rules[$rule_type][0]['requires'] = ['anonymous-client-ip', 'https'];
        } elseif (!empty($config['speculative_loading_anonymous_ip'])) {
            $rules[$rule_type][0]['requires'] = ['anonymous-client-ip'];
        }

        return $rules;
    }
}
