<?php

namespace ZiziCache;

/**
 * Class SecurityConfig
 * 
 * Handles generation of security configuration files (.htaccess, web.config, nginx)
 * for different directories in ZiziCache plugin.
 * 
 * @package ZiziCache
 */
class SecurityConfig
{    /**
     * Generate .htaccess content for main page cache directory
     * Allows HTML, GZ (page cache), SQLite, TXT, CSS, JS, and font files
     * 
     * @return string .htaccess content for main cache directory
     */    public static function get_main_cache_htaccess(): string
    {
        return '# ZiziCache Main Cache Security - SELECTIVE PROTECTION
# Generated: ' . date('Y-m-d H:i:s') . '
# This directory contains page cache files (HTML, GZ), CSS, JS, TXT, fonts, and SQLite files

# Block dangerous file types
<FilesMatch "\.(php|phtml|phar|db|sql|log|inc|conf|sh|py|pl|rb|exe|dll|bat|cmd)$">
    Order deny,allow
    Deny from all
</FilesMatch>

# Allow page cache files, CSS, JS, TXT, and font files
<FilesMatch "\.(html|htm|gz|css|js|txt|woff2|woff|ttf|eot|svg)$">
    Order allow,deny
    Allow from all
    
    # Set security headers
    Header always set X-Content-Type-Options nosniff
</FilesMatch>

# Special handling for cached .gz files (OLS server compatibility)
<FilesMatch "index\.html\.gz$">
    Order allow,deny
    Allow from all
    
    # Force correct MIME type for gzipped HTML cache files
    Header unset Content-Type
    Header set Content-Type "text/html; charset=UTF-8"
    Header set Content-Encoding "gzip"
    Header unset Content-Disposition
    Header set Content-Disposition "inline"
    Header set Cache-Control "public, max-age=31536000"
    Header set Vary "Accept-Encoding"
    Header always set X-Content-Type-Options nosniff
</FilesMatch>

# Handle other .gz files (CSS, JS) differently
<FilesMatch "\.(css|js)\.gz$">
    Order allow,deny
    Allow from all
    
    # Set appropriate content type based on file extension
    RewriteEngine On
    RewriteCond %{REQUEST_FILENAME} \.css\.gz$
    RewriteRule .* - [E=FILETYPE:text/css]
    RewriteCond %{REQUEST_FILENAME} \.js\.gz$
    RewriteRule .* - [E=FILETYPE:application/javascript]
    
    Header set Content-Type "%{FILETYPE}e; charset=UTF-8"
    Header set Content-Encoding "gzip"
    Header always set X-Content-Type-Options nosniff
</FilesMatch>

# Allow SQLite files for async warmup (with forced download)
<FilesMatch "\.sqlite$">
    Order allow,deny
    Allow from all
    
    # Force download for SQLite files
    ForceType application/octet-stream
    Header set Content-Disposition "attachment"
    Header always set X-Content-Type-Options nosniff
</FilesMatch>

# Block access to hidden files
<FilesMatch "^\.">
    Order deny,allow
    Deny from all
</FilesMatch>

# Prevent directory browsing
Options -Indexes

# Block backup and temporary files
<FilesMatch "\.(bak|backup|tmp|temp|old|orig|save|swp|swo|~)$">
    Order deny,allow
    Deny from all
</FilesMatch>

# Basic security rules
RewriteEngine On
RewriteCond %{QUERY_STRING} \.\./\.\. [OR]
RewriteCond %{QUERY_STRING} (exec|system|passthru|shell_exec) [NC]
RewriteRule .* - [F,L]';
    }

    /**
     * Generate .htaccess content for plugin root directory
     * Only blocks log files, allows other plugin files to function
     * 
     * @return string .htaccess content for plugin root with log file protection
     */
    public static function get_plugin_root_htaccess(): string
    {
        return '# ZiziCache Plugin Root Security - LOG FILES ONLY
# Generated: ' . date('Y-m-d H:i:s') . '
# MINIMAL protection: blocks ONLY log files, allows plugin functionality

# Block ONLY log files (CRITICAL SECURITY)
<FilesMatch "\.(log|logs)$">
    Order deny,allow
    Deny from all
</FilesMatch>

# Security headers for plugin files
<FilesMatch "\.(php|js|css)$">
    Header always set X-Content-Type-Options nosniff
</FilesMatch>';
    }    /**
     * Generate .htaccess content for external files directory (3rd-css-js/)
     * Allows CSS, JS, and font files, blocks everything else  
     * 
     * @return string .htaccess content for external files directory
     */    public static function get_external_files_htaccess(): string
    {
        return '# ZiziCache External Files Security - CSS/JS/FONTS ONLY
# Generated: ' . date('Y-m-d H:i:s') . '
# This directory contains external CSS/JS files and font files

# Allow CSS, JS, and font files
<FilesMatch "\.(css|js|woff2|woff|ttf|eot|svg)$">
    Order allow,deny
    Allow from all
    
    # Security headers
    Header always set X-Content-Type-Options nosniff
</FilesMatch>

# Block ALL other file types (including PHP, DB, logs)
<FilesMatch "\.(php|phtml|phar|db|sql|log|inc|conf|sh|py|pl|rb|exe|dll|bat|cmd|txt|html|htm|gz|sqlite)$">
    Order deny,allow
    Deny from all
</FilesMatch>

# Block access to hidden files
<FilesMatch "^\.">
    Order deny,allow
    Deny from all
</FilesMatch>

# Prevent directory browsing
Options -Indexes

# Block backup and temporary files
<FilesMatch "\.(bak|backup|tmp|temp|old|orig|save|swp|swo|~)$">
    Order deny,allow
    Deny from all
</FilesMatch>

# Default deny for unknown extensions
<Files "*">
    Order deny,allow
    Deny from all
</Files>

# Re-allow CSS, JS, and font files (override default deny)
<FilesMatch "\.(css|js|woff2|woff|ttf|eot|svg)$">
    Order allow,deny
    Allow from all
</FilesMatch>';
    }

    /**
     * Generate web.config content for IIS servers
     * 
     * @param string $type Type of directory: 'main_cache', 'plugin_root', 'external_files'
     * @return string web.config content
     */
    public static function get_web_config(string $type): string
    {
        $base_config = '<?xml version="1.0" encoding="UTF-8"?>
<configuration>
    <system.webServer>
        <httpHeaders>
            <add name="X-Content-Type-Options" value="nosniff" />
        </httpHeaders>
        <security>
            <requestFiltering>';

        switch ($type) {
            case 'main_cache':
                return $base_config . '
                <!-- Block dangerous file types -->
                <fileExtensions>
                    <add fileExtension=".php" allowed="false" />
                    <add fileExtension=".phtml" allowed="false" />
                    <add fileExtension=".phar" allowed="false" />
                    <add fileExtension=".db" allowed="false" />
                    <add fileExtension=".sql" allowed="false" />
                    <add fileExtension=".log" allowed="false" />
                    <add fileExtension=".inc" allowed="false" />
                    <add fileExtension=".conf" allowed="false" />                </fileExtensions>
            </requestFiltering>
        </security>
    </system.webServer>
</configuration>';

            case 'plugin_root':
                return $base_config . '
                <!-- Block ONLY log files -->
                <fileExtensions>
                    <add fileExtension=".log" allowed="false" />
                    <add fileExtension=".logs" allowed="false" />
                </fileExtensions>
            </requestFiltering>
        </security>
    </system.webServer>
</configuration>';            case 'external_files':
                return $base_config . '
                <!-- Allow CSS, JS, and font files -->
                <fileExtensions allowUnlisted="false">
                    <add fileExtension=".css" allowed="true" />
                    <add fileExtension=".js" allowed="true" />
                    <add fileExtension=".woff2" allowed="true" />
                    <add fileExtension=".woff" allowed="true" />
                    <add fileExtension=".ttf" allowed="true" />
                    <add fileExtension=".eot" allowed="true" />
                    <add fileExtension=".svg" allowed="true" />
                </fileExtensions>
            </requestFiltering>
        </security>
    </system.webServer>
</configuration>';

            default:
                return $base_config . '
            </requestFiltering>
        </security>
    </system.webServer>
</configuration>';
        }
    }

    /**
     * Generate Nginx configuration
     * 
     * @param array $config Configuration array with 'type' and cache directory info
     * @return string Nginx configuration
     */
    public static function get_nginx_config(array $config): string
    {
        $cache_dir = str_replace(ABSPATH, '/', ZIZI_CACHE_CACHE_DIR);

        switch ($config['type']) {            case 'external_files':
                return "# CRITICAL: External files directory protection
location " . $cache_dir . "3rd-css-js/ {
    # Allow CSS, JS, and font files
    location ~* \\.(css|js|woff2|woff|ttf|eot|svg)$ {
        access_log off;
        add_header X-Content-Type-Options nosniff;
    }
    
    # Block ALL other files (including PHP, DB, logs)
    location ~* \\.(php|phtml|phar|db|sql|log|txt)$ {
        deny all;
        return 403;
    }
    
    # Block hidden files and directories
    location ~ /\\. {
        deny all;
        return 403;
    }
    
    # Default deny for unknown file types
    location ~ .* {
        deny all;
        return 403;
    }
}";

            case 'plugin_root':
                $plugin_dir = str_replace(ABSPATH, '/', dirname(ZIZI_CACHE_LOG_FILE) . '/');
                return "# LOG FILES: Plugin root directory protection
location " . $plugin_dir . " {
    # Block access to log files ONLY (CRITICAL)
    location ~* \\.(log|logs)$ {
        deny all;
        return 403;
    }
}";            default: // main_cache
                return "# Selective protection for main cache directory
location " . $cache_dir . " {
    # Allow page cache files, CSS, JS, TXT, and font files
    location ~* \\.(html|htm|gz|css|js|txt|woff2|woff|ttf|eot|svg)$ {
        access_log off;
        add_header X-Content-Type-Options nosniff;
    }
    
    # Allow SQLite files (for async warmup)
    location ~* \\.sqlite$ {
        access_log off;
        add_header Content-Type \"application/octet-stream\";
        add_header Content-Disposition \"attachment\";
    }
    
    # Block dangerous files
    location ~* \\.(php|phtml|phar|db|sql|log)$ {
        deny all;
        return 403;
    }
    
    # Block hidden files
    location ~ /\\. {
        deny all;
        return 403;
    }
}";
        }
    }

    /**
     * Get allowed file extensions for specific directory type
     * 
     * @param string $type Directory type: 'main_cache', 'plugin_root', 'external_files'
     * @return array List of allowed file extensions
     */
    public static function get_allowed_extensions(string $type): array
    {        switch ($type) {
            case 'main_cache':
                return ['html', 'htm', 'gz', 'sqlite', 'css', 'js', 'txt', 'woff2', 'woff', 'ttf', 'eot', 'svg'];
            case 'external_files':
                return ['css', 'js', 'woff2', 'woff', 'ttf', 'eot', 'svg'];
            case 'plugin_root':
                return ['php', 'js', 'css', 'png', 'jpg', 'jpeg', 'gif', 'svg', 'woff', 'woff2', 'ttf', 'eot'];
            default:
                return [];
        }
    }

    /**
     * Get blocked file extensions for specific directory type
     * 
     * @param string $type Directory type: 'main_cache', 'plugin_root', 'external_files'
     * @return array List of blocked file extensions
     */
    public static function get_blocked_extensions(string $type): array
    {
        $dangerous_extensions = ['php', 'phtml', 'phar', 'db', 'sql', 'inc', 'conf', 'sh', 'py', 'pl', 'rb', 'exe', 'dll', 'bat', 'cmd'];
        
        switch ($type) {
            case 'plugin_root':
                return ['log', 'logs']; // Block only log files in plugin root
            default:
                return $dangerous_extensions;
        }
    }
}
