<?php
namespace ZiziCache;

/**
 * Security class for ZiziCache plugin
 * 
 * Provides encryption and decryption functionality for sensitive data
 * using WordPress AUTH_KEY and AUTH_SALT constants with AES-256-CBC encryption.
 * 
 * @package    ZiziCache
 * @subpackage Security
 * @since      1.0.0
 * @author     ZiziCache Team
 */
class Security
{
    /**
     * Encrypts a password using OpenSSL with keys from AUTH_KEY and AUTH_SALT
     * 
     * Uses AES-256-CBC encryption with a random initialization vector (IV).
     * The encryption key is derived from WordPress AUTH_KEY and AUTH_SALT constants
     * using SHA-256 hashing. If encryption fails or requirements are not met,
     * returns the original password as fallback.
     * 
     * @since  1.0.0
     * @param  string $password The password to encrypt
     * @return string Returns encrypted password (base64 encoded) or original password on failure
     * 
     * @example
     * $encrypted = Security::encrypt_password('my_secret_password');
     * // Returns: base64 encoded string with IV + encrypted data
     */
    public static function encrypt_password($password)
    {
        if (!defined('AUTH_KEY') || !defined('AUTH_SALT')) {
            return $password; // Fallback: don't encrypt if keys are not available
        }
        if (!function_exists('openssl_encrypt')) {
            return $password; // Fallback: don't encrypt if OpenSSL is not available
        }
        $key = hash('sha256', AUTH_KEY . AUTH_SALT, true);
        $ivlen = openssl_cipher_iv_length('aes-256-cbc');
        $iv = openssl_random_pseudo_bytes($ivlen);
        $ciphertext = openssl_encrypt($password, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        if ($ciphertext === false) {
            return $password; // Fallback: don't encrypt on error
        }
        return base64_encode($iv . $ciphertext);
    }

    /**
     * Decrypts a password that was encrypted with encrypt_password()
     * 
     * Decodes the base64 encoded data, extracts the initialization vector (IV)
     * and ciphertext, then decrypts using AES-256-CBC with the same key derivation
     * as the encryption method. If decryption fails or requirements are not met,
     * returns the original encrypted string as fallback.
     * 
     * @since  1.0.0
     * @param  string $encrypted The encrypted password (base64 encoded with IV)
     * @return string Returns decrypted password or original encrypted string on failure
     * 
     * @example
     * $decrypted = Security::decrypt_password($encrypted_password);
     * // Returns: original password string or encrypted string on failure
     */
    public static function decrypt_password($encrypted)
    {
        if (!defined('AUTH_KEY') || !defined('AUTH_SALT')) {
            return $encrypted; // Fallback: return as-is if keys are not available
        }
        if (!function_exists('openssl_decrypt')) {
            return $encrypted; // Fallback: return as-is if OpenSSL is not available
        }
        $key = hash('sha256', AUTH_KEY . AUTH_SALT, true);
        $data = base64_decode($encrypted, true);
        if ($data === false) {
            return $encrypted; // Fallback: return as-is if base64 decode fails
        }
        $ivlen = openssl_cipher_iv_length('aes-256-cbc');
        if (strlen($data) < $ivlen) {
            return $encrypted; // Fallback: return as-is if data is too short
        }
        $iv = substr($data, 0, $ivlen);
        $ciphertext = substr($data, $ivlen);
        $decrypted = openssl_decrypt($ciphertext, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        if ($decrypted === false) {
            return $encrypted; // Fallback: return as-is on decryption error
        }
        return $decrypted;
    }
}
