<?php
namespace ZiziCache\RestApi;

/**
 * Trait ValidationTrait
 *
 * Provides centralized input validation helpers for REST API endpoints.
 *
 * @package ZiziCache\RestApi
 */
trait ValidationTrait
{    /**
     * Validates a string by length.
     *
     * @param mixed $value
     * @param int $min
     * @param int|null $max
     * @return bool
     */
    protected function validate_string($value, $min = 0, $max = null)
    {
        if (!is_string($value)) return false;
        $len = strlen($value);
        if ($len < $min) return false;
        if ($max !== null && $len > $max) return false;
        return true;
    }    /**
     * Validates an integer value.
     *
     * @param mixed $value
     * @param int|null $min
     * @param int|null $max
     * @return bool
     */
    protected function validate_int($value, $min = null, $max = null)
    {
        if (!is_numeric($value)) return false;
        $int = (int)$value;
        if ($min !== null && $int < $min) return false;
        if ($max !== null && $int > $max) return false;
        return true;
    }    /**
     * Validates a URL for format and security (path traversal protection).
     *
     * @param mixed $url
     * @param array $allowed_schemes
     * @return bool
     */
    protected static function validate_url($url, $allowed_schemes = ['http', 'https'])
    {
        if (!is_string($url) || empty($url)) return false;
        
        // Basic URL validation
        $parsed = parse_url($url);
        if (!$parsed || !isset($parsed['scheme'], $parsed['host'])) return false;
        
        // Scheme whitelist
        if (!in_array($parsed['scheme'], $allowed_schemes)) return false;
        
        // Path traversal detection
        $path = $parsed['path'] ?? '/';
        $decoded_path = urldecode($path);
        
        // Block common path traversal sequences
        $dangerous_patterns = [
            '../', '..\\', '%2e%2e%2f', '%2e%2e%5c',
            '..%2f', '..%5c', '.%2e/', '.%2e\\',
            '%2e%2e/', '%2e%2e\\', '..%252f', '..%255c'
        ];
        
        foreach ($dangerous_patterns as $pattern) {
            if (stripos($decoded_path, $pattern) !== false) {
                return false;
            }
        }
        
        // Additional security: block null bytes and control characters
        if (strpos($decoded_path, "\0") !== false || preg_match('/[\x00-\x1f\x7f]/', $decoded_path)) {
            return false;
        }
        
        return true;
    }

    // Add more validation helpers as needed (arrays, enums, etc.)
}
