<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\RestApi\ValidationTrait;

/**
 * Class ServerInfo
 *
 * Handles REST API endpoint for server and WordPress environment information.
 *
 * @package ZiziCache\RestApi
 */
class ServerInfo
{
    use ValidationTrait;

    /**
     * Endpoint definitions for server info.
     *
     * @var array
     */    protected static $endpoints = [
        [
            'route' => '/server-info',
            'methods' => 'GET',
            'callback' => 'get_server_info_callback',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/performance-metrics',
            'methods' => 'GET',
            'callback' => 'get_performance_metrics_callback',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all server info endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'], [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Returns server and WordPress environment information.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function get_server_info_callback($request)
    {
        global $wpdb;
        $db_version = $wpdb->db_version();
        $db_type = strpos(strtolower($db_version), 'mariadb') !== false ? 'MariaDB' : 'MySQL';
        preg_match('/^[0-9\.]+/', $db_version, $matches);
        $db_version_number = $matches[0] ?? $db_version;
        $php_version = PHP_VERSION;
        $wp_memory_limit = WP_MEMORY_LIMIT;
        $wp_max_memory_limit = WP_MAX_MEMORY_LIMIT;
        $to_mb = function($size) {
            if (empty($size)) return 0;
            $unit = strtoupper(substr($size, -1));
            $value = (int) $size;
            switch ($unit) {
                case 'G': return $value * 1024;
                case 'M': return $value;
                case 'K': return $value / 1024;
                default: return $value / (1024 * 1024);
            }
        };
        $info = [
            'php_version' => $php_version,
            'php_version_ok' => version_compare($php_version, '8.0', '>='),
            'php_recommendation' => '8.2+',
            'db_type' => $db_type,
            'db_version' => $db_version,
            'db_version_number' => $db_version_number,
            'db_version_ok' => ($db_type === 'MariaDB')
                ? version_compare($db_version_number, '10.5', '>=')
                : version_compare($db_version_number, '8.0', '>='),
            'db_recommendation' => ($db_type === 'MariaDB') ? 'MariaDB 10.5+' : 'MySQL 8.0+',
            'wp_memory_limit' => $wp_memory_limit,
            'wp_max_memory_limit' => $wp_max_memory_limit,
            'memory_limit_ok' => ($to_mb($wp_memory_limit) >= 128 && $to_mb($wp_max_memory_limit) >= 128),
            'memory_recommendation' => '512M',
            'server_arch' => php_uname('m'),
            'web_server' => $_SERVER['SERVER_SOFTWARE'] ?? 'N/A',
            'php_sapi' => php_sapi_name(),
            'php_max_input_vars' => ini_get('max_input_vars'),
            'php_time_limit' => ini_get('max_execution_time'),
            'php_memory_limit' => ini_get('memory_limit'),
            'php_max_input_time' => ini_get('max_input_time'),
            'php_upload_max_filesize' => ini_get('upload_max_filesize'),
            'php_post_max_size' => ini_get('post_max_size'),
        ];
        return new \WP_REST_Response($info, 200);
    }

    /**
     * Returns performance metrics and analysis.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */    public static function get_performance_metrics_callback($request)
    {
        // Get current performance metrics
        $metrics = \ZiziCache\PerformanceMonitor::getMetrics();
        
        // Get historical data from SQLite
        $historical = [];
        if (class_exists('\\ZiziCache\\PreloadSqlite')) {
            $historical = [
                'database' => \ZiziCache\PreloadSqlite::getPerformanceHistory('database', 50, 7),
                'http' => \ZiziCache\PreloadSqlite::getPerformanceHistory('http', 50, 7),
                'stats' => \ZiziCache\PreloadSqlite::getPerformanceStats()
            ];
        }
        
        // Analyze performance health
        $health_status = self::analyzePerformanceHealth($metrics);
        $recommendations = self::getPerformanceRecommendations($metrics);
        
        return new \WP_REST_Response([
            'success' => true,
            'current_metrics' => $metrics,
            'historical_data' => $historical,
            'analysis' => [
                'status' => $health_status,
                'recommendations' => $recommendations,
                'timestamp' => current_time('mysql')
            ]
        ], 200);
    }

    /**
     * Analyze overall performance health
     * 
     * @param array $metrics
     * @return string
     */
    private static function analyzePerformanceHealth($metrics): string
    {
        $issues = 0;
        
        // Check database performance
        if (isset($metrics['database_performance']['efficiency_rating'])) {
            $rating = $metrics['database_performance']['efficiency_rating'];
            if ($rating === 'poor' || $rating === 'fair') {
                $issues++;
            }
        }
        
        // Check HTTP performance
        if (isset($metrics['http_performance']['error_rate'])) {
            if ($metrics['http_performance']['error_rate'] > 5) {
                $issues++;
            }
        }
        
        // Check memory usage
        if (isset($metrics['memory_usage']['current'], $metrics['memory_usage']['peak'])) {
            $current_mb = $metrics['memory_usage']['current'] / 1024 / 1024;
            if ($current_mb > 128) {
                $issues++;
            }
        }
        
        // Return status based on issues found
        if ($issues === 0) {
            return 'excellent';
        } elseif ($issues === 1) {
            return 'good';
        } elseif ($issues === 2) {
            return 'fair';
        } else {
            return 'poor';
        }
    }

    /**
     * Get performance recommendations
     * 
     * @param array $metrics
     * @return array
     */
    private static function getPerformanceRecommendations($metrics): array
    {
        $recommendations = [];
        
        // Database recommendations
        if (isset($metrics['database_performance']['recommendations'])) {
            $recommendations = array_merge($recommendations, $metrics['database_performance']['recommendations']);
        }
        
        // HTTP recommendations
        if (isset($metrics['http_performance']['error_rate']) && $metrics['http_performance']['error_rate'] > 5) {
            $recommendations[] = 'High HTTP error rate detected - check external API connections';
        }
        
        // Memory recommendations
        if (isset($metrics['memory_usage']['current'])) {
            $current_mb = $metrics['memory_usage']['current'] / 1024 / 1024;
            if ($current_mb > 128) {
                $recommendations[] = 'High memory usage detected - consider optimizing memory-intensive operations';
            }
        }
        
        // Cache recommendations
        if (isset($metrics['cache_efficiency']['opcache_enabled']) && !$metrics['cache_efficiency']['opcache_enabled']) {
            $recommendations[] = 'Enable OPcache for better PHP performance';
        }
        
        if (empty($recommendations)) {
            $recommendations[] = 'Performance is optimal - no recommendations at this time';
        }
        
        return $recommendations;
    }
}
