<?php

namespace ZiziCache\RestApi;

/**
 * Class ResponseCleaner
 * 
 * Provides utility methods to ensure clean JSON responses from REST API endpoints
 * by suppressing PHP warnings/notices and capturing unwanted output.
 * 
 * @package ZiziCache\RestApi
 */
class ResponseCleaner
{
    /**
     * Wraps a callback to ensure clean JSON output
     * 
     * @param callable $callback The original callback function
     * @param array $args Arguments to pass to the callback
     * @return mixed The callback result with clean output
     */
    public static function clean_response($callback, $args = [])
    {
        // Start output buffering to catch any unexpected output
        ob_start();
        
        // Temporarily suppress warnings/notices for clean JSON output
        $old_error_reporting = error_reporting();
        error_reporting(E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR | E_USER_ERROR | E_RECOVERABLE_ERROR);
        
        // Temporarily disable WordPress debug output
        $old_wp_debug = defined('WP_DEBUG') ? WP_DEBUG : false;
        if (defined('WP_DEBUG')) {
            // Cannot redefine constants, but we can control display
            $old_wp_debug_display = defined('WP_DEBUG_DISPLAY') ? WP_DEBUG_DISPLAY : true;
        }
        
        try {
            // Call the original callback
            if (is_array($args) && !empty($args)) {
                $result = call_user_func_array($callback, $args);
            } else {
                $result = call_user_func($callback, $args);
            }
            
            // Clean any output that might have been captured
            $captured_output = ob_get_clean();
            if (!empty($captured_output)) {
                // Log any captured output for debugging
                error_log('ZiziCache REST API: Captured unexpected output: ' . trim($captured_output));
            }
            
            // Restore original error reporting
            error_reporting($old_error_reporting);
            
            return $result;
            
        } catch (\Exception $e) {
            // Clean output buffer and restore error reporting
            ob_end_clean();
            error_reporting($old_error_reporting);
            
            // Log the error
            error_log('ZiziCache REST API Error: ' . $e->getMessage());
            
            return [
                'status' => 'error',
                'message' => 'Internal server error: ' . $e->getMessage(),
                'code' => 'internal_error'
            ];
        } catch (\Throwable $e) {
            // Clean output buffer and restore error reporting  
            ob_end_clean();
            error_reporting($old_error_reporting);
            
            // Log the error
            error_log('ZiziCache REST API Critical Error: ' . $e->getMessage());
            
            return [
                'status' => 'error', 
                'message' => 'Critical server error',
                'code' => 'critical_error'
            ];
        }
    }
    
    /**
     * Creates a clean REST API callback wrapper
     * 
     * @param callable $callback The callback to wrap
     * @return \Closure Clean callback wrapper
     */
    public static function wrap_callback($callback)
    {
        return function($request = null) use ($callback) {
            return self::clean_response($callback, [$request]);
        };
    }
}