<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\SysConfig;
use ZiziCache\RestApi\ValidationTrait;
use ZiziCache\Security;

/**
 * Class Redis
 *
 * Handles REST API endpoints for Redis status and cache flushing.
 *
 * @package ZiziCache\RestApi
 */
class Redis
{
    use ValidationTrait;

    /**
     * Endpoint definitions for Redis operations.
     *
     * @var array
     */    protected static $endpoints = [
        [
            'route' => '/redis-status',
            'methods' => 'GET',
            'callback' => 'redis_status',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/redis-flush',
            'methods' => 'POST',
            'callback' => 'flush_redis',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all Redis endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'] . '/?', [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Returns the Redis status.
     *
     * @return \WP_REST_Response
     */
    public static function redis_status()
    {
        try {
            $config = SysConfig::$config;
            $status = [
                'active'     => is_file(WP_CONTENT_DIR . '/object-cache.php'),
                'filesystem' => is_writable(WP_CONTENT_DIR),
                'redis'      => false,
                'error'      => null,
            ];
            
            // NOTE: Status API checks TECHNICAL state (object-cache.php + constants)
            // not UI state (redis_enabled checkbox). This ensures accurate status reporting.
            
            // Determine connection type (TCP vs Unix socket)
            $scheme = defined('WP_REDIS_SCHEME') ? WP_REDIS_SCHEME : 'tcp';
            $isUnixSocket = (strcasecmp($scheme, 'unix') === 0);
            
            if ($isUnixSocket) {
                // Unix socket connection
                $path = defined('WP_REDIS_PATH') ? WP_REDIS_PATH : null;
                if (!$path) {
                    $status['error'] = 'Unix socket path not configured (WP_REDIS_PATH missing)';
                    return rest_ensure_response($status);
                }
                
                $client_params = [
                    'scheme' => 'unix',
                    'path'   => $path,
                ];
            } else {
                // TCP connection
                $host = defined('WP_REDIS_HOST') ? WP_REDIS_HOST : ($config['redis_host'] ?? '127.0.0.1');
                $port = defined('WP_REDIS_PORT') ? WP_REDIS_PORT : ($config['redis_port'] ?? 6379);
                
                $client_params = [
                    'host' => (string) $host,
                    'port' => (int) $port,
                ];
            }
            
            // Common parameters for both connection types
            $password = defined('WP_REDIS_PASSWORD') ? WP_REDIS_PASSWORD : (Security::decrypt_password($config['redis_password']) ?: null);
            $database = defined('WP_REDIS_DATABASE') ? WP_REDIS_DATABASE : ($config['redis_database'] ?? 0);
            $timeout = defined('WP_REDIS_TIMEOUT') ? WP_REDIS_TIMEOUT : ($config['redis_timeout'] ?? 1);
            $readWriteTimeout = defined('WP_REDIS_READ_TIMEOUT') ? WP_REDIS_READ_TIMEOUT : ($config['redis_read_timeout'] ?? 1);
            
            $client_params['database'] = (int) $database;
            $client_params['timeout'] = (float) $timeout;
            $client_params['read_write_timeout'] = (float) $readWriteTimeout;
            
            // Only add password if it's not empty
            if (!empty($password)) {
                $client_params['password'] = $password;
            }
            
            // Enhanced Predis loading with fallback - ensure we use the same library as object-cache.php
            if (!class_exists('Predis\\Client') && !class_exists('\\Predis\\Client')) {
                error_log('ZiziCache: Predis\\Client class not found, attempting to load from plugin directory');
                
                // Try to load Predis the same way as object-cache.php does
                $predis_paths = [
                    ZIZI_CACHE_PLUGIN_DIR . 'vendor/predis/predis/autoload.php',
                    WP_PLUGIN_DIR . '/zizi-cache/vendor/predis/predis/autoload.php',
                    WP_CONTENT_DIR . '/plugins/zizi-cache/vendor/predis/predis/autoload.php'
                ];
                
                $predis_loaded = false;
                foreach ($predis_paths as $predis_path) {
                    if (is_readable($predis_path)) {
                        require_once $predis_path;
                        $predis_loaded = true;
                        error_log('ZiziCache: Predis loaded from: ' . $predis_path);
                        break;
                    }
                }
                
                if (!$predis_loaded) {
                    error_log('ZiziCache: Could not load Predis from any path');
                    $status['error'] = 'Predis library not found. Re-install ZiziCache plugin or delete the object-cache.php file if you want to disable Redis.';
                    return rest_ensure_response($status);
                }
            }
            
            // Check again after potential loading
            if (!class_exists('Predis\\Client') && !class_exists('\\Predis\\Client')) {
                error_log('ZiziCache: Predis\\Client class still not available after loading attempt');
                $status['error'] = 'Predis library could not be loaded. Please check plugin installation.';
                return rest_ensure_response($status);
            }
            
            try {
                // Debug log connection parameters (without password)
                if ($isUnixSocket) {
                    error_log('ZiziCache: Attempting Redis Unix socket connection to ' . $path . ' (database: ' . $database . ')');
                } else {
                    error_log('ZiziCache: Attempting Redis TCP connection to ' . $client_params['host'] . ':' . $client_params['port'] . ' (database: ' . $database . ')');
                }
                
                // Use the correct namespace - try both variants for compatibility
                $client_class = class_exists('Predis\\Client') ? 'Predis\\Client' : '\\Predis\\Client';
                
                // Debug log final parameters (without sensitive data)
                $debug_params = $client_params;
                if (isset($debug_params['password'])) {
                    $debug_params['password'] = '[REDACTED]';
                }
                error_log('ZiziCache: Client parameters: ' . json_encode($debug_params));
                
                $client = new $client_class($client_params);
                
                // Test connection with ping
                $ping_result = $client->ping();
                error_log('ZiziCache: Redis ping result: ' . var_export($ping_result, true));
                
                // Check various possible response formats
                $ping_ok = false;
                if ($ping_result === 'PONG') {
                    $ping_ok = true;
                } elseif (is_array($ping_result) && isset($ping_result[0]) && $ping_result[0] === 'PONG') {
                    $ping_ok = true;
                } elseif (is_object($ping_result)) {
                    // Handle Predis\Response\Status objects
                    if (method_exists($ping_result, 'getPayload')) {
                        $ping_ok = ($ping_result->getPayload() === 'PONG');
                    } elseif (isset($ping_result->payload)) {
                        $ping_ok = ($ping_result->payload === 'PONG');
                    } elseif (method_exists($ping_result, '__toString')) {
                        $ping_ok = ((string)$ping_result === 'PONG');
                    }
                }
                
                if ($ping_ok) {
                    $status['redis'] = true;
                    error_log('ZiziCache: Redis connection successful');
                } else {
                    error_log('ZiziCache: Redis ping returned unexpected result: ' . var_export($ping_result, true));
                    $status['error'] = 'Redis server responded but ping result was unexpected: ' . var_export($ping_result, true);
                }
            } catch (\Exception $e) {
                error_log('ZiziCache Redis connection error: ' . $e->getMessage());
                $status['error'] = 'Redis connection error: ' . $e->getMessage();
            }
            
            return rest_ensure_response($status);
        } catch (\Exception $e) {
            error_log('ZiziCache Redis status error: ' . $e->getMessage());
            return new \WP_Error('redis-status-error', $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * Flushes the Redis cache.
     *
     * @return array|\WP_Error
     */
    public static function flush_redis()
    {
        try {
            $config = SysConfig::$config;
            
            // Check technical Redis state first
            if (!is_file(WP_CONTENT_DIR . '/object-cache.php')) {
                return new \WP_Error('redis-not-active', 'Redis Object Cache is not active (object-cache.php not found)');
            }
            
            // UI state check - warn but don't block if checkbox is disabled
            if (!isset($config['redis_enabled']) || !$config['redis_enabled']) {
                error_log('ZiziCache: Flushing Redis despite UI being disabled (checkbox unchecked)');
            }
            
            // Enhanced Predis loading with fallback
            if (!class_exists('Predis\\Client') && !class_exists('\\Predis\\Client')) {
                error_log('ZiziCache: Predis\\Client class not found during flush, attempting to load');
                
                // Try to load Predis the same way as object-cache.php does
                $predis_paths = [
                    ZIZI_CACHE_PLUGIN_DIR . 'vendor/predis/predis/autoload.php',
                    WP_PLUGIN_DIR . '/zizi-cache/vendor/predis/predis/autoload.php',
                    WP_CONTENT_DIR . '/plugins/zizi-cache/vendor/predis/predis/autoload.php'
                ];
                
                $predis_loaded = false;
                foreach ($predis_paths as $predis_path) {
                    if (is_readable($predis_path)) {
                        require_once $predis_path;
                        $predis_loaded = true;
                        break;
                    }
                }
                
                if (!$predis_loaded) {
                    return new \WP_Error(
                        'redis-library-not-found', 
                        'Predis library not found. Re-install ZiziCache plugin or delete the object-cache.php file.'
                    );
                }
            }
            
            $config = SysConfig::$config;
            
            // Use the correct namespace - try both variants for compatibility
            $client_class = class_exists('Predis\\Client') ? 'Predis\\Client' : '\\Predis\\Client';
            
            // Determine connection type (TCP vs Unix socket)
            $scheme = defined('WP_REDIS_SCHEME') ? WP_REDIS_SCHEME : 'tcp';
            $isUnixSocket = (strcasecmp($scheme, 'unix') === 0);
            
            if ($isUnixSocket) {
                // Unix socket connection
                $path = defined('WP_REDIS_PATH') ? WP_REDIS_PATH : null;
                if (!$path) {
                    return new \WP_Error('redis-config-error', 'Unix socket path not configured (WP_REDIS_PATH missing)');
                }
                
                $client_params = [
                    'scheme' => 'unix',
                    'path'   => $path,
                ];
            } else {
                // TCP connection (fallback to config if constants not defined)
                $host = defined('WP_REDIS_HOST') ? WP_REDIS_HOST : ($config['redis_host'] ?? '127.0.0.1');
                $port = defined('WP_REDIS_PORT') ? WP_REDIS_PORT : ($config['redis_port'] ?? 6379);
                
                $client_params = [
                    'host' => (string) $host,
                    'port' => (int) $port,
                ];
            }
            
            // Common parameters for both connection types
            $database = defined('WP_REDIS_DATABASE') ? WP_REDIS_DATABASE : ($config['redis_database'] ?? 0);
            $timeout = defined('WP_REDIS_TIMEOUT') ? WP_REDIS_TIMEOUT : ($config['redis_timeout'] ?? 1);
            $readWriteTimeout = defined('WP_REDIS_READ_TIMEOUT') ? WP_REDIS_READ_TIMEOUT : ($config['redis_read_timeout'] ?? 1);
            
            $client_params['database'] = (int) $database;
            $client_params['timeout'] = (float) $timeout;
            $client_params['read_write_timeout'] = (float) $readWriteTimeout;
            
            // Only add password if it's not empty
            $password = defined('WP_REDIS_PASSWORD') ? WP_REDIS_PASSWORD : (Security::decrypt_password($config['redis_password']) ?: null);
            if (!empty($password)) {
                $client_params['password'] = $password;
            }
            
            $client = new $client_class($client_params);
            $client->flushdb();
            
            error_log('ZiziCache: Redis cache flushed successfully');
            return ['success' => true];
        } catch (\Exception $e) {
            error_log('ZiziCache Redis flush error: ' . $e->getMessage());
            return new \WP_Error('redis-flush-error', $e->getMessage());
        }
    }
}
