<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\OPcache;
use ZiziCache\CacheSys;

/**
 * Class OPcacheApi
 *
 * Handles REST API endpoints for OPcache management (stats, flush, domain usage).
 *
 * @package ZiziCache\RestApi
 */
class OPcacheApi
{
    /**
     * Endpoint definitions for OPcache management.
     *
     * @var array
     */    protected static $endpoints = [
        [
            'route' => '/opcache/stats',
            'methods' => 'GET',
            'callback' => 'get_stats',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/opcache/flush',
            'methods' => 'POST',
            'callback' => 'flush_opcache',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/opcache/domain-usage',
            'methods' => 'GET',
            'callback' => 'get_domain_usage',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all OPcache management endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'] . '/?', [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Returns OPcache statistics.
     *
     * @param \WP_REST_Request $request The REST request object.
     * @return \WP_REST_Response The response containing OPcache statistics.
     */
    public static function get_stats(\WP_REST_Request $request)
    {
        return rest_ensure_response(\ZiziCache\OPcache::get_basic_stats());
    }

    /**
     * Flushes PHP OPcache.
     *
     * @param \WP_REST_Request $request The REST request object.
     * @return \WP_REST_Response|\WP_Error The response or error after attempting to flush OPcache.
     */
    public static function flush_opcache(\WP_REST_Request $request)
    {
        try {
            $result = \ZiziCache\OPcache::flush();
            
            if ($result) {
                CacheSys::writeLog('[INFO] ZiziCache: OPcache flush succeeded');
                return rest_ensure_response(['success' => true, 'message' => 'OPcache successfully flushed']);
            } else {
                // Get detailed error information
                $error_message = 'OPcache flush failed';
                
                // Check if opcache is enabled
                if (!extension_loaded('Zend OPcache')) {
                    $error_message = 'Zend OPcache extension is not loaded';
                } elseif (!function_exists('opcache_reset')) {
                    $error_message = 'opcache_reset function does not exist';
                } elseif (ini_get('opcache.restrict_api') && !in_array($_SERVER['SERVER_ADDR'], explode(',', ini_get('opcache.restrict_api')))) {
                    $error_message = 'OPcache is restricted via opcache.restrict_api setting';
                }
                
                CacheSys::writeLog('[ERROR] ZiziCache: OPcache flush failed: ' . $error_message);
                return new \WP_Error(
                    'opcache_flush_failed',
                    $error_message,
                    ['status' => 500]
                );
            }
        } catch (\Throwable $e) {
            $error_message = 'Exception while flushing OPcache: ' . $e->getMessage();
            CacheSys::writeLog('[ERROR] ZiziCache: ' . $error_message . ' in ' . $e->getFile() . ':' . $e->getLine());
            
            return new \WP_Error(
                'opcache_flush_exception',
                $error_message,
                ['status' => 500]
            );
        }
    }

    /**
     * Returns OPcache memory usage by domain and type.
     *
     * @param \WP_REST_Request $request The REST request object.
     * @return \WP_REST_Response The response containing OPcache domain memory usage.
     */
    public static function get_domain_usage(\WP_REST_Request $request)
    {
        try {
            $data = \ZiziCache\OPcache::get_domain_memory_usage();
            if (isset($data['error'])) {
                CacheSys::writeLog('[WARNING] ZiziCache: OPcache domain usage error: ' . $data['error']);
            }
            return rest_ensure_response($data);
        } catch (\Throwable $e) {
            $error_message = 'Exception while getting OPcache domain usage: ' . $e->getMessage();
            CacheSys::writeLog('[ERROR] ZiziCache: ' . $error_message . ' in ' . $e->getFile() . ':' . $e->getLine());
            
            return new \WP_Error(
                'opcache_domain_usage_exception',
                $error_message,
                ['status' => 500]
            );
        }
    }
}