<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\LicenseManager;
use ZiziCache\RestApi\ValidationTrait;
use ZiziCache\CacheSys;

/**
 * ZiziCache REST API Endpoints for License Management - Production Ready
 * 
 * Provides secure endpoints for license activation, validation, and status checking
 * All endpoints are protected by Authority permission callbacks
 * Updated for ZiziCache proxy API integration
 * 
 * @package ZiziCache
 * @version 2.0
 * @security Protected by nonce validation and sanitization, proxy-based communication
 */
class License 
{
    use ValidationTrait;

    /**
     * Register all license-related REST endpoints
     */    public static function register(): void 
    {
        register_rest_route('zizi-cache/v1', '/license/activate', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'activate'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
            'args' => [
                'license_key' => [
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => [__CLASS__, 'validate_license_key'],
                    'sanitize_callback' => 'sanitize_text_field',
                ]
            ]
        ]);
        
        register_rest_route('zizi-cache/v1', '/license/status', [
            'methods' => 'GET', 
            'callback' => [__CLASS__, 'status'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
        
        register_rest_route('zizi-cache/v1', '/license/validate', [
            'methods' => 'POST', 
            'callback' => [__CLASS__, 'validate'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
            'args' => [
                'force_remote' => [
                    'type' => 'boolean',
                    'default' => false,
                    'sanitize_callback' => 'rest_sanitize_boolean',
                ]
            ]
        ]);
        
        register_rest_route('zizi-cache/v1', '/license/deactivate', [
            'methods' => 'POST', 
            'callback' => [__CLASS__, 'deactivate'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
          // Registrujeme oba endpointy pro kompatibilitu - updater-status (s pomlčkou)
        register_rest_route('zizi-cache/v1', '/license/updater-status', [
            'methods' => ['GET', 'POST'],
            'callback' => [__CLASS__, 'updater_status'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
        
        // Registrujeme duplicitní endpoint - updaterstatus (bez pomlčky)
        register_rest_route('zizi-cache/v1', '/license/updaterstatus', [
            'methods' => ['GET', 'POST'],
            'callback' => [__CLASS__, 'updater_status'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
        
        // Debug endpoint for cache statistics
        register_rest_route('zizi-cache/v1', '/license/debug', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'debug_cache_stats'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
        
        // Test endpoint for license cache optimization
        register_rest_route('zizi-cache/v1', '/license/test-cache', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'test_license_cache'],
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'],
        ]);
    }
    
    /**
     * Handle license activation request
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response|\WP_Error
     */
    public static function activate(\WP_REST_Request $request) 
    {
        $license_key = $request->get_param('license_key');
        
        if (!$license_key) {
            CacheSys::writeError('License activation API called without license key', 'License');
            return new \WP_Error('missing_key', 'License key is required', ['status' => 400]);
        }
        
        CacheSys::writeLog('License activation API called', 'License');
        
        $result = LicenseManager::activate($license_key);
        
        if ($result['success']) {
            $license_status = LicenseManager::get_status();
            
            // Remove sensitive data from response
            if ($license_status) {
                unset($license_status['license_key_encrypted']);
                unset($license_status['license_key_hash']);
            }
            
            return rest_ensure_response([
                'success' => true,
                'message' => $result['message'],
                'license_status' => $license_status,
                'timestamp' => current_time('mysql')
            ]);
        } else {
            return new \WP_Error('activation_failed', $result['message'], ['status' => 400]);
        }
    }
      /**
     * Get current license status
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response
     */
    public static function status(\WP_REST_Request $request) 
    {
        $license = LicenseManager::get_status();
        $is_active = LicenseManager::is_active();
        
        // Remove sensitive data from response
        if ($license) {
            unset($license['license_key_encrypted']);
            unset($license['license_key_hash']);
        }
        
        return rest_ensure_response([
            'license' => $license,
            'is_active' => $is_active,
            'last_check' => current_time('mysql'),
            'production_mode' => true // Always production mode now
        ]);
    }
    
    /**
     * Validate current license
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response
     */
    public static function validate(\WP_REST_Request $request) 
    {
        $force_remote = $request->get_param('force_remote');
        
        CacheSys::writeLog('License validation API called' . ($force_remote ? ' (forced remote)' : ''), 'License');
        
        $result = LicenseManager::validate($force_remote);
        
        return rest_ensure_response([
            'valid' => $result['valid'],
            'message' => $result['message'],
            'cached' => $result['cached'] ?? false,
            'fallback' => $result['fallback'] ?? false,
            'remote' => $result['remote'] ?? false,
            'timestamp' => current_time('mysql')
        ]);
    }
    
    /**
     * Deactivate current license
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response|\WP_Error
     */
    public static function deactivate(\WP_REST_Request $request) 
    {
        CacheSys::writeLog('License deactivation API called', 'License');
        
        $result = LicenseManager::deactivate();
        
        if ($result['success']) {
            return rest_ensure_response([
                'success' => true,
                'message' => $result['message'],
                'timestamp' => current_time('mysql')
            ]);
        } else {
            return new \WP_Error('deactivation_failed', $result['message'], ['status' => 400]);
        }
    }    /**
     * Get updater status and configuration
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response
     */
    public static function updater_status(\WP_REST_Request $request) 
    {
        $is_active = LicenseManager::is_active();
        
        // Clear update cache before checking - ensures fresh data
        $cache_key = 'zizi_cache_ls_update_' . md5(plugin_basename(ZIZI_CACHE_FILE));
        delete_transient($cache_key);
        delete_site_transient('update_plugins');
        
        // Clear object cache if available
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete($cache_key, 'transient');
        }
        
        CacheSys::writeLog('Updater status API called and all update caches cleared', 'License');
        
        $status = [
            'license_active' => $is_active,
            'production_mode' => true, // Always production mode
            'proxy_integration' => true, // ZiziCache proxy integration
            'updater_enabled' => $is_active && class_exists('ZiziCache\LemonSqueezyUpdater'),
            'api_url' => 'https://api.zizicache.com',
            'current_version' => ZIZI_CACHE_VERSION,
            'check_frequency' => '5 minutes cache',
            'last_check' => current_time('mysql'),
            'requirements' => [
                'license_active' => $is_active,
                'updater_class_exists' => class_exists('ZiziCache\LemonSqueezyUpdater'),
                'proxy_configured' => true // Always true with ZiziCache proxy
            ],
            'info' => 'Auto-updater uses ZiziCache proxy API (production ready)'
        ];
        
        // If license is active, try to get remote version info
        if ($is_active && class_exists('ZiziCache\LemonSqueezyUpdater')) {
            $updater = new \ZiziCache\LemonSqueezyUpdater(ZIZI_CACHE_FILE, ZIZI_CACHE_VERSION);
            $remote_version = $updater->get_remote_version();
            if ($remote_version) {
                $status['remote_version'] = $remote_version->version;
                $status['update_available'] = version_compare(ZIZI_CACHE_VERSION, $remote_version->version, '<');
                $status['download_url'] = $remote_version->download_url ?? null;
            } else {
                $status['remote_version'] = 'Could not fetch';
                $status['update_available'] = false;
                $status['error'] = 'Failed to connect to ZiziCache API';
            }
        }
        
        return rest_ensure_response($status);
    }
    
    /**
     * Debug cache statistics
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response
     */
    public static function debug_cache_stats(\WP_REST_Request $request) 
    {
        $stats = LicenseManager::get_cache_stats();
        
        // Add SQL query count if available
        global $wpdb;
        $stats['total_sql_queries'] = $wpdb->num_queries ?? 'not_available';
        
        return rest_ensure_response([
            'cache_stats' => $stats,
            'timestamp' => current_time('mysql'),
            'info' => 'Cache statistics for current request'
        ]);
    }
    
    /**
     * Test license cache optimization
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response
     */
    public static function test_license_cache(\WP_REST_Request $request) 
    {
        // Clear any existing cache first
        LicenseManager::clear_memory_cache();
        
        // Count SQL queries before test
        global $wpdb;
        $queries_before = $wpdb->num_queries ?? 0;
        
        $test_results = [];
        
        // Test multiple calls to is_active()
        for ($i = 1; $i <= 5; $i++) {
            $queries_before_call = $wpdb->num_queries ?? 0;
            
            $is_active = LicenseManager::is_active();
            
            $queries_after_call = $wpdb->num_queries ?? 0;
            $queries_this_call = $queries_after_call - $queries_before_call;
            
            $test_results[] = [
                'call_number' => $i,
                'is_active' => $is_active,
                'sql_queries_this_call' => $queries_this_call,
                'cached' => $queries_this_call <= 1
            ];
        }
        
        // Count total queries
        $queries_after = $wpdb->num_queries ?? 0;
        $total_queries_test = $queries_after - $queries_before;
        
        $optimization_success = $total_queries_test <= 1;
        
        return rest_ensure_response([
            'test_results' => $test_results,
            'cache_stats' => LicenseManager::get_cache_stats(),
            'summary' => [
                'total_sql_queries_during_test' => $total_queries_test,
                'optimization_success' => $optimization_success,
                'message' => $optimization_success ? 
                    'SUCCESS: Multiple is_active() calls resulted in only ' . $total_queries_test . ' SQL queries' :
                    'FAILED: Multiple is_active() calls resulted in ' . $total_queries_test . ' SQL queries (should be 1 or less)'
            ],
            'timestamp' => current_time('mysql')
        ]);
    }
    
    /**
     * Validate license key format
     * 
     * @param mixed $value License key value
     * @param \WP_REST_Request $request REST request object
     * @param string $param Parameter name
     * @return bool|\WP_Error True if valid, WP_Error if invalid
     */
    public static function validate_license_key($value, $request, $param) 
    {
        if (!is_string($value)) {
            return new \WP_Error('invalid_type', 'License key must be a string');
        }
        
        if (strlen($value) < 20) {
            return new \WP_Error('invalid_length', 'License key is too short');
        }
        
        // UUID format check (Lemon Squeezy license keys are UUIDs)
        if (!preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $value)) {
            return new \WP_Error('invalid_format', 'License key must be in UUID format (xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx)');
        }
        
        return true;
    }
}
