<?php
declare(strict_types=1);

namespace ZiziCache\RestApi;

use ZiziCache\ImageConverter;
use ZiziCache\SysConfig;
use ZiziCache\Authority;

/**
 * Image Converter REST API Admin endpoints
 * @package ZiziCache\RestApi
 */
class ImageConverterAdmin
{

    /**
     * Register REST API routes
     */
    public static function register(array $namespaces = ['zizi-cache/v1']): void
    {
        $route_base = 'image-converter';
        
        foreach ($namespaces as $namespace) {
            register_rest_route($namespace, '/' . $route_base . '/settings', [
                ['methods' => 'GET', 'callback' => [__CLASS__, 'get_settings'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']],
                ['methods' => 'POST', 'callback' => [__CLASS__, 'update_settings'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback'], 'args' => self::get_settings_schema()]
            ]);

            register_rest_route($namespace, '/' . $route_base . '/environment', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_environment'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/diagnostics', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_diagnostics'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/bulk/start', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'start_bulk'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/bulk/status', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_bulk_status'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/bulk/stop', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'stop_bulk'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/bulk-original-cleanup', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'start_bulk_original_cleanup'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/performance-analysis', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_performance_analysis'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/run-performance-analysis', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'run_performance_analysis'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/performance-settings', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'update_performance_settings'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/apply-recommended-settings', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'apply_recommended_settings'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/convert', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'test_convert'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback'], 'args' => ['attachment_id' => ['required' => true, 'type' => 'integer']]
            ]);

            register_rest_route($namespace, '/' . $route_base . '/stats', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_stats'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/media-indexer/scan', [
                'methods' => 'POST', 'callback' => [__CLASS__, 'scan_media_uploads'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);

            register_rest_route($namespace, '/' . $route_base . '/media-indexer/stats', [
                'methods' => 'GET', 'callback' => [__CLASS__, 'get_media_indexer_stats'], 'permission_callback' => [Authority::class, 'rest_api_permission_callback']
            ]);
        }
    }

    /**
     * Get Image Converter settings
     */
    public static function get_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        $config = SysConfig::$config;
        
        $settings = [
            'image_converter_enabled' => $config['image_converter_enabled'] ?? false,
            'image_converter_formats' => $config['image_converter_formats'] ?? ['avif', 'webp'],
            'image_converter_quality' => $config['image_converter_quality'] ?? 80,
            'image_converter_avif_quality' => $config['image_converter_avif_quality'] ?? 75,
            'image_converter_max_width' => $config['image_converter_max_width'] ?? 1920,
            'image_converter_max_height' => $config['image_converter_max_height'] ?? 1080,
            'image_converter_keep_originals' => $config['image_converter_keep_originals'] ?? true,
            'image_converter_allowed_types' => $config['image_converter_allowed_types'] ?? ['image/jpeg', 'image/png', 'image/gif'],
            'image_converter_auto_alt' => $config['image_converter_auto_alt'] ?? false,
            'image_converter_frontend_serve' => $config['image_converter_frontend_serve'] ?? true,
            'image_converter_htaccess_rules' => $config['image_converter_htaccess_rules'] ?? true
        ];

        return new \WP_REST_Response(['success' => true, 'data' => $settings], 200);
    }

    /**
     * Update Image Converter settings
     */
    public static function update_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        $params = $request->get_params();
        $config = SysConfig::$config;
        
        $config_updates = [
            'image_converter_enabled' => $params['image_converter_enabled'] ?? false,
            'image_converter_formats' => $params['image_converter_formats'] ?? ['avif', 'webp'],
            'image_converter_quality' => intval($params['image_converter_quality'] ?? 80),
            'image_converter_avif_quality' => intval($params['image_converter_avif_quality'] ?? 75),
            'image_converter_max_width' => intval($params['image_converter_max_width'] ?? 1920),
            'image_converter_max_height' => intval($params['image_converter_max_height'] ?? 1080),
            'image_converter_keep_originals' => $params['image_converter_keep_originals'] ?? true,
            'image_converter_allowed_types' => $params['image_converter_allowed_types'] ?? ['image/jpeg', 'image/png', 'image/gif'],
            'image_converter_auto_alt' => $params['image_converter_auto_alt'] ?? false,
            'image_converter_frontend_serve' => $params['image_converter_frontend_serve'] ?? true,
            'image_converter_htaccess_rules' => $params['image_converter_htaccess_rules'] ?? true
        ];

        foreach ($config_updates as $key => $value) {
            $config[$key] = $value;
        }

        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        // Save configuration using WordPress option
        $result = update_option('ZIZI_CACHE_CONFIG', $config);
        
        // WordPress může vrátit false, pokud se hodnota nezměnila - to není chyba
        // Aktualizujeme vždy statickou config proměnnou
        SysConfig::$config = $config; // Update static config
        
        // Voláme hook pouze pokud došlo k relevantním změnám týkajícím se .htaccess
        $htaccess_relevant_change = isset($params['image_converter_enabled']) || 
                                   isset($params['image_converter_htaccess_rules']) ||
                                   isset($params['image_converter_formats']);
        
        // Pouze pokud došlo ke změně, která by ovlivnila .htaccess
        if ($htaccess_relevant_change) {
            do_action('zizi_cache_update_config:after');
        }
        
        // Vraťme vždy úspěch, i když se hodnota nezměnila
        return new \WP_REST_Response(['success' => true, 'message' => 'Settings updated', 'updated' => $result], 200);
    }

    /**
     * Get environment information
     */
    public static function get_environment(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        $gd_available = extension_loaded('gd');
        $imagick_available = extension_loaded('imagick');
        
        $env = [
            'php_version' => PHP_VERSION,
            'gd_available' => $gd_available,
            'imagick_available' => $imagick_available,
            'gd_webp' => $gd_available && function_exists('imagewebp'),
            'gd_avif' => $gd_available && function_exists('imageavif'),
            'imagick_webp' => false,
            'imagick_avif' => false,
            'action_scheduler' => function_exists('as_schedule_single_action'),
            'upload_dir' => wp_upload_dir(),
            'max_upload_size' => wp_max_upload_size(),
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time')
        ];

        if ($imagick_available) {
            try {
                $imagick = new \Imagick();
                $formats = $imagick->queryFormats();
                $env['imagick_webp'] = in_array('WEBP', $formats);
                $env['imagick_avif'] = in_array('AVIF', $formats);
            } catch (\Exception $e) {
                // Imagick error
            }
        }

        return new \WP_REST_Response(['success' => true, 'data' => $env], 200);
    }

    /**
     * Get comprehensive diagnostics for troubleshooting
     */
    public static function get_diagnostics(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        $diagnostics = [
            'environment' => ImageConverter::get_environment_info(),
            'settings' => SysConfig::$config,
            'temp_dirs' => [
                'sys_temp_dir' => sys_get_temp_dir(),
                'sys_temp_writable' => is_writable(sys_get_temp_dir()),
                'wp_temp_dir' => get_temp_dir(),
                'wp_temp_writable' => is_writable(get_temp_dir()),
            ],
            'upload_info' => wp_upload_dir(),
            'server_info' => [
                'php_version' => PHP_VERSION,
                'os' => PHP_OS,
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true),
            ],
            'action_scheduler' => [
                'available' => function_exists('as_schedule_single_action'),
                'pending_actions' => function_exists('as_get_scheduled_actions') ? count(as_get_scheduled_actions([
                    'hook' => 'zizi_image_converter_process_single',
                    'status' => 'pending'
                ])) : 0,
            ],
            'recent_errors' => self::get_recent_error_logs(),
            'sample_test' => self::perform_sample_conversion_test(),
            'file_permissions' => self::check_file_permissions(),
        ];

        return new \WP_REST_Response(['success' => true, 'data' => $diagnostics], 200);
    }

    /**
     * Get recent error logs related to image conversion
     */
    private static function get_recent_error_logs(): array
    {
        $errors = [];
        
        // Check if CacheSys log exists and read recent errors
        if (class_exists('\\ZiziCache\\CacheSys')) {
            try {
                // This is simplified - in real implementation you'd read from log files
                $errors = [
                    'note' => 'Check WordPress debug.log and ZiziCache logs for detailed error information',
                    'debug_status' => WP_DEBUG ? 'enabled' : 'disabled',
                    'log_status' => WP_DEBUG_LOG ? 'enabled' : 'disabled'
                ];
            } catch (\Exception $e) {
                $errors = ['error' => 'Could not access log files'];
            }
        }
        
        return $errors;
    }

    /**
     * Perform a quick conversion test
     */
    private static function perform_sample_conversion_test(): array
    {
        try {
            // Find a suitable test image
            $query = new \WP_Query([
                'post_type' => 'attachment',
                'post_mime_type' => ['image/jpeg', 'image/png'],
                'post_status' => 'inherit',
                'posts_per_page' => 1,
                'orderby' => 'date',
                'order' => 'DESC'
            ]);

            if (empty($query->posts)) {
                return ['status' => 'no_test_images', 'message' => 'No suitable test images found'];
            }

            $attachment_id = $query->posts[0]->ID;
            $file_path = get_attached_file($attachment_id);
            
            if (!$file_path || !file_exists($file_path)) {
                return ['status' => 'file_not_found', 'message' => 'Test image file not accessible'];
            }

            $test_result = [
                'status' => 'tested',
                'test_image' => [
                    'id' => $attachment_id,
                    'path' => $file_path,
                    'size' => filesize($file_path),
                    'mime_type' => get_post_mime_type($attachment_id),
                    'readable' => is_readable($file_path),
                ],
                'environment_checks' => [
                    'gd_available' => extension_loaded('gd'),
                    'gd_avif' => function_exists('imageavif'),
                    'imagick_available' => extension_loaded('imagick'),
                    'imagick_avif' => ImageConverter::imagick_supports_avif(),
                ],
                'temp_file_test' => self::test_temp_file_creation($file_path),
            ];

            return $test_result;

        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
                'trace' => WP_DEBUG ? $e->getTraceAsString() : 'Enable WP_DEBUG for trace'
            ];
        }
    }

    /**
     * Test temporary file creation and access
     */
    private static function test_temp_file_creation(string $source_file): array
    {
        try {
            // Test wp_tempnam() with proper namespace
            $temp_file = \wp_tempnam();
            $temp_result = [
                'wp_tempnam' => [
                    'created' => !empty($temp_file),
                    'path' => $temp_file,
                    'writable' => $temp_file ? is_writable($temp_file) : false,
                    'readable' => $temp_file ? is_readable($temp_file) : false,
                ]
            ];

            // Test with extension
            $original_extension = pathinfo($source_file, PATHINFO_EXTENSION);
            $temp_with_ext = $temp_file . '.' . $original_extension;
            
            // Copy source to temp with extension
            if ($temp_file && copy($source_file, $temp_with_ext)) {
                $temp_result['temp_with_extension'] = [
                    'created' => true,
                    'path' => $temp_with_ext,
                    'size' => filesize($temp_with_ext),
                    'readable' => is_readable($temp_with_ext),
                    'getimagesize_works' => @getimagesize($temp_with_ext) !== false,
                ];
                
                // Test direct AVIF conversion
                if (function_exists('imageavif')) {
                    $avif_test = self::test_avif_conversion($temp_with_ext);
                    $temp_result['avif_test'] = $avif_test;
                }
                
                // Test úložišť souborů
                $uploads_dir = wp_upload_dir();
                $temp_result['storage_test'] = [
                    'uploads_dir_exists' => is_dir($uploads_dir['path']),
                    'uploads_dir_writable' => is_writable($uploads_dir['path']),
                    'uploads_basedir_writable' => is_writable($uploads_dir['basedir']),
                    'plugin_dir' => plugin_dir_path(ZIZI_CACHE_FILE),
                    'plugin_dir_writable' => is_writable(plugin_dir_path(ZIZI_CACHE_FILE)),
                    'attachment_dir' => dirname($source_file),
                    'attachment_dir_writable' => is_writable(dirname($source_file)),
                ];
                
                // Testování přesunování souborů
                $test_copy_path = $uploads_dir['path'] . '/' . uniqid('test-copy-') . '.jpg';
                $copy_success = @copy($source_file, $test_copy_path);
                $temp_result['file_operations'] = [
                    'copy_success' => $copy_success,
                    'target_path' => $test_copy_path,
                    'target_exists' => $copy_success && file_exists($test_copy_path),
                    'target_readable' => $copy_success && is_readable($test_copy_path),
                ];
                
                // Cleanup
                if ($copy_success && file_exists($test_copy_path)) {
                    @unlink($test_copy_path);
                }
                
                // Cleanup temp file
                @unlink($temp_with_ext);
            } else {
                $temp_result['temp_with_extension'] = [
                    'created' => false,
                    'error' => 'Could not copy source to temp file with extension'
                ];
            }

            // Cleanup original temp file
            if ($temp_file) {
                @unlink($temp_file);
            }

            return $temp_result;

        } catch (\Exception $e) {
            return [
                'error' => $e->getMessage(),
                'status' => 'failed'
            ];
        }
    }
    
    /**
     * Test AVIF conversion
     */
    private static function test_avif_conversion(string $source_file): array
    {
        try {
            // Kontrola, zda AVIF konverze je dostupná
            if (!function_exists('imageavif')) {
                return [
                    'status' => 'unavailable',
                    'message' => 'imageavif function is not available'
                ];
            }
            
            $image_info = @getimagesize($source_file);
            if (!$image_info) {
                return [
                    'status' => 'error',
                    'message' => 'Cannot get image size from source file'
                ];
            }
            
            $mime_type = $image_info['mime'];
            $source_image = null;
            
            switch ($mime_type) {
                case 'image/jpeg':
                    $source_image = @imagecreatefromjpeg($source_file);
                    break;
                case 'image/png':
                    $source_image = @imagecreatefrompng($source_file);
                    break;
                case 'image/gif':
                    $source_image = @imagecreatefromgif($source_file);
                    break;
                default:
                    return [
                        'status' => 'unsupported',
                        'message' => 'Unsupported MIME type: ' . $mime_type
                    ];
            }
            
            if (!$source_image) {
                return [
                    'status' => 'error',
                    'message' => 'Failed to create image resource from source'
                ];
            }
            
            // Testovací AVIF soubor
            $avif_output = \wp_tempnam() . '.avif';
            $quality = 75;
            $result = @imageavif($source_image, $avif_output, $quality);
            
            $test_result = [
                'conversion_result' => $result,
                'output_file' => $avif_output,
                'output_exists' => file_exists($avif_output),
                'output_size' => $result && file_exists($avif_output) ? filesize($avif_output) : 0
            ];
            
            // Vyčištění
            imagedestroy($source_image);
            if (file_exists($avif_output)) {
                @unlink($avif_output);
            }
            
            return $test_result;
            
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Check file permissions in relevant directories
     */
    private static function check_file_permissions(): array
    {
        $upload_dir = wp_upload_dir();
        
        return [
            'upload_dir' => [
                'path' => $upload_dir['basedir'],
                'writable' => is_writable($upload_dir['basedir']),
                'readable' => is_readable($upload_dir['basedir']),
            ],
            'temp_dir' => [
                'path' => get_temp_dir(),
                'writable' => is_writable(get_temp_dir()),
                'readable' => is_readable(get_temp_dir()),
            ],
            'sys_temp_dir' => [
                'path' => sys_get_temp_dir(),
                'writable' => is_writable(sys_get_temp_dir()),
                'readable' => is_readable(sys_get_temp_dir()),
            ],
        ];
    }

    /**
     * Start bulk optimization
     */
    public static function start_bulk(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        $config = SysConfig::$config;
        if (!($config['image_converter_enabled'] ?? false)) {
            return new \WP_REST_Response(['success' => false, 'message' => 'Image Converter is disabled'], 400);
        }

        if (!class_exists('\\ZiziCache\\ImageConverterBulk')) {
            return new \WP_REST_Response(['success' => false, 'message' => 'Bulk optimizer not available'], 500);
        }
        
        // Kontrola dostupnosti Action Scheduleru před jeho použitím
        if (!function_exists('as_schedule_single_action')) {
            // Log error do jednotného logu
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    'Bulk optimization failed: Action Scheduler not available',
                    'ImageConverterAdmin'
                );
            }
            return new \WP_REST_Response(['success' => false, 'message' => 'Action Scheduler not available. Please install WooCommerce or Action Scheduler plugin.'], 500);
        }

        try {
            // Get request parameters including enabled_formats from frontend
            $params = $request->get_json_params();
            $options = [];
            
            // Pass enabled_formats if provided by frontend
            if (isset($params['enabled_formats']) && is_array($params['enabled_formats'])) {
                $options['enabled_formats'] = $params['enabled_formats'];
            }
            
            $result = \ZiziCache\ImageConverterBulk::start_bulk_optimization($options);
            return new \WP_REST_Response(['success' => true, 'data' => $result], 200);
        } catch (\Exception $e) {
            // Log error do jednotného logu
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    'Bulk optimization error: ' . $e->getMessage(),
                    'ImageConverterAdmin'
                );
            }
            return new \WP_REST_Response(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Get bulk optimization status (with error handling for large datasets)
     */
    public static function get_bulk_status(\WP_REST_Request $request): \WP_REST_Response
    {
        // Enhanced error handling and output buffer management
        $start_time = microtime(true);
        
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        // Set memory and time limits for large datasets
        $original_memory_limit = ini_get('memory_limit');
        $original_time_limit = ini_get('max_execution_time');
        
        // Temporarily increase limits for bulk operations
        if (function_exists('ini_set')) {
            ini_set('memory_limit', '512M');
            ini_set('max_execution_time', 60);
        }
        
        try {
            if (!class_exists('\\ZiziCache\\ImageConverterBulk')) {
                return new \WP_REST_Response([
                    'success' => false, 
                    'message' => 'Bulk optimizer not available'
                ], 500);
            }

            // Use optimized bulk status method with comprehensive error handling
            $status = \ZiziCache\ImageConverterBulk::get_bulk_status();
            
            // Add endpoint performance metrics
            $execution_time = microtime(true) - $start_time;
            $status['endpoint_performance'] = [
                'total_execution_time_ms' => round($execution_time * 1000, 2),
                'memory_peak_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2)
            ];
            
            // Check if this looks like a large dataset and log performance
            if (($status['total_images'] ?? 0) > 10000) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Large dataset bulk status: {$status['total_images']} images, " .
                        "{$execution_time}s execution time, " .
                        round(memory_get_peak_usage(true) / 1024 / 1024, 2) . "MB peak memory",
                        'ImageConverterAdmin'
                    );
                }
            }
            
            return new \WP_REST_Response(['success' => true, 'data' => $status], 200);
            
        } catch (\Error $e) {
            // Catch fatal PHP errors (like memory exhaustion)
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    'Fatal error in bulk status endpoint: ' . $e->getMessage() . 
                    ' | Memory peak: ' . round(memory_get_peak_usage(true) / 1024 / 1024, 2) . 'MB',
                    'ImageConverterAdmin'
                );
            }
            
            return new \WP_REST_Response([
                'success' => false, 
                'message' => 'Server error: ' . $e->getMessage(),
                'error_type' => 'fatal_error',
                'memory_usage_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2)
            ], 500);
            
        } catch (\Exception $e) {
            // Catch regular exceptions
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    'Exception in bulk status endpoint: ' . $e->getMessage(),
                    'ImageConverterAdmin'
                );
            }
            
            return new \WP_REST_Response([
                'success' => false, 
                'message' => 'Error retrieving bulk status: ' . $e->getMessage(),
                'error_type' => 'exception'
            ], 500);
            
        } finally {
            // Restore original limits
            if (function_exists('ini_set')) {
                ini_set('memory_limit', $original_memory_limit);
                ini_set('max_execution_time', $original_time_limit);
            }
        }
    }

    /**
     * Stop bulk optimization
     */
    public static function stop_bulk(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        if (!class_exists('\\ZiziCache\\ImageConverterBulk')) {
            return new \WP_REST_Response(['success' => false, 'message' => 'Bulk optimizer not available'], 500);
        }

        try {
            $result = \ZiziCache\ImageConverterBulk::stop_bulk_optimization();
            return new \WP_REST_Response(['success' => true, 'data' => $result], 200);
        } catch (\Exception $e) {
            return new \WP_REST_Response(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Test convert single image
     */
    public static function test_convert(\WP_REST_Request $request): \WP_REST_Response
    {
        // Prevent any accidental output that could corrupt JSON response
        if (ob_get_length()) {
            ob_clean();
        }
        
        $attachment_id = intval($request->get_param('attachment_id'));
        
        if (!wp_attachment_is_image($attachment_id)) {
            return new \WP_REST_Response(['success' => false, 'message' => 'Invalid image attachment'], 400);
        }

        try {
            $result = ImageConverter::convert_image($attachment_id);
            return new \WP_REST_Response(['success' => true, 'data' => $result], 200);
        } catch (\Exception $e) {
            return new \WP_REST_Response(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Get settings validation schema
     */
    private static function get_settings_schema(): array
    {
        return [
            'enabled' => ['type' => 'boolean', 'default' => false],
            'formats' => ['type' => 'array', 'items' => ['type' => 'string'], 'default' => ['avif', 'webp']],
            'quality' => ['type' => 'integer', 'minimum' => 1, 'maximum' => 100, 'default' => 80],
            'avif_quality' => ['type' => 'integer', 'minimum' => 1, 'maximum' => 100, 'default' => 75],
            'max_width' => ['type' => 'integer', 'minimum' => 100, 'maximum' => 5000, 'default' => 1920],
            'max_height' => ['type' => 'integer', 'minimum' => 100, 'maximum' => 5000, 'default' => 1080],
            'keep_originals' => ['type' => 'boolean', 'default' => true],
            'allowed_types' => ['type' => 'array', 'items' => ['type' => 'string'], 'default' => ['image/jpeg', 'image/png', 'image/gif']],
            'auto_alt' => ['type' => 'boolean', 'default' => false],
            'frontend_serve' => ['type' => 'boolean', 'default' => true],
            'htaccess_rules' => ['type' => 'boolean', 'default' => true]
        ];
    }

    /**
     * Get statistics for admin dashboard
     */
    public static function get_statistics(): array
    {
        $query = new \WP_Query([
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'meta_query' => [['key' => '_zizi_image_optimized', 'compare' => 'EXISTS']]
        ]);

        $total_original = 0;
        $total_optimized = 0;
        $formats_count = ['webp' => 0, 'avif' => 0];

        foreach ($query->posts as $post) {
            $metadata = get_post_meta($post->ID, '_zizi_image_optimized', true);
            if (is_array($metadata)) {
                $total_original += $metadata['sizes']['original'] ?? 0;
                $total_optimized += $metadata['sizes']['webp'] ?? 0;
                $total_optimized += $metadata['sizes']['avif'] ?? 0;
                
                // Fixed: Count formats correctly from metadata
                if (isset($metadata['formats']) && is_array($metadata['formats'])) {
                    foreach ($metadata['formats'] as $format) {
                        if (isset($formats_count[$format])) {
                            $formats_count[$format]++;
                        }
                    }
                } else {
                    // Fallback: check if specific format files exist
                    foreach (['webp', 'avif'] as $format) {
                        if (isset($metadata['sizes'][$format]) && $metadata['sizes'][$format] > 0) {
                            $formats_count[$format]++;
                        }
                    }
                }
            }
        }

        $savings_percent = $total_original > 0 ? round((($total_original - $total_optimized) / $total_original) * 100, 2) : 0;

        return [
            'optimized_count' => $query->found_posts,
            'total_original_size' => $total_original,
            'total_optimized_size' => $total_optimized,
            'savings_bytes' => $total_original - $total_optimized,
            'savings_percent' => $savings_percent,
            'formats_count' => $formats_count
        ];
    }

    /**
     * Reset all optimization data
     */
    public static function reset_optimization_data(): bool
    {
        $query = new \WP_Query([
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'meta_query' => [['key' => '_zizi_image_optimized', 'compare' => 'EXISTS']]
        ]);

        foreach ($query->posts as $post) {
            delete_post_meta($post->ID, '_zizi_image_optimized');
            
            $file_path = get_attached_file($post->ID);
            if ($file_path) {
                $webp_path = $file_path . '.webp';
                $avif_path = $file_path . '.avif';
                
                if (file_exists($webp_path)) unlink($webp_path);
                if (file_exists($avif_path)) unlink($avif_path);
            }
        }

        return true;
    }

    /**
     * Get recent activity log
     */
    public static function get_activity_log(int $limit = 20): array
    {
        $query = new \WP_Query([
            'post_type' => 'attachment',
            'post_status' => 'inherit', 
            'posts_per_page' => $limit,
            'orderby' => 'modified',
            'order' => 'DESC',
            'meta_query' => [['key' => '_zizi_image_optimized', 'compare' => 'EXISTS']]
        ]);

        $activities = [];
        foreach ($query->posts as $post) {
            $metadata = get_post_meta($post->ID, '_zizi_image_optimized', true);
            $activities[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'date' => $post->post_modified,
                'formats' => $metadata['formats'] ?? [],
                'savings' => $metadata['savings'] ?? []
            ];
        }

        return $activities;
    }

    /**
     * Get Image Converter statistics
     */
    public static function get_stats(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            // Prevent any accidental output that could corrupt JSON response
            if (ob_get_length()) {
                ob_clean();
            }

            if (!class_exists('\\ZiziCache\\ImageConverterStats')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'ImageConverterStats class not found'
                ], 500);
            }

            // Check if refresh is requested
            $refresh = $request->get_param('refresh');
            if ($refresh === 'true' || $refresh === true) {
                // Clear cache to force fresh calculation
                \ZiziCache\ImageConverterStats::clearCache();
                delete_transient('zizi_image_converter_stats');
                delete_transient('zizi_image_converter_recent_activity');
            }

            $stats = \ZiziCache\ImageConverterStats::getStats();
            $recent_activity = \ZiziCache\ImageConverterStats::getRecentActivity(5);

            return new \WP_REST_Response([
                'success' => true,
                'data' => $stats,  // Return stats directly, not nested in overview
                'recent_activity' => $recent_activity,
                'generated_at' => current_time('c'),
                'refreshed' => $refresh === 'true' || $refresh === true
            ], 200);

        } catch (\Throwable $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get statistics: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Start bulk cleanup of original files
     */
    public static function start_bulk_original_cleanup(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('manage_options')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            $result = ImageConverter::start_bulk_original_cleanup();

            return new \WP_REST_Response([
                'success' => $result['success'],
                'message' => $result['message'],
                'data' => $result
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to start bulk original cleanup: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get performance analysis data
     */
    public static function get_performance_analysis(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('manage_options')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            // Import PerformanceProfiler if available
            if (!class_exists('\ZiziCache\PerformanceProfiler')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Performance profiler not available'
                ], 404);
            }

            $profiler = new \ZiziCache\PerformanceProfiler();
            $analysis = $profiler->get_comprehensive_analysis();

            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get performance analysis: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Run new performance analysis
     */
    public static function run_performance_analysis(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('manage_options')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            // Import PerformanceProfiler if available
            if (!class_exists('\ZiziCache\PerformanceProfiler')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Performance profiler not available'
                ], 404);
            }

            $profiler = new \ZiziCache\PerformanceProfiler();
            $analysis = $profiler->run_fresh_analysis();

            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis,
                'message' => 'Performance analysis completed successfully'
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to run performance analysis: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update performance settings
     */
    public static function update_performance_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('manage_options')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            $params = $request->get_json_params();
            if (!$params) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'No performance settings provided'
                ], 400);
            }

            // Validate and save performance settings
            $allowed_settings = [
                'bulk_performance_mode',
                'bulk_max_batch_size',
                'bulk_batch_delay', 
                'bulk_memory_threshold',
                'bulk_concurrent_limit',
                'enable_performance_monitoring'
            ];

            foreach ($allowed_settings as $setting) {
                if (isset($params[$setting])) {
                    SysConfig::set($setting, $params[$setting]);
                }
            }

            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Performance settings updated successfully'
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to update performance settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Apply recommended performance settings
     */
    public static function apply_recommended_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('manage_options')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            // Import PerformanceProfiler if available
            if (!class_exists('\ZiziCache\PerformanceProfiler')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Performance profiler not available'
                ], 404);
            }

            $profiler = new \ZiziCache\PerformanceProfiler();
            $profile = $profiler->get_performance_profile();
            
            // Apply recommended settings
            SysConfig::set('bulk_performance_mode', 'optimized');
            SysConfig::set('bulk_max_batch_size', $profile['batch_size']);
            SysConfig::set('bulk_batch_delay', $profile['batch_delay']);
            SysConfig::set('bulk_memory_threshold', $profile['memory_threshold']);
            SysConfig::set('bulk_concurrent_limit', $profile['concurrent_limit']);

            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Recommended settings applied successfully',
                'data' => [
                    'settings' => [
                        'bulk_performance_mode' => 'optimized',
                        'bulk_max_batch_size' => $profile['batch_size'],
                        'bulk_batch_delay' => $profile['batch_delay'],
                        'bulk_memory_threshold' => $profile['memory_threshold'],
                        'bulk_concurrent_limit' => $profile['concurrent_limit']
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to apply recommended settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Scan uploads directory and index unregistered media files
     */
    public static function scan_media_uploads(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('upload_files')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            if (!class_exists('\ZiziCache\MediaLibraryIndexer')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Media Library Indexer not available'
                ], 404);
            }

            $result = \ZiziCache\MediaLibraryIndexer::scan_and_index_uploads();

            return new \WP_REST_Response([
                'success' => true,
                'message' => "Indexing completed: {$result['indexed_files']} files indexed, {$result['skipped_files']} skipped",
                'data' => $result
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Media indexing failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics about unindexed media files
     */
    public static function get_media_indexer_stats(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!current_user_can('upload_files')) {
            return new \WP_REST_Response(['error' => 'Insufficient permissions'], 403);
        }

        try {
            if (!class_exists('\ZiziCache\MediaLibraryIndexer')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Media Library Indexer not available'
                ], 404);
            }

            $stats = \ZiziCache\MediaLibraryIndexer::get_unindexed_stats();

            if (isset($stats['error'])) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => $stats['error']
                ], 500);
            }

            return new \WP_REST_Response([
                'success' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get media indexer stats: ' . $e->getMessage()
            ], 500);
        }
    }
}
