<?php

namespace ZiziCache\RestApi;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Class FontStatistics
 *
 * REST API endpoints for font statistics.
 *
 * @package ZiziCache\RestApi
 */
class FontStatistics
{
    /**
     * Register Font Statistics REST API endpoints.
     *
     * @param array $namespaces Array of REST API namespaces to register under
     * @return void
     */
    public static function register(array $namespaces): void
    {
        foreach ($namespaces as $namespace) {
            // Font statistics endpoint
            register_rest_route($namespace, '/font-statistics', [
                'methods' => 'GET',
                'callback' => [__CLASS__, 'get_statistics'],
                'permission_callback' => [__CLASS__, 'check_permissions'],
            ]);

            // Font data collection endpoint (for frontend JavaScript)
            register_rest_route($namespace, '/font-intelligence/collect', [
                'methods' => 'POST',
                'callback' => [__CLASS__, 'collect_font_data'],
                'permission_callback' => '__return_true', // Allow public access for frontend collection
                'args' => [
                    'pageUrl' => [
                        'required' => true,
                        'type' => 'string',
                    ],
                    'loadedFonts' => [
                        'required' => false,
                        'type' => 'array',
                    ],
                    'aboveFoldFonts' => [
                        'required' => false,
                        'type' => 'array',
                    ],
                ],
            ]);

            // Generate optimized Google Fonts CSS endpoint
            register_rest_route($namespace, '/font-statistics/generate-optimized-fonts', [
                'methods' => 'POST',
                'callback' => [__CLASS__, 'generate_optimized_fonts'],
                'permission_callback' => [__CLASS__, 'check_permissions'],
            ]);

            // Reset font statistics endpoint
            register_rest_route($namespace, '/reset-statistics', [
                'methods' => 'POST',
                'callback' => [__CLASS__, 'reset_statistics'],
                'permission_callback' => [__CLASS__, 'check_permissions'],
            ]);
        }
    }

    /**
     * Check if user has permission to access font statistics endpoints.
     *
     * @param WP_REST_Request $request The REST request
     * @return bool|WP_Error True if the user has permission, WP_Error otherwise
     */
    public static function check_permissions(WP_REST_Request $request): bool|WP_Error
    {
        // Basic permission check for all requests
        if (!current_user_can('manage_options')) {
            return new WP_Error(
                'rest_forbidden',
                __('Sorry, you are not allowed to access font statistics.', 'zizi-cache'),
                ['status' => rest_authorization_required_code()]
            );
        }
        
        // For state-changing operations (POST, PUT, DELETE), also verify nonce
        $request_method = $request->get_method();
        if (in_array($request_method, ['POST', 'PUT', 'DELETE'])) {
            // Get nonce from X-WP-Nonce header
            $nonce = $request->get_header('X-WP-Nonce');
            
            if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
                return new WP_Error(
                    'rest_forbidden',
                    __('Security verification failed.', 'zizi-cache'),
                    ['status' => 403]
                );
            }
        }
        
        return true;
    }

    /**
     * Get font statistics across the site.
     *
     * @param WP_REST_Request $request The REST request
     * @return WP_REST_Response|WP_Error Response with font statistics
     */
    public static function get_statistics(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            // Get font statistics
            $font_stats = self::get_font_statistics();

            return new WP_REST_Response([
                'success' => true,
                'data' => $font_stats,
                'timestamp' => current_time('timestamp'),
            ], 200);

        } catch (\Exception $e) {
            return new WP_Error(
                'font_statistics_error',
                sprintf(__('Failed to get font statistics: %s', 'zizi-cache'), $e->getMessage()),
                ['status' => 500]
            );
        }
    }

    /**
     * Get statistics about fonts across the site.
     *
     * @return array Font statistics
     */
    private static function get_font_statistics(): array
    {
        global $wpdb;
        
        $options = $wpdb->get_results("
            SELECT option_name, option_value
            FROM {$wpdb->options} 
            WHERE option_name LIKE 'zizi_cache_font_data_%'
        ");
        
        if (empty($options)) {
            return [];
        }
        
        // Agregace dat
        $font_stats = [];
        
        foreach ($options as $option) {
            $data = maybe_unserialize($option->option_value);
            
            // Skip expired or invalid data
            if (!$data || empty($data['fonts']) || (isset($data['updated']) && (time() - $data['updated']) > 604800)) { // 7 days
                continue;
            }
            
            foreach ($data['fonts'] as $font) {
                if (empty($font['family'])) continue;
                
                $family = $font['family'];
                
                if (!isset($font_stats[$family])) {
                    $font_stats[$family] = [
                        'family' => $family,
                        'pages_count' => 1,
                        'weights' => isset($font['weight']) ? [$font['weight']] : [],
                        'isHeading' => isset($font['isHeading']) ? $font['isHeading'] : false,
                        'isBody' => isset($font['isHeading']) ? !$font['isHeading'] : true
                    ];
                } else {
                    $font_stats[$family]['pages_count']++;
                    if (isset($font['weight']) && !in_array($font['weight'], $font_stats[$family]['weights'])) {
                        $font_stats[$family]['weights'][] = $font['weight'];
                    }
                    if (isset($font['isHeading'])) {
                        $font_stats[$family]['isHeading'] = $font_stats[$family]['isHeading'] || $font['isHeading'];
                        $font_stats[$family]['isBody'] = $font_stats[$family]['isBody'] || !$font['isHeading'];
                    }
                }
            }
        }
        
        // Seřazení podle počtu stránek
        usort($font_stats, function($a, $b) {
            return $b['pages_count'] - $a['pages_count'];
        });
        
        // Convert to indexed array and limit to top 10
        return array_values(array_slice($font_stats, 0, 10));
    }

    /**
     * Collect font data from frontend JavaScript.
     *
     * @param WP_REST_Request $request The REST request
     * @return WP_REST_Response|WP_Error Response
     */
    public static function collect_font_data(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            $page_url = $request->get_param('pageUrl') ?: '';
            $loaded_fonts = $request->get_param('loadedFonts') ?: [];
            $above_fold_fonts = $request->get_param('aboveFoldFonts') ?: [];

            if (empty($page_url)) {
                return new WP_Error('missing_url', 'Page URL is required', ['status' => 400]);
            }

            // Normalize font data
            $font_data = [];
            
            // Process above-fold fonts first (they have priority)
            if (!empty($above_fold_fonts)) {
                foreach ($above_fold_fonts as $font) {
                    if (empty($font['family'])) continue;
                    
                    $font_data[] = [
                        'family' => sanitize_text_field($font['family']),
                        'weight' => isset($font['weight']) ? sanitize_text_field($font['weight']) : '400',
                        'style' => isset($font['style']) ? sanitize_text_field($font['style']) : 'normal',
                        'isHeading' => !empty($font['isHeading']),
                        'isAboveFold' => true,
                    ];
                }
            }

            // Process loaded fonts (add only if not already included from above-fold)
            if (!empty($loaded_fonts)) {
                foreach ($loaded_fonts as $font) {
                    if (empty($font['family'])) continue;
                    
                    // Check if this font is not already in font_data
                    $already_added = false;
                    foreach ($font_data as $existing_font) {
                        if ($existing_font['family'] === $font['family'] && 
                            $existing_font['weight'] === ($font['weight'] ?? '400')) {
                            $already_added = true;
                            break;
                        }
                    }
                    
                    if (!$already_added) {
                        $font_data[] = [
                            'family' => sanitize_text_field($font['family']),
                            'weight' => isset($font['weight']) ? sanitize_text_field($font['weight']) : '400',
                            'style' => isset($font['style']) ? sanitize_text_field($font['style']) : 'normal',
                            'isHeading' => false,
                            'isAboveFold' => false,
                        ];
                    }
                }
            }

            if (empty($font_data)) {
                return new WP_REST_Response(['success' => true, 'message' => 'No fonts to save'], 200);
            }

            // Store font data with URL hash
            $cache_key = 'zizi_cache_font_data_' . md5($page_url);
            
            $data_to_store = [
                'url' => $page_url,
                'fonts' => $font_data,
                'updated' => time(),
                'user_agent' => $request->get_header('user-agent') ?: '',
            ];

            update_option($cache_key, $data_to_store);

            // Debug logging if enabled
            $config = get_option('ZIZI_CACHE_CONFIG', []);
            if (!empty($config['fonts_debug'])) {
                $font_names = array_map(function($f) { return $f['family']; }, $font_data);
                \ZiziCache\CacheSys::writeLog(
                    sprintf('Font data collected for %s: %s', $page_url, implode(', ', $font_names)),
                    'INFO',
                    'FontStatistics'
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => 'Font data collected successfully',
                'fonts_count' => count($font_data),
            ], 200);

        } catch (\Exception $e) {
            return new WP_Error('collection_error', 'Failed to collect font data: ' . $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * Generate optimized Google Fonts CSS based on statistics.
     *
     * @param WP_REST_Request $request The REST request
     * @return WP_REST_Response|WP_Error Response with generated CSS URL
     */
    public static function generate_optimized_fonts(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            $optimized_url = \ZiziCache\Font::generate_optimized_google_fonts_url();

            if (!$optimized_url) {
                return new WP_Error(
                    'no_fonts_available',
                    __('No font statistics available to generate optimized CSS. Please ensure visitors browse your site to collect font data.', 'zizi-cache'),
                    ['status' => 400]
                );
            }

            // Log generation if debug is enabled
            $config = get_option('ZIZI_CACHE_CONFIG', []);
            if (!empty($config['fonts_debug'])) {
                \ZiziCache\CacheSys::writeLog(
                    'Optimized Google Fonts CSS generated via admin request: ' . $optimized_url,
                    'INFO',
                    'FontStatistics'
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => 'Optimized Google Fonts CSS generated successfully',
                'generated_url' => $optimized_url,
                'instructions' => 'This URL will be automatically injected into your pages when font optimization is enabled.',
                'timestamp' => current_time('timestamp'),
            ], 200);

        } catch (\Exception $e) {
            return new WP_Error('generation_error', 'Failed to generate optimized fonts: ' . $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * Reset all font statistics data.
     *
     * @param WP_REST_Request $request The REST request
     * @return WP_REST_Response|WP_Error Response confirming reset
     */
    public static function reset_statistics(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            global $wpdb;
            
            // Delete all font statistics data
            $deleted_count = $wpdb->query("
                DELETE FROM {$wpdb->options} 
                WHERE option_name LIKE 'zizi_cache_font_data_%'
            ");

            // Log reset action if debug is enabled
            $config = get_option('ZIZI_CACHE_CONFIG', []);
            if (!empty($config['fonts_debug'])) {
                \ZiziCache\CacheSys::writeLog(
                    sprintf('Font statistics reset via admin request. Deleted %d records.', $deleted_count),
                    'INFO',
                    'FontStatistics'
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf('Font statistics reset successfully. Deleted %d records.', $deleted_count),
                'deleted_count' => $deleted_count,
                'timestamp' => current_time('timestamp'),
            ], 200);

        } catch (\Exception $e) {
            return new WP_Error(
                'reset_error', 
                sprintf(__('Failed to reset font statistics: %s', 'zizi-cache'), $e->getMessage()), 
                ['status' => 500]
            );
        }
    }
}
