<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\Database as DBModel;
use ZiziCache\DatabaseStatus;
use ZiziCache\DatabaseMonitor;
use ZiziCache\CacheSys;
use ZiziCache\RestApi\ValidationTrait;
use ZiziCache\RestApi\ResponseCleaner;

/**
 * Class Database
 *
 * Handles REST API endpoints for database optimization, statistics, logs, and cleanup.
 *
 * @package ZiziCache\RestApi
 */
class Database
{
    use ValidationTrait;

    /**
     * Endpoint definitions for database operations.
     *
     * @var array
     */    protected static $endpoints = [
        [
            'route' => '/optimize-db-tables',
            'methods' => 'POST',
            'callback' => 'optimize_database',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/optimize-db-tables-batch',
            'methods' => 'POST',
            'callback' => 'optimize_database_in_batches',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/database-statistics',
            'methods' => 'GET',
            'callback' => 'get_database_statistics',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/database-growth-history',
            'methods' => 'GET',
            'callback' => 'get_database_growth_history',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/db-event-log',
            'methods' => 'POST',
            'callback' => 'log_database_event',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/database-operation-logs',
            'methods' => 'GET',
            'callback' => 'get_database_operation_logs',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/database-status',
            'methods' => 'GET',
            'callback' => 'get_database_status',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/cleanup-counts',
            'methods' => ['GET','POST'],
            'callback' => 'get_cleanup_counts',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/index-recommendations',
            'methods' => 'GET',
            'callback' => 'get_index_recommendations',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/apply-index-recommendations',
            'methods' => 'POST',
            'callback' => 'apply_index_recommendations',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all database endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                // Wrap callback with ResponseCleaner for clean JSON output
                $callback = [__CLASS__, $def['callback']];
                if ($def['callback'] === 'get_database_status') {
                    $callback = ResponseCleaner::wrap_callback($callback);
                }
                
                register_rest_route($namespace, $def['route'] . '/?', [
                    'methods' => $def['methods'],
                    'callback' => $callback,
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Optimizes database tables and removes unnecessary data.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function optimize_database($request)
    {
        $params = $request->get_json_params();
        $options = isset($params['options']) && is_array($params['options']) ? $params['options'] : null;
        $before = DBModel::get_counts($options);
        DBModel::clean($options);
        $after = DBModel::get_counts($options);
        $deleted = [];
        foreach ($before as $opt => $count) {
            $deleted[$opt] = $count - ($after[$opt] ?? 0);
        }
        return rest_ensure_response($deleted);
    }

    /**
     * Optimizes database tables in batches for large databases.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function optimize_database_in_batches($request)
    {
        $params = $request->get_json_params();
        $batch_size = isset($params['batch_size']) ? intval($params['batch_size']) : 500;
        $tables = isset($params['tables']) && is_array($params['tables']) ? $params['tables'] : null;
        if (!defined('ZIZI_BATCH_START_TIME')) {
            define('ZIZI_BATCH_START_TIME', microtime(true));
        }
        $result = DBModel::optimize_tables_safely($tables);
        return rest_ensure_response($result);
    }

    /**
     * Returns detailed database statistics including table sizes and other info.
     *
     * @return \WP_REST_Response
     */
    public static function get_database_statistics()
    {
        global $wpdb;
        $db_info = [
            'name' => DB_NAME,
            'user' => DB_USER,
            'host' => DB_HOST,
            'charset' => $wpdb->charset,
            'collate' => $wpdb->collate,
            'prefix' => $wpdb->prefix,
            'version' => $wpdb->db_version(),
            'has_innodb' => $wpdb->get_var("SHOW VARIABLES LIKE 'have_innodb'") === 'YES',
        ];
        $tables = $wpdb->get_results("
            SELECT
                table_name,
                table_rows,
                data_length,
                index_length,
                engine,
                ROUND((data_length + index_length) / 1024 / 1024, 2) as size_mb,
                ROUND(data_free / 1024 / 1024, 2) as fragmentation_mb
            FROM information_schema.TABLES
            WHERE table_schema = '" . DB_NAME . "'
            ORDER BY (data_length + index_length) DESC
        ", ARRAY_A);
        $total_size = 0;
        $total_rows = 0;
        $fragmentation_size = 0;
        foreach ($tables as $table) {
            $total_size += (isset($table['data_length']) ? $table['data_length'] : 0)
                + (isset($table['index_length']) ? $table['index_length'] : 0);
            $total_rows += isset($table['table_rows']) ? $table['table_rows'] : 0;
            $fragmentation_size += isset($table['data_free']) ? $table['data_free'] : 0;
        }
        $growth_rate = class_exists('ZiziCache\\DatabaseMonitor') ? DatabaseMonitor::calculate_growth_rate(7) : 0;
        $stats = [
            'db_info' => $db_info,
            'summary' => [
                'total_tables' => count($tables),
                'total_size_mb' => round($total_size / 1024 / 1024, 2),
                'total_rows' => $total_rows,
                'fragmentation_mb' => round($fragmentation_size / 1024 / 1024, 2),
                'average_weekly_growth_percent' => $growth_rate,
                'collection_time' => current_time('mysql')
            ],
            'tables' => $tables,
            'table_engines' => array_count_values(array_column($tables, 'engine')),
        ];
        return rest_ensure_response($stats);
    }

    /**
     * Returns database growth history for visualization.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function get_database_growth_history($request)
    {
        $days = $request->get_param('days') ? intval($request->get_param('days')) : 30;
        $force_refresh = $request->get_param('t') ? true : false;
        if ($force_refresh) {
            global $wpdb;
            $current_size = $wpdb->get_var(
                "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) FROM information_schema.TABLES WHERE table_schema = '{$wpdb->dbname}'"
            );
            $history = get_option('zizi_cache_db_size_history', []);
            $latest = end($history);
            if (empty($history) || 
                (float)$latest['size'] !== (float)$current_size || 
                (time() - $latest['timestamp']) > 6 * HOUR_IN_SECONDS) {
                $history[] = [
                    'size' => (float)$current_size,
                    'timestamp' => time(),
                    'change' => !empty($latest) ? (float)$current_size - (float)$latest['size'] : 0
                ];
                if (count($history) > 60) {
                    $history = array_slice($history, -60);
                }
                update_option('zizi_cache_db_size_history', $history);
            }
        }
        $history = get_option('zizi_cache_db_size_history', []);
        $formatted_data = [];
        usort($history, function($a, $b) {
            return $a['timestamp'] - $b['timestamp'];
        });
        foreach ($history as $entry) {
            if ($days > 0 && (time() - $entry['timestamp']) > $days * DAY_IN_SECONDS) {
                continue;
            }
            $formatted_data[] = [
                'date' => date('Y-m-d', $entry['timestamp']),
                'size_mb' => round((float)$entry['size'], 2),
                'change_mb' => round((float)$entry['change'], 2)
            ];
        }
        $growth_rate = class_exists('ZiziCache\\DatabaseMonitor') ? DatabaseMonitor::calculate_growth_rate(7) : 0;
        $alerts = [];
        $db_alerts = get_option('zizi_cache_db_alerts', []);
        if (!empty($db_alerts) && is_array($db_alerts)) {
            $alerts = $db_alerts;
        }
        return rest_ensure_response([
            'history' => $formatted_data,
            'days' => $days,
            'growth_rate' => round($growth_rate, 2),
            'alerts' => $alerts
        ]);
    }

    /**
     * Logs database-related events to the zizi-log.log file.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function log_database_event($request)
    {
        $action = sanitize_text_field($request->get_param('action') ?? 'unknown');
        $source = sanitize_text_field($request->get_param('source') ?? 'unknown');
        $details = sanitize_text_field($request->get_param('details') ?? '{}');
        $message = sprintf('[ZiziCache DB Event] Action: %s, Source: %s, Details: %s',
            $action,
            $source,
            $details
        );
        CacheSys::writeLog($message);
        return rest_ensure_response([
            'status' => 'success',
            'message' => 'Event logged successfully'
        ]);
    }

    /**
     * Returns logs of database operations.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public static function get_database_operation_logs($request)
    {
        $limit = $request->get_param('limit') ? intval($request->get_param('limit')) : 50;
        $logs = DBModel::get_database_operation_logs($limit);
        foreach ($logs as &$log) {
            $log['time'] = date('Y-m-d H:i:s', $log['timestamp']);
            $log['user'] = $log['user_id'] ? get_userdata($log['user_id'])->user_login : 'system';
        }
        return rest_ensure_response([
            'logs' => $logs,
            'total' => count($logs)
        ]);
    }

    /**
     * Returns the database status.
     *
     * @return array
     */
    public static function get_database_status()
    {
        return DatabaseStatus::get_status();
    }

    /**
     * Returns cleanup counts for manual DB options.
     *
     * @param \WP_REST_Request $request
     * @return array
     */
    public static function get_cleanup_counts($request)
    {
        $params = $request->get_method() === 'POST' ? 
            $request->get_json_params() : 
            $request->get_params();
        $options = ($request->get_method() === 'GET') ? null :
            (isset($params['options']) && is_array($params['options']) ? $params['options'] : null);
        if ($request->get_method() === 'GET') {
            delete_transient('zizi_cache_db_counts');
        }
        $counts = DBModel::get_counts($options);
        $counts['_request_method'] = $request->get_method();
        $counts['_cached'] = false;
        return rest_ensure_response($counts);
    }

    /**
     * Returns index recommendations for the database.    /**
     * Gets database index recommendations.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */    public static function get_index_recommendations($request)
    {
        try {
            $database = new DBModel();
            $recommendations = $database->get_index_recommendations();
            
            // Convert complex recommendations to simple format for frontend
            $simple_recommendations = [];
            if (isset($recommendations['recommended_indexes']) && is_array($recommendations['recommended_indexes'])) {
                foreach ($recommendations['recommended_indexes'] as $index => $rec) {
                    // Handle columns array - join them for display
                    $columns = isset($rec['columns']) && is_array($rec['columns']) 
                        ? implode(', ', $rec['columns']) 
                        : ($rec['columns'] ?? 'unknown');
                        
                    $simple_recommendations[] = [
                        'id' => $index, // Add index ID for applying later
                        'table' => $rec['table'] ?? 'unknown',
                        'column' => $columns,
                        'reason' => $rec['reason'] ?? 'Performance optimization',
                        'priority' => $rec['priority'] ?? 'medium',
                        'index_name' => $rec['index_name'] ?? '',
                        'sql' => $rec['sql'] ?? ''
                    ];
                }
            }
            
            // Store full recommendations in transient for later use
            set_transient('zizi_cache_index_recommendations', $recommendations, 3600);
            
            return rest_ensure_response([
                'status' => 'success',
                'data' => $simple_recommendations
            ]);
        } catch (\Exception $e) {
            return new \WP_Error('db_error', 'Failed to get index recommendations: ' . $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * Applies selected database index recommendations.
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */    public static function apply_index_recommendations($request)
    {
        $params = $request->get_json_params();
        
        if (empty($params['recommendations']) || !is_array($params['recommendations'])) {
            return new \WP_Error('invalid_request', 'Missing or invalid recommendations parameter', ['status' => 400]);
        }

        try {
            // Get stored recommendations from transient
            $stored_recommendations = get_transient('zizi_cache_index_recommendations');
            if (!$stored_recommendations || !isset($stored_recommendations['recommended_indexes'])) {
                return new \WP_Error('invalid_request', 'No stored recommendations found. Please refresh recommendations first.', ['status' => 400]);
            }

            $database = new DBModel();
            $results = [];
            $to_apply = [];
            
            // Map selected recommendations by ID to full recommendation data
            foreach ($params['recommendations'] as $recommendation) {
                if (!isset($recommendation['id'])) {
                    continue;
                }
                
                $id = intval($recommendation['id']);
                if (isset($stored_recommendations['recommended_indexes'][$id])) {
                    $to_apply[] = $stored_recommendations['recommended_indexes'][$id];
                }
            }
            
            if (empty($to_apply)) {
                return new \WP_Error('invalid_request', 'No valid recommendations to apply', ['status' => 400]);
            }
            
            // Apply the indexes
            $apply_results = $database->apply_index_recommendations($to_apply);
            
            return rest_ensure_response([
                'status' => 'success',
                'data' => $apply_results,
                'message' => 'Index recommendations processed'
            ]);
            
        } catch (\Exception $e) {
            return new \WP_Error('db_error', 'Failed to apply index recommendations: ' . $e->getMessage(), ['status' => 500]);
        }
    }
}
