<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\SysConfig;
use ZiziCache\Purge;
use ZiziCache\Preload;
use ZiziCache\ObjectCacheHandler;
use ZiziCache\RestApi\ValidationTrait;

/**
 * Class Config
 *
 * Handles REST API endpoints for plugin configuration.
 *
 * @package ZiziCache\RestApi
 */
class Config
{
    use ValidationTrait;

    /**
     * Endpoint definitions for configuration.
     *
     * @var array
     */
    protected static $endpoints = [
        [
            'route' => '/config',
            'methods' => 'GET',
            'callback' => 'get_config',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/config',
            'methods' => 'POST',
            'callback' => 'update_config',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all configuration endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'] . '/?', [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Returns the current plugin configuration.
     *
     * @return array
     */
    public static function get_config()
    {
        $config = SysConfig::$config;
        
        // Neposílat heslo k Redis, ale pouze informaci, zda je nastaveno
        $config['redis_password_set'] = !empty($config['redis_password']);
        $config['redis_password'] = '';
        
        return $config;
    }

    /**
     * Updates the plugin configuration.
     *
     * @param \WP_REST_Request $request
     * @return array|\WP_Error
     */
    public static function update_config($request)
    {
        $config = $request->get_json_params();
        if (empty($config)) {
            return new \WP_Error('zizi-cache/invalid-config', 'Invalid config');
        }

        // Store original keys for optimized response
        $original_keys = array_keys($config);
        
        if (isset($config['speculative_loading_exclusions']) && is_string($config['speculative_loading_exclusions'])) {
            $config['speculative_loading_exclusions'] = array_filter(array_map('trim', explode("\n", $config['speculative_loading_exclusions'])));
        }
        
        if (isset($config['cdn_url']) && !empty($config['cdn_url'])) {
            $cdn_url = trim($config['cdn_url']);
            $cdn_url = rtrim(trim($cdn_url), '/');
            if (!preg_match('~^https?://~i', $cdn_url)) {
                $cdn_url = 'http://' . $cdn_url;
            }
            $is_local = preg_match('~^https?://(localhost|127\\.0\\.0\\.1|::1|.*\\.localhost|.*\\.local|.*\\.test|.*\\.dev)~i', $cdn_url);
            if ($is_local && strpos($cdn_url, 'https://') === 0 && empty($config['cdn_force_https'])) {
                $cdn_url = 'http://' . substr($cdn_url, 8);
            }
            $is_local_domain = preg_match('~^https?://(localhost|127\\.0\\.0\\.1|::1|.*\\.localhost|.*\\.local|.*\\.test|.*\\.dev)~i', $cdn_url);
            if (!$is_local_domain && filter_var($cdn_url, FILTER_VALIDATE_URL) === false) {
                return new \WP_Error(
                    'zizi-cache/invalid-cdn-url', 
                    'Invalid CDN URL format. URL must be in a valid format like "domain.com" or "cdn.example.com".'
                );
            }
            $parsed_url = parse_url($cdn_url);
            if (empty($parsed_url['host'])) {
                return new \WP_Error(
                    'zizi-cache/invalid-cdn-url', 
                    'Missing host name in CDN URL.'
                );
            }
            $host = $parsed_url['host'];
            if (!$is_local_domain && !preg_match('/\\.[a-z]{2,}$/i', $host) && !filter_var($host, FILTER_VALIDATE_IP)) {
                if (preg_match('/(localhost|\\.local|\\.test|\\.dev)$/i', $host)) {
                    // ok
                } else {
                    return new \WP_Error(
                        'zizi-cache/invalid-cdn-url',
                        'URL must contain a top-level domain (e.g., ".com", ".org") or be a valid local domain.'
                    );
                }
            }
            if ($is_local_domain) {
                \ZiziCache\CacheSys::writeLog('[CDN-CONFIG] Detected local CDN URL: ' . $cdn_url);
            }
            $config['cdn_url'] = $cdn_url;
        }
        
        // Check if Font Intelligence was disabled
        $previous_config = get_option('ZIZI_CACHE_CONFIG', []);
        SysConfig::update_config($config);
        
        Purge::purge_pages();
        Preload::preload_cache(false);
        ObjectCacheHandler::maybe_update_dropin(SysConfig::$config);
        
        // OPTIMALIZACE: Vracet pouze relevantní data místo celé konfigurace
        // Check if client wants full config (backward compatibility)
        $return_full = $request->get_param('return_full_config') === 'true';
        
        if ($return_full) {
            // Backward compatibility - return full config
            $updatedConfig = SysConfig::$config;
            $updatedConfig['redis_password_set'] = !empty($updatedConfig['redis_password']);
            $updatedConfig['redis_password'] = '';
            return $updatedConfig;
        } else {
            // OPTIMALIZOVANA RESPONSE: Vracet jen relevantní data
            $optimized_response = [
                'success' => true,
                'message' => 'Configuration updated successfully',
                'updated_at' => current_time('timestamp'),
                'updated_keys' => $original_keys
            ];
            
            // Vratit jen aktualizovane hodnoty z konfigurace
            $current_config = SysConfig::$config;
            foreach ($original_keys as $key) {
                if ($key === 'redis_password') {
                    $optimized_response['updated_config'][$key] = '';
                    $optimized_response['updated_config']['redis_password_set'] = !empty($current_config[$key]);
                } else if (isset($current_config[$key])) {
                    $optimized_response['updated_config'][$key] = $current_config[$key];
                }
            }
            
            return $optimized_response;
        }
    }
}