<?php
namespace ZiziCache\RestApi;

use ZiziCache\Authority;
use ZiziCache\CacheSys;
use ZiziCache\Purge;
use ZiziCache\Preload;
use ZiziCache\RestApi\ValidationTrait;

/**
 * Class Cache
 *
 * Handles REST API endpoints for cache management (purge, preload, count, etc.).
 *
 * @package ZiziCache\RestApi
 */
class Cache
{
    use ValidationTrait;

    /**
     * Endpoint definitions for cache management.
     *
     * @var array
     */    protected static $endpoints = [
        [
            'route' => '/cached-pages-count',
            'methods' => 'GET',
            'callback' => 'get_cached_pages_count',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-pages',
            'methods' => 'POST',
            'callback' => 'purge_pages',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-current-page',
            'methods' => 'POST',
            'callback' => 'purge_current_page',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-pages-and-preload',
            'methods' => 'POST',
            'callback' => 'purge_pages_and_preload',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-everything',
            'methods' => 'POST',
            'callback' => 'purge_everything',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-everything-and-preload',
            'methods' => 'POST',
            'callback' => 'purge_everything_and_preload',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/preload-cache',
            'methods' => 'POST',
            'callback' => 'preload_cache',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
        [
            'route' => '/purge-critical-css',
            'methods' => 'POST',
            'callback' => 'purge_critical_css',
            'permission_callback' => [Authority::class, 'rest_api_permission_callback'], // CSRF protected
        ],
    ];

    /**
     * Registers all cache management endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'] . '/?', [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }    /**
     * Returns the number of cached pages with optional extended statistics.
     *
     * @param \WP_REST_Request $request REST API request object
     * @return array Response with count and optional statistics
     */
    public static function get_cached_pages_count($request = null)
    {
        $cache_dir = defined('ZIZI_CACHE_CACHE_DIR') ? ZIZI_CACHE_CACHE_DIR : null;
        
        if (!$cache_dir) {
            return ['error' => 'Cache directory not defined'];
        }
        
        // Check if extended statistics are requested
        $extended = $request ? $request->get_param('extended') : false;
        $validate = $request ? $request->get_param('validate') : false;
        $force_refresh = $request ? $request->get_param('force_refresh') : false;
        
        if ($extended) {
            $stats = CacheSys::count_pages_extended($cache_dir, (bool) $validate);
            return [
                'count' => $stats['count'],
                'statistics' => $stats
            ];
        } else {
            $count = CacheSys::count_pages($cache_dir, (bool) $force_refresh);
            return ['count' => $count];
        }
    }

    /**
     * Purges all cached pages.
     *
     * @return array
     */
    public static function purge_pages()
    {
        try {
            Purge::purge_pages();
            return ['success' => true];
        } catch (\Throwable $e) {
            return [
                'success' => false,
                'error' => 'Exception: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Purges the cache for the current page and preloads it.
     *
     * @param \WP_REST_Request $request
     * @return array|\WP_Error
     */    public static function purge_current_page($request)
    {
        $url = $request->get_param('url');
        
        // Enhanced validation: URL format + security check
        if (empty($url) || !self::validate_url($url)) {
            \ZiziCache\CacheSys::writeError(
                "Invalid/malicious URL in purge request: " . ($url ?? 'empty') . " from IP: " . 
                ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
                'Security'
            );
            return new \WP_Error('zizi-cache/invalid-url', 'Invalid or unsafe URL');
        }
        
        // Additional length check for extremely long URLs
        if (strlen($url) > 2048) {
            \ZiziCache\CacheSys::writeError(
                "URL too long in purge request: " . strlen($url) . " chars from IP: " . 
                ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
                'Security'
            );
            return new \WP_Error('zizi-cache/invalid-url', 'URL too long');
        }
        
        Purge::purge_urls([$url]);
        Preload::preload_url($url);
        return ['success' => true];
    }

    /**
     * Purges all cached pages and preloads them.
     *
     * @return array
     */
    public static function purge_pages_and_preload()
    {
        Purge::purge_pages();
        Preload::preload_cache();
        return ['success' => true];
    }

    /**
     * Purges all cache (pages, objects, etc.).
     *
     * @return array
     */
    public static function purge_everything()
    {
        Purge::purge_everything();
        return ['success' => true];
    }

    /**
     * Purges all cache and preloads it.
     *
     * @return array
     */
    public static function purge_everything_and_preload()
    {
        Purge::purge_everything();
        Preload::preload_cache();
        return ['success' => true];
    }

    /**
     * Starts cache preloading.
     *
     * @return array
     */
    public static function preload_cache()
    {
        Preload::preload_cache();
        return ['success' => true];
    }

    /**
     * Purges Critical CSS cache.
     *
     * @return array
     */
    public static function purge_critical_css()
    {
        try {
            // Use CSSEnhanced class if available, otherwise fallback
            if (class_exists('ZiziCache\CSSEnhanced')) {
                \ZiziCache\CSSEnhanced::clear_critical_css_cache();
            } else {
                // Fallback: manually clear critical CSS files
                $cache_dir = ZIZI_CACHE_CACHE_DIR . 'css/';
                if (is_dir($cache_dir)) {
                    $files = glob($cache_dir . '*.critical.css');
                    foreach ($files as $file) {
                        if (is_file($file)) {
                            unlink($file);
                        }
                    }
                }
            }
            
            CacheSys::writeWarning('Critical CSS cache cleared via admin bar', 'AdminBar');
            return ['success' => true, 'message' => 'Critical CSS cache cleared successfully'];
        } catch (\Exception $e) {
            CacheSys::writeError('Failed to clear Critical CSS cache: ' . $e->getMessage(), 'AdminBar');
            return ['success' => false, 'message' => 'Failed to clear Critical CSS cache'];
        }
    }
}
