<?php
namespace ZiziCache\RestApi;

use ZiziCache\ActionSchedulerCleanup;

/**
 * Class ActionScheduler
 *
 * Handles REST API endpoints for ActionScheduler status and cleanup.
 *
 * @package ZiziCache\RestApi
 */
class ActionScheduler
{
    /**
     * Endpoint definitions for ActionScheduler operations.
     *
     * @var array
     */
    protected static $endpoints = [
        [
            'route' => '/actionscheduler-status',
            'methods' => 'GET',
            'callback' => 'get_actionscheduler_status',
            'permission_callback' => [__CLASS__, 'can_manage_options'],
        ],
        [
            'route' => '/clean-actionscheduler',
            'methods' => 'POST',
            'callback' => 'clean_actionscheduler',
            'permission_callback' => [__CLASS__, 'can_manage_options'],
        ],
    ];

    /**
     * Registers all ActionScheduler endpoints under the given namespaces.
     *
     * @param array $namespaces
     * @return void
     */
    public static function register($namespaces)
    {
        foreach ($namespaces as $namespace) {
            foreach (self::$endpoints as $def) {
                register_rest_route($namespace, $def['route'], [
                    'methods' => $def['methods'],
                    'callback' => [__CLASS__, $def['callback']],
                    'permission_callback' => $def['permission_callback'],
                ]);
            }
        }
    }

    /**
     * Permission callback: checks if the current user can manage options.
     *
     * @return bool
     */
    public static function can_manage_options() {
        return current_user_can('manage_options');
    }

    /**
     * Returns the status of ActionScheduler database tables.
     *
     * @return \WP_REST_Response
     */
    public static function get_actionscheduler_status()
    {
        $status = ActionSchedulerCleanup::get_status_info();
        return rest_ensure_response($status);
    }

    /**
     * Runs cleanup on ActionScheduler tables.
     *
     * @return \WP_REST_Response
     */
    public static function clean_actionscheduler()
    {
        $result = ActionSchedulerCleanup::cleanup();
        return rest_ensure_response([
            'success' => true,
            'logs_deleted' => $result['logs_deleted'],
            'actions_deleted' => $result['actions_deleted']
        ]);
    }
}
