<?php

namespace ZiziCache;

use ZiziCache\ObjectCacheHandler;

/**
 * Class RestApi
 *
 * Delegates registration of all custom REST API endpoints for the ZiziCache plugin.
 * Each endpoint group (cache, database, redis, CDN, ActionScheduler, config, server info)
 * is registered via its own domain class.
 *
 * @package ZiziCache
 */

use ZiziCache\RestApi\Cache as CacheApi;
use ZiziCache\RestApi\FontStatistics as FontStatisticsApi;

class RestApi
{
    /**
     * Initializes the REST API and delegates endpoint registration to domain-specific classes.
     *
     * @return void
     */
    public static function init()
    {
        add_action('rest_api_init', [__CLASS__, 'register_rest_apis']);
        
        // Add global error handling for our REST API endpoints
        add_filter('rest_pre_dispatch', [__CLASS__, 'handle_rest_errors'], 10, 3);
    }

    /**
     * Global error handler for ZiziCache REST API endpoints
     * Ensures clean JSON output by suppressing PHP warnings/notices
     *
     * @param mixed $result
     * @param WP_REST_Server $server
     * @param WP_REST_Request $request
     * @return mixed
     */
    public static function handle_rest_errors($result, $server, $request)
    {
        // Only handle our own endpoints
        $route = $request->get_route();
        if (strpos($route, '/zizi-cache/') === false) {
            return $result;
        }
        
        // Start output buffering to catch any unexpected output
        if (!isset($GLOBALS['zizi_rest_ob_started'])) {
            ob_start();
            $GLOBALS['zizi_rest_ob_started'] = true;
            
            // Register shutdown handler to clean up output buffer
            add_action('rest_api_shutdown', function() {
                if (ob_get_level() > 0) {
                    $captured = ob_get_clean();
                    if (!empty($captured)) {
                        error_log('ZiziCache REST API: Unexpected output captured: ' . $captured);
                    }
                }
                unset($GLOBALS['zizi_rest_ob_started']);
            });
        }
        
        return $result;
    }

    /**
     * Delegates endpoint registration to domain-specific classes according to the refactoring design.
     * Endpoints are registered under multiple namespaces for compatibility.
     *
     * @return void
     */
    public static function register_rest_apis()
    {
        // Dynamic plugin directory name
        $plugin_dir_name = basename(dirname(ZIZI_CACHE_FILE));
        // All namespaces for compatibility
        $namespaces = [
            $plugin_dir_name,
            'zizi-cache',
            'zizi-cache/v1',
        ];

        // Register cache management endpoints
        CacheApi::register($namespaces);
        
        // Register Font Statistics endpoints
        FontStatisticsApi::register($namespaces);

        // Register database endpoints
        \ZiziCache\RestApi\Database::register($namespaces);
        // Register Redis endpoints
        \ZiziCache\RestApi\Redis::register($namespaces);
        // Register OPcache endpoints
        \ZiziCache\RestApi\OPcacheApi::register($namespaces);
        // Register CDN endpoints
        \ZiziCache\RestApi\CDN::register($namespaces);
        // Register ActionScheduler endpoints
        \ZiziCache\RestApi\ActionScheduler::register($namespaces);
        // Register configuration endpoints
        \ZiziCache\RestApi\Config::register($namespaces);
        // Register server info endpoints
        \ZiziCache\RestApi\ServerInfo::register($namespaces);
        
        // Register Image Converter endpoints
        if (class_exists('\\ZiziCache\\RestApi\\ImageConverterAdmin')) {
            \ZiziCache\RestApi\ImageConverterAdmin::register($namespaces);
        }
    }
}
