<?php

declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use WP_Scripts;
use WP_Post;

/**
 * Redundancy Removal and Resource Optimization
 *
 * The Redundancy class provides a comprehensive set of static methods to disable or optimize
 * various WordPress features and third-party plugin integrations that are often unnecessary or
 * detrimental to performance. Its main goal is to reduce bloat, improve security, and enhance
 * the speed of WordPress sites by removing redundant scripts, styles, and functionalities.
 *
 * Key responsibilities include:
 *
 * - Disabling unnecessary WordPress features (block CSS, oEmbeds, emojis, XML-RPC, RSS feeds, etc.)
 * - Removing unused CSS and JavaScript assets, including those from WooCommerce and Google Maps
 * - Optimizing database queries and controlling the number of post revisions
 * - Managing the WordPress Heartbeat API (frequency and behavior)
 * - Disabling or restricting REST API endpoints for non-logged-in users
 * - Preventing self-pingbacks and hiding WordPress version information
 * - Disabling comments system-wide, including widgets and admin bar counters
 * - Removing Google Fonts and replacing them with system font stacks
 * - Providing utility methods for asset dequeuing and regular expression pattern caching
 *
 * This class is designed to be initialized via the static init() method, which registers all
 * relevant hooks and filters based on the plugin's configuration. All methods are static and
 * intended for use as WordPress action/filter callbacks or internal helpers.
 *
 * @package ZiziCache
 */
class Redundancy
{
    /**
     * Cache for frequently used values to avoid redundant function calls.
     *
     * @var bool|null $is_user_logged_in   Cached result of is_user_logged_in().
     * @var bool|null $is_woocommerce_active Cached result of WooCommerce activation check.
     * @var array     $compiled_patterns   Cache for compiled regular expression patterns.
     */
    private static $is_user_logged_in = null;
    private static $is_woocommerce_active = null;
    private static $compiled_patterns = [];
    
    /**
     * Initializes redundancy removal and optimization features.
     *
     * This is the main entry point for the Redundancy class. It dynamically registers all
     * WordPress hooks and filters required to disable or optimize features, based on the
     * plugin's configuration array. Each configuration key is mapped to a specific method
     * and WordPress hook, allowing granular control over which optimizations are enabled.
     *
     * Example usage:
     *   Redundancy::init();
     *
     * @return void
     */
    public static function init(): void
    {
        $config = SysConfig::$config; // Cache config for slightly better performance

        // Definition of configuration key to method mapping
        $hooks = [
            'redundancy_disable_block_css' => [
                'hook' => 'wp_enqueue_scripts',
                'method' => 'disable_block_library_css',
                'priority' => 100,
                'type' => 'action'
            ],
            'redundancy_disable_oembeds' => [
                'hook' => 'init',
                'method' => 'disable_oembed',
                'type' => 'action'
            ],
            'redundancy_disable_cron' => [
                'hook' => 'init',
                'method' => 'disable_cron',
                'priority' => 1,
                'type' => 'action'
            ],
            'redundancy_disable_emojis' => [
                'hooks' => [
                    ['hook' => 'init', 'method' => 'disable_emojis'],
                    ['hook' => 'tiny_mce_plugins', 'method' => 'disable_emojis_tinymce'],
                    ['hook' => 'wp_resource_hints', 'method' => 'disable_emojis_dns_prefetch', 'args' => 2],
                ],
                'type' => 'mixed'
            ],
            'redundancy_disable_jquery_migrate' => [
                'hook' => 'wp_default_scripts',
                'method' => 'disable_jquery_migrate',
                'type' => 'filter'
            ],
            'redundancy_disable_dashicons' => [
                'hook' => 'wp_enqueue_scripts',
                'method' => 'disable_dashicons',
                'priority' => 100,
                'type' => 'action'
            ],
            'redundancy_disable_xml_rpc' => [
                'hook' => 'init',
                'method' => 'disable_xml_rpc',
                'type' => 'action'
            ],
            'redundancy_disable_rss_feed' => [
                'hooks' => [
                    ['hook' => 'init', 'method' => 'disable_rss_feed'],
                    ['hook' => 'do_feed', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_rdf', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_rss', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_rss2', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_atom', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_rss2_comments', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                    ['hook' => 'do_feed_atom_comments', 'method' => 'redirect_feed_to_home', 'priority' => 1],
                ],
                'type' => 'mixed'
            ],
            'redundancy_disable_woo_cart_fragments' => [
                'hook' => 'wp_enqueue_scripts',
                'method' => 'disable_woocommerce_cart_fragments',
                'priority' => 11,
                'type' => 'action'
            ],
            'redundancy_disable_woo_assets' => [
                'hook' => 'wp_enqueue_scripts',
                'method' => 'disable_woocommerce_assets',
                'priority' => 11,
                'type' => 'action'
            ],
            'redundancy_post_revisions_control' => [
                'hook' => 'wp_revisions_to_keep',
                'method' => 'control_post_revisions',
                'priority' => 10,
                'args' => 2,
                'type' => 'filter'
            ],
            'redundancy_heartbeat_control' => [
                'hooks' => [
                    ['hook' => 'heartbeat_settings', 'method' => 'set_heartbeat_frequency'],
                    ['hook' => 'init', 'method' => 'set_heartbeat_behaviour', 'priority' => 1],
                ],
                'type' => 'mixed'
            ],
            'redundancy_disable_self_pingbacks' => [
                'hook' => 'pre_ping',
                'method' => 'disable_self_pingbacks',
                'type' => 'action'
            ],
            'redundancy_disable_rest_api' => [
                'hook' => 'init',
                'method' => 'disable_rest_api',
                'type' => 'action'
            ],
            'redundancy_disable_google_maps' => [
                'hook' => 'wp_enqueue_scripts',
                'method' => 'disable_google_maps',
                'type' => 'action'
            ],
            'redundancy_disable_password_strength_meter' => [
                'hook' => 'wp_print_scripts',
                'method' => 'disable_password_strength_meter',
                'priority' => 100,
                'type' => 'action'
            ],
            'redundancy_disable_comments' => [
                'hook' => 'init',
                'method' => 'disable_comments',
                'type' => 'action'
            ],
            'redundancy_hide_wp_version' => [
                'hooks' => [
                    ['hook' => 'the_generator', 'method' => '__return_empty_string'],
                    ['hook' => 'init', 'method' => 'hide_wp_version'],
                ],
                'type' => 'mixed'
            ]
        ];
        
        // Dynamic registration of hooks based on configuration
        foreach ($hooks as $config_key => $hook_data) {
            if (!isset($config[$config_key]) || !$config[$config_key]) {
                continue;
            }

            // Pokud má hook více definovaných hooků (např. script_loader_src a style_loader_src)
            if (isset($hook_data['hooks']) && is_array($hook_data['hooks'])) {
                foreach ($hook_data['hooks'] as $subhook) {
                    $priority = $subhook['priority'] ?? 10;
                    $args = $subhook['args'] ?? 1;
                    if ($hook_data['type'] === 'filter' || $subhook['hook'] === 'tiny_mce_plugins' || $subhook['hook'] === 'wp_resource_hints' || $subhook['hook'] === 'heartbeat_settings') {
                        add_filter($subhook['hook'], [__CLASS__, $subhook['method']], $priority, $args);
                    } else {
                        add_action($subhook['hook'], [__CLASS__, $subhook['method']], $priority);
                    }
                }
            } else {
                $priority = $hook_data['priority'] ?? 10;
                $args = $hook_data['args'] ?? 1;
                $method = $hook_data['method'];
                
                if ($hook_data['type'] === 'filter') {
                    add_filter($hook_data['hook'], [__CLASS__, $method], $priority, $args);
                } else {
                    add_action($hook_data['hook'], [__CLASS__, $method], $priority);
                }
            }
        }
    }
    
    /**
     * Disables block library CSS.
     *
     * Removes unnecessary block library CSS files (including WooCommerce block styles, if active)
     * to reduce HTTP requests and improve page load performance. This method dequeues styles such as
     * 'wp-block-library', 'wp-block-library-theme', and 'global-styles', as well as WooCommerce block styles
     * if WooCommerce is active.
     *
     * @return void
     */
    public static function disable_block_library_css(): void
    {
        $assets = ['wp-block-library', 'wp-block-library-theme', 'global-styles'];
        
        // WooCommerce-specific styles
        if (self::is_woocommerce_active()) {
            $assets[] = 'wc-blocks-vendors-style';
            $assets[] = 'wc-all-blocks-style';
            $assets[] = 'wc-blocks-style';
        }
        
        self::dequeue_assets($assets, 'style');
    }

    /**
     * Disables oEmbed functionality.
     *
     * Removes oEmbed discovery links from the HTML head, disables oEmbed REST API endpoints,
     * and removes related rewrite rules. This reduces HTTP requests, improves security, and
     * prevents unnecessary oEmbed processing.
     *
     * @return void
     */
    public static function disable_oembed(): void
    {
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
        
        // Odstranit rewrite rules pro oEmbed
        add_filter('rewrite_rules_array', function (array $rules): array {
            foreach ($rules as $rule => $rewrite) {
                if (strpos($rewrite, 'embed=true') !== false) {
                    unset($rules[$rule]);
                }
            }
            return $rules;
        });

        // Odstranit REST API endpointy pro oEmbed
        add_filter('rest_endpoints', function (array $endpoints): array {
            unset($endpoints['/oembed/1.0/embed']);
            return $endpoints;
        });

        remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
        remove_action('rest_api_init', 'wp_oembed_register_route');
        add_filter('embed_oembed_discover', '__return_false');
        remove_filter('the_content_feed', '_oembed_filter_feed_content');
        // Note: rest_enabled filter was deprecated since WP 4.7.0
        // REST API cannot be completely disabled, use rest_authentication_errors instead if needed
    }

    /**
     * Disables the default WordPress cron system.
     *
     * Sets the DISABLE_WP_CRON constant to true, preventing WordPress from running scheduled tasks
     * on page loads. This is recommended for high-traffic sites, but requires a real server-side cron job
     * to trigger wp-cron.php manually.
     *
     * @return void
     */
    public static function disable_cron(): void
    {
        if (!defined('DISABLE_WP_CRON') || DISABLE_WP_CRON === false) {
            define('DISABLE_WP_CRON', true);
        }
    }

    /**
     * Disables emoji support in WordPress.
     *
     * Removes all emoji-related scripts, styles, and filters from both frontend and admin,
     * reducing HTTP requests and improving page load performance.
     *
     * @return void
     */
    public static function disable_emojis(): void
    {
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
        add_filter('emoji_svg_url', '__return_false');
    }

    /**
     * Removes the emoji plugin from the TinyMCE editor.
     *
     * Filters the list of TinyMCE plugins to remove 'wpemoji', preventing emoji support in the editor.
     *
     * @param array $plugins List of TinyMCE plugins.
     * @return array Filtered list of TinyMCE plugins.
     */
    public static function disable_emojis_tinymce($plugins): array
    {
        if (!is_array($plugins)) {
            return [];
        }
        $key = array_search('wpemoji', $plugins);
        if ($key !== false) {
            unset($plugins[$key]);
        }
        return $plugins;
    }

    /**
     * Removes emoji DNS prefetch resource hints.
     *
     * Filters the resource hints array to remove any DNS prefetch links to s.w.org (used for emojis).
     *
     * @param array $urls URLs to print for resource hints.
     * @param string $relation_type The relation type the URLs are printed for.
     * @return array Filtered resource hints.
     */
    public static function disable_emojis_dns_prefetch(array $urls, string $relation_type): array
    {
        if ($relation_type !== 'dns-prefetch') {
            return $urls;
        }
        $filtered = [];
        foreach ($urls as $url) {
            if (strpos($url, 's.w.org') === false) {
                $filtered[] = $url;
            }
        }
        return $filtered;
    }

    /**
     * Disables jQuery Migrate script.
     *
     * Removes 'jquery-migrate' from the dependencies of all registered jQuery scripts. This reduces
     * page weight, but should only be used if you are certain your theme and plugins do not require jQuery Migrate.
     *
     * @param WP_Scripts $scripts WP_Scripts instance.
     * @return void
     */
    public static function disable_jquery_migrate($scripts): void
    {
        if (!is_a($scripts, 'WP_Scripts')) {
            return;
        }
        
        foreach (array_keys($scripts->registered) as $handle) {
            if (strpos($handle, 'jquery') !== false) {
                $script = $scripts->registered[$handle];
                if (is_array($script->deps)) {
                    $script->deps = array_diff($script->deps, ['jquery-migrate']);
                }
            }
        }
    }    

    /**
     * Disables Dashicons CSS for non-logged-in users.
     *
     * Dequeues the 'dashicons' style on the frontend for users who are not logged in, reducing
     * unnecessary CSS on public pages. Dashicons are retained for logged-in users and in the admin area.
     *
     * @return void
     */
    public static function disable_dashicons(): void
    {
        // Zachovat dashicons pro přihlášené uživatele nebo v adminu
        if (!is_admin() && !self::is_user_logged_in()) {
            self::dequeue_assets(['dashicons'], 'style');
        }
    }

    /**
     * Disables XML-RPC functionality.
     *
     * Prevents XML-RPC requests by disabling the feature and removing related links from the HTML head.
     * This improves security by reducing potential attack vectors.
     *
     * @return void
     */
    public static function disable_xml_rpc(): void
    {
        add_filter('xmlrpc_enabled', '__return_false');
        remove_action('wp_head', 'rsd_link');
        remove_action('wp_head', 'wlwmanifest_link');
    }

    /**
     * Disables RSS feeds.
     *
     * Removes RSS feed links from the HTML head and disables feed functionality, reducing server load
     * and improving security by preventing content scraping and feed-based attacks.
     *
     * @return void
     */
    public static function disable_rss_feed(): void
    {
        remove_action('wp_head', 'feed_links', 2);
        remove_action('wp_head', 'feed_links_extra', 3);
    }

    /**
     * Redirects feed visitors to the homepage.
     *
     * If a user directly accesses an RSS feed URL, this method issues a 301 redirect to the site's homepage.
     * If headers have already been sent, it falls back to displaying a message using wp_die().
     *
     * @return void
     */
    public static function redirect_feed_to_home(): void
    {
        $home_url = home_url();
        $message = sprintf(
            wp_kses(
                /* translators: %s: home URL */
                __('No feed available, please visit our <a href="%s">homepage</a>!', 'zizi-cache'),
                ['a' => ['href' => []]]
            ),
            esc_url($home_url)
        );
        
        // Ensure headers are not sent before wp_die
        if (!headers_sent()) {
            status_header(301); // Permanent redirect
            wp_redirect(esc_url_raw($home_url), 301);
            exit;
        } else {
            // Fallback if headers already sent, though less ideal
            wp_die($message, 'Feed Disabled', ['response' => 301]);
        }
    }

    /**
     * Disables WooCommerce cart fragments on non-WooCommerce pages.
     *
     * Dequeues the 'wc-cart-fragments' script on all pages except WooCommerce, cart, and checkout pages.
     * This reduces unnecessary JavaScript execution and improves frontend performance.
     *
     * @return void
     */
    public static function disable_woocommerce_cart_fragments(): void
    {
        if (is_admin() || !self::is_woocommerce_active() || is_woocommerce() || is_cart() || is_checkout()) {
            return;
        }
        
        self::dequeue_assets(['wc-cart-fragments'], 'script');
    }

    /**
     * Disables WooCommerce assets on non-WooCommerce pages.
     *
     * Dequeues a list of WooCommerce-related scripts and styles on all pages except WooCommerce, cart,
     * checkout, and account pages. This significantly reduces frontend bloat on non-shop pages.
     *
     * @return void
     */
    public static function disable_woocommerce_assets(): void
    {
        if (is_admin() || !self::is_woocommerce_active() || is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
            return;
        }

        $woo_scripts = [
            'wc-add-to-cart', 'wc-cart-fragments', 'wc-checkout', 'wc-add-to-cart-variation',
            'wc-single-product', 'wc-cart', 'wc-chosen', 'woocommerce', 'prettyPhoto',
            'prettyPhoto-init', 'jquery-blockui', 'jquery-placeholder', 'fancybox', 'jqueryui'
        ];
        
        $woo_styles = [
            'woocommerce-layout', 'woocommerce-smallscreen', 'woocommerce-general',
            'woocommerce_fancybox_styles', 'woocommerce_chosen_styles', 'woocommerce_prettyPhoto_css',
            'wc-blocks-style'
        ];
        
        self::dequeue_assets($woo_scripts, 'script');
        self::dequeue_assets($woo_styles, 'style');
    }

    /**
     * Removes WordPress version from script and style URLs.
     *
     * Strips the 'ver' query parameter (containing the WordPress version) from asset URLs to prevent
     * information disclosure and improve browser caching of static assets.
     *
     * @param string $src The source URL of the enqueued style/script.
     * @return string The modified URL without the version parameter.
     */
    public static function remove_wp_version_strings($src): string
    {
        if (!is_string($src)) {
            return '';
        }
        
        // Skip version removal on WooCommerce admin pages to prevent conflicts
        if (is_admin() && isset($_GET['page']) && $_GET['page'] === 'wc-admin') {
            return $src;
        }
        
        // Skip version removal if admin bypass is active
        if (defined('ZIZI_CACHE_ADMIN_BYPASS') && ZIZI_CACHE_ADMIN_BYPASS) {
            return $src;
        }
        
        // Skip version removal for WooCommerce scripts/styles
        if (strpos($src, '/woocommerce/') !== false || strpos($src, 'wc-admin') !== false) {
            return $src;
        }
        
        global $wp_version;
        $version_str = 'ver=' . $wp_version;
        
        if (strpos($src, $version_str) !== false) {
            $src = remove_query_arg('ver', $src);
        }
        
        return $src;
    }
    
    /**
     * Removes Google Fonts from HTML content and injects a system font stack.
     *
     * Uses regular expressions to remove <link> and @import references to Google Fonts from the provided HTML.
     * If no system font stack is present, injects a <style> block with a recommended system font stack for body and code elements.
     *
     * @param string $html HTML content.
     * @return string Modified HTML content without Google Fonts and with a system font stack.
     */
    public static function remove_google_fonts(string $html): string
    {
        if (!is_string($html)) {
            return '';
        }

        // Optimized regular expressions with pattern caching
        $link_pattern = self::get_compiled_pattern(
            'google_fonts_link',
            '/<link[^>]+href=([\'\"])(https?:)?\/\/fonts\.googleapis\.com\/css[^\'\"]*\1[^>]*>/i'
        );
        $import_pattern = self::get_compiled_pattern(
            'google_fonts_import',
            '/@import\s+url\(([^)]*fonts\.googleapis\.com[^)]*)\);?/i'
        );

        // Remove Google Fonts
        $html = preg_replace($link_pattern, '', $html);
        $html = preg_replace($import_pattern, '', $html);

        // Add system fonts if not already present
        $system_font_style_id = 'zizi-cache-system-font-stack';
        if (strpos($html, $system_font_style_id) === false) {
            $style = "<style id=\"{$system_font_style_id}\">body{font-family:-apple-system,system-ui,BlinkMacSystemFont,'Segoe UI',Roboto,Oxygen,Ubuntu,Cantarell,'Fira Sans','Droid Sans','Helvetica Neue',sans-serif!important;}code,pre,kbd,samp{font-family:Menlo,Consolas,Monaco,Liberation Mono,Lucida Console,monospace!important;}</style>";            if (strpos($html, '</head>') !== false) {
                $html = SysTool::str_replace_first('</head>', $style . PHP_EOL . '</head>', $html);
            } else {
                // Only append if the content appears to be HTML (contains HTML tags)
                if (preg_match('/<[^>]+>/', $html)) {
                    $html .= $style;
                }
            }
        }

        return $html;
    }
    
    /**
     * Controls the number of post revisions to keep.
     *
     * Determines the number of post revisions WordPress should retain for each post, based on plugin configuration.
     * Can disable revisions entirely or set a custom limit. If no configuration is set, falls back to the default value.
     *
     * @param int|false $num  The current number of revisions to keep, or false to disable revisions.
     * @param WP_Post   $post The post object being saved.
     * @return int|false Number of revisions to keep, or false to disable revisions.
     */
    public static function control_post_revisions($num, WP_Post $post)
    {
        $limit_config = SysConfig::$config['redundancy_post_revisions_limit'] ?? null;

        if ($limit_config === 'disable') {
            return false;
        }
        if (is_numeric($limit_config)) {
            return (int) $limit_config;
        }
        return $num;
    }

    /**
     * Sets the WordPress Heartbeat API frequency.
     *
     * Modifies the interval (in seconds) at which the Heartbeat API sends requests, based on plugin configuration.
     * If the configuration is not set or invalid, the default interval is used.
     *
     * @param array<string, mixed> $settings Heartbeat settings.
     * @return array<string, mixed> Modified heartbeat settings.
     */
    public static function set_heartbeat_frequency(array $settings): array
    {
        $frequency_config = SysConfig::$config['redundancy_heartbeat_frequency'] ?? 'default';

        if ($frequency_config === 'default' || !is_numeric($frequency_config)) {
            return $settings;
        }

        $settings['interval'] = (int) $frequency_config;
        return $settings;
    }

    /**
     * Sets the behavior of the WordPress Heartbeat API.
     *
     * Controls whether the Heartbeat API is enabled, disabled, or limited to post editing screens,
     * based on plugin configuration. Can deregister the 'heartbeat' script on certain admin pages.
     *
     * @return void
     */
    public static function set_heartbeat_behaviour(): void
    {
        $behaviour_config = SysConfig::$config['redundancy_heartbeat_behaviour'] ?? 'default';

        if ($behaviour_config === 'default') {
            return;
        }

        global $pagenow;

        if (!is_string($pagenow)) {
            return;
        }

        if ($behaviour_config === 'edit_post') {
            if ($pagenow !== 'post.php' && $pagenow !== 'post-new.php') {
                wp_deregister_script('heartbeat');
            }
        } elseif ($behaviour_config === 'disable') {
            if (is_admin()) {
                wp_deregister_script('heartbeat');
            }
        }
    }

    /**
     * Disables self-pingbacks.
     *
     * Removes links to the site's own domain from the list of pingback URLs, preventing WordPress
     * from sending pingbacks to itself when linking to its own content.
     *
     * @param array $links List of links to ping (passed by reference).
     * @return void
     */
    public static function disable_self_pingbacks(&$links): void
    {
        $home_url = home_url();
        
        foreach ($links as $key => $link) {
            if (strpos($link, $home_url) !== false) {
                unset($links[$key]);
            }
        }
    }

    /**
     * Disables REST API for non-logged-in users.
     *
     * Removes REST API links from HTML headers and restricts access to REST API endpoints for unauthenticated users.
     * Allows certain endpoints (such as block rendering for Gutenberg) to remain accessible. Returns a 401 error for
     * all other REST API requests from non-logged-in users.
     *
     * @return void
     */
    public static function disable_rest_api(): void
    {
        if (self::is_user_logged_in()) {
            return;
        }

        // Hide REST API from headers
        remove_action('wp_head', 'rest_output_link_wp_head', 10);
        remove_action('template_redirect', 'rest_output_link_header', 11);

        // Restrict REST API functionality for non-logged-in users
        add_filter('rest_authentication_errors', function ($access) {
            // Allow basic endpoints (e.g., for Gutenberg)
            if (isset($_SERVER['REQUEST_URI'])) {
                $request_uri = sanitize_text_field($_SERVER['REQUEST_URI']);
                // Allow specific endpoints that must remain public for functionality
                $allowed_endpoints = [
                    'wp/v2/block-renderer',
                    // WooCommerce onboarding endpoints used by the admin wizard
                    'wc-admin/onboarding',
                    'wc-admin'
                ];

                foreach ($allowed_endpoints as $endpoint) {
                    if (strpos($request_uri, $endpoint) !== false) {
                        // Log allowed endpoint match for debugging
                        try {
                            if (class_exists('\\ZiziCache\\CacheSys')) {
                                \ZiziCache\CacheSys::writeLog(sprintf('Redundancy: Allowing REST endpoint %s for request %s (redundancy_disable_rest_api)', $endpoint, $request_uri), 'Redundancy');
                            } else {
                                error_log('Redundancy: Allowing REST endpoint ' . $endpoint . ' for request ' . $request_uri);
                            }
                        } catch (\Throwable $e) {
                            // noop - never break auth flow due to logging
                        }

                        return $access;
                    }
                }
            }
            
            if (!self::is_user_logged_in() && !is_null($access)) {
                return $access;
            }
            
            if (!self::is_user_logged_in()) {
                return new \WP_Error('rest_not_logged_in', 'REST API is only available to logged-in users.', ['status' => 401]);
            }
            
            return $access;
        });
    }

    /**
     * Disables loading of Google Maps API scripts.
     *
     * Dequeues any registered scripts whose source URLs contain 'maps.google.com' or 'maps.googleapis.com',
     * preventing Google Maps from loading and improving frontend performance.
     *
     * @return void
     */
    public static function disable_google_maps(): void
    {
        global $wp_scripts;
        
        if (!is_object($wp_scripts) || !is_array($wp_scripts->registered)) {
            return;
        }
        
        $google_maps_scripts = [];
        
        foreach ($wp_scripts->registered as $key => $script) {
            if (isset($script->src) && is_string($script->src) && 
                (strpos($script->src, 'maps.google.com') !== false || 
                 strpos($script->src, 'maps.googleapis.com') !== false)) {
                $google_maps_scripts[] = $key;
            }
        }
        
        self::dequeue_assets($google_maps_scripts, 'script');
    }

    /**
     * Disables the Password Strength Meter script on unnecessary pages.
     *
     * Dequeues the 'password-strength-meter' and 'wc-password-strength-meter' scripts except on WooCommerce checkout,
     * account, and add-payment-method pages, where password strength validation is required.
     *
     * @return void
     */
    public static function disable_password_strength_meter(): void
    {
        // Check if WooCommerce is active
        $woocommerce_active = self::is_woocommerce_active();
        
        // Keep on pages where it's needed
        if ($woocommerce_active && (is_checkout() || is_account_page() || is_page('my-account') || 
            (function_exists('is_wc_endpoint_url') && is_wc_endpoint_url('add-payment-method')))) {
            return;
        }
        
        self::dequeue_assets([
            'password-strength-meter',
            'wc-password-strength-meter'
        ], 'script');
    }
    
    /**
     * Disables comments system-wide.
     *
     * Closes comments and pings, hides existing comments, removes comment-related menu items and widgets,
     * removes comment counters from the admin bar, and removes comment/trackback support from all post types.
     *
     * @return void
     */
    public static function disable_comments(): void
    {
        // Close comments globally
        add_filter('comments_open', '__return_false', 20, 2);
        add_filter('pings_open', '__return_false', 20, 2);
        
        // Hide existing comments
        add_filter('comments_array', '__return_empty_array', 10, 2);
        
        // Remove menu items
        add_action('admin_menu', function() {
            remove_menu_page('edit-comments.php');
            remove_submenu_page('options-general.php', 'options-discussion.php');
        });
        
        // Remove comment widgets
        add_action('widgets_init', function () {
            unregister_widget('WP_Widget_Recent_Comments');
        });
        
        // Remove comment counters from admin bar
        add_action('admin_bar_menu', function ($wp_admin_bar) {
            $wp_admin_bar->remove_node('comments');
        }, 999);
        
        // Remove comment support from post types
        add_action('init', function () {
            $post_types = get_post_types();
            foreach ($post_types as $post_type) {
                if (post_type_supports($post_type, 'comments')) {
                    remove_post_type_support($post_type, 'comments');
                    remove_post_type_support($post_type, 'trackbacks');
                }
            }
        }, 99);
    }
    
    /**
     * Hides the WordPress version from public output.
     *
     * Removes version information from HTML headers, RSS feeds, and asset URLs (scripts and styles),
     * reducing information disclosure and improving security.
     *
     * @return void
     */
    public static function hide_wp_version(): void
    {
        // Skip on WooCommerce admin pages to prevent script conflicts
        if (is_admin() && isset($_GET['page']) && $_GET['page'] === 'wc-admin') {
            return;
        }
        
        // Skip if admin bypass is active
        if (defined('ZIZI_CACHE_ADMIN_BYPASS') && ZIZI_CACHE_ADMIN_BYPASS) {
            return;
        }
        
        remove_action('wp_head', 'wp_generator');
        
        // Remove version from RSS feeds
        add_filter('the_generator', '__return_empty_string');
        
        // Remove version from scripts and styles
        add_filter('style_loader_src', [__CLASS__, 'remove_wp_version_strings']);
        add_filter('script_loader_src', [__CLASS__, 'remove_wp_version_strings']);
    }
    
    /**
     * General utility for dequeuing scripts or styles.
     *
     * Iterates over a list of asset handles and dequeues each one, either as a script or style, if the provided
     * condition is true. Used internally by other methods to remove unnecessary assets from the page.
     *
     * @param array  $handles   List of script or style handles to dequeue.
     * @param string $type      Type of asset: 'script' or 'style'.
     * @param bool   $condition Optional. Condition under which to dequeue assets. Default true.
     * @return void
     */
    private static function dequeue_assets(array $handles, string $type = 'script', bool $condition = true): void
    {
        if (!$condition) {
            return;
        }
        
        foreach ($handles as $handle) {
            if ($type === 'script') {
                wp_dequeue_script($handle);
            } elseif ($type === 'style') {
                wp_dequeue_style($handle);
            }
        }
    }
    
    /**
     * Checks if the current user is logged in, with result caching.
     *
     * Caches the result of is_user_logged_in() to avoid redundant function calls during a single request.
     *
     * @return bool True if the user is logged in, false otherwise.
     */
    private static function is_user_logged_in(): bool
    {
        if (self::$is_user_logged_in === null) {
            self::$is_user_logged_in = is_user_logged_in();
        }
        return self::$is_user_logged_in;
    }

    /**
     * Checks if WooCommerce is active, with result caching.
     *
     * Caches the result of class_exists('WooCommerce') to avoid redundant checks during a single request.
     *
     * @return bool True if WooCommerce is active, false otherwise.
     */
    private static function is_woocommerce_active(): bool
    {
        if (self::$is_woocommerce_active === null) {
            self::$is_woocommerce_active = class_exists('WooCommerce');
        }
        return self::$is_woocommerce_active;
    }

    /**
     * Retrieves a compiled regular expression pattern for repeated use.
     *
     * Stores and returns a regular expression pattern by key, allowing for efficient reuse and
     * potential future optimization (e.g., using pre-compiled patterns if supported).
     *
     * @param string $key     Key to identify the pattern in the cache.
     * @param string $pattern Regular expression pattern to store if not already cached.
     * @return string The regular expression pattern (currently just returns the string).
     */
    private static function get_compiled_pattern(string $key, string $pattern): string
    {
        if (!isset(self::$compiled_patterns[$key])) {
            self::$compiled_patterns[$key] = $pattern;
        }
        return self::$compiled_patterns[$key];
    }
}
