<?php

namespace ZiziCache;

/**
 * Quicklink Integration for ZiziCache
 *
 * Implements Google ChromeLabs quicklink library as complement to Speculation Rules API.
 * Provides viewport-based link prefetching/prerendering with idle time detection,
 * network condition awareness, and enhanced control over concurrent requests.
 *
 * @package ZiziCache
 * @link https://github.com/GoogleChromeLabs/quicklink Quicklink Library
 */
class QuicklinkIntegration
{
    /**
     * Initialize the Quicklink functionality
     *
     * Sets up WordPress hooks for script enqueuing and configuration output.
     * Only activates when quicklink is enabled in configuration.
     *
     * @return void
     */
    public static function init()
    {
        // Only initialize if quicklink is enabled
        if (empty(SysConfig::$config['quicklink_enabled'])) {
            return;
        }

        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_scripts']);
        
        // Output configuration to frontend
        add_action('wp_head', [__CLASS__, 'output_quicklink_config'], 98);

        // Conditional execution - check page type
        if (!empty(SysConfig::$config['quicklink_conditional'])) {
            add_filter('zizi_cache_should_apply_quicklink', [__CLASS__, 'check_conditional'], 10, 1);
        }

        // Log coexistence status with Speculation Rules
        self::check_coexistence();
    }

    /**
     * Enqueue quicklink scripts and dependencies
     *
     * Loads quicklink library and configuration wrapper based on settings.
     * Enhanced with better loading strategies inspired by Flying Press 5.0.6
     *
     * @return void
     */
    public static function enqueue_scripts()
    {
        // Conditional execution - applies a filter that can prevent loading based on page type
        if (apply_filters('zizi_cache_should_apply_quicklink', true) === false) {
            return;
        }

        $config = SysConfig::$config;
        $plugin_url = plugin_dir_url(dirname(__FILE__));
        
        // Enhanced strategy: Use different loading strategies based on user preferences
        $loading_strategy = $config['quicklink_loading_strategy'] ?? 'intersection';
        
        // Use main quicklink library for all strategies (behavior controlled by config)
        wp_enqueue_script(
            'zizi-quicklink',
            $plugin_url . 'assets/js/quicklink.min.js',
            [],
            '3.0.1',
            true
        );

        // Enhanced configuration wrapper with better error handling
        wp_enqueue_script(
            'zizi-quicklink-config',
            $plugin_url . 'assets/js/quicklink-integration.min.js',
            ['zizi-quicklink'],
            ZIZI_CACHE_VERSION,
            true
        );
    }

    /**
     * Output quicklink configuration to page head
     *
     * Generates JavaScript configuration object for quicklink based on admin settings.
     * Includes sanitization, validation, and smart defaults.
     *
     * @return void
     */
    public static function output_quicklink_config()
    {
        // Conditional execution check
        if (apply_filters('zizi_cache_should_apply_quicklink', true) === false) {
            return;
        }

        $config = SysConfig::$config;
        
        // Build quicklink configuration
        $quicklink_config = [
            'enabled' => true, // Explicitly set to true for JS
            'prerender' => self::get_prerender_setting($config),
            'prerenderAndPrefetch' => self::get_prerender_and_prefetch_setting($config),
            'delay' => self::sanitize_number($config['quicklink_delay'] ?? 0, 0, 5000),
            'timeout' => self::sanitize_number($config['quicklink_timeout'] ?? 2000, 500, 10000),
            'throttle' => self::sanitize_number($config['quicklink_throttle'] ?? 5, 1, 10),
            'limit' => self::sanitize_number($config['quicklink_limit'] ?? 20, 1, 50),
            'threshold' => self::sanitize_number($config['quicklink_threshold'] ?? 0, 0, 100) / 100,
            'priority' => !empty($config['quicklink_priority']),
            'origins' => self::sanitize_origins($config['quicklink_origins'] ?? []),
            'ignores' => self::sanitize_ignores($config['quicklink_ignores'] ?? []),
            'debug' => defined('WP_DEBUG') && WP_DEBUG,
            'idleDetection' => true, // Always use idle detection for optimal performance
            'adjustPriorityOnScroll' => true, // Adjust priority during scroll
            'enableMonitoring' => true, // Enable performance monitoring
        ];

        // Add network conditions if enabled
        if (!empty($config['quicklink_respect_data_saver'])) {
            $quicklink_config['respectDataSaver'] = true;
            $quicklink_config['respectNetworkConditions'] = true;
        }

        // Add minimum connection type
        if (!empty($config['quicklink_min_connection'])) {
            $quicklink_config['minConnectionType'] = sanitize_text_field($config['quicklink_min_connection']);
        }

        // Add ad blocking patterns if enabled
        if (!empty($config['quicklink_block_ads'])) {
            $quicklink_config['ignores'] = array_merge(
                $quicklink_config['ignores'],
                self::get_ad_blocking_patterns($config)
            );
        }

        // Add analytics guard if it's enabled for Speculation Rules
        if (!empty($config['speculative_loading_analytics_guard'])) {
            // Output analytics guard script
            self::output_analytics_guard();
        }

        // Output configuration
        echo '<script id="zizi-quicklink-config">' . "\n";
        echo 'window.ziziQuicklinkConfig = ' . json_encode($quicklink_config, JSON_UNESCAPED_SLASHES) . ';' . "\n";
        
        // Add coexistence detection
        echo 'window.ziziQuicklinkCoexistence = {' . "\n";
        echo '  speculationRulesActive: ' . (self::is_speculation_rules_active() ? 'true' : 'false') . ',' . "\n";
        echo '  mode: "' . esc_js(self::get_coexistence_mode()) . '"' . "\n";
        echo '};' . "\n";
        echo '</script>' . "\n";
    }

    /**
     * Output analytics guard script
     * 
     * Reuses the analytics guard from Speculation Rules API
     * to prevent duplicate pageviews during prerendering
     * 
     * @return void
     */
    private static function output_analytics_guard()
    {
        $config = SysConfig::$config;
        
        echo '<script id="zizi-quicklink-analytics-guard">' . "\n";
        echo 'document.addEventListener("DOMContentLoaded", function() {' . "\n";
        echo '  // Generic guard for prerendering analytics' . "\n";
        echo '  function guardAnalytics() {' . "\n";
        echo '    if (window.ziziQuicklinkConfig && window.ziziQuicklinkConfig.prerender && document.prerendering) {' . "\n";
        
        // Google Analytics
        if (!empty($config['speculative_loading_ga_id'])) {
            $ga_id = esc_js($config['speculative_loading_ga_id']);
            echo "      // Block GA during prerender\n";
            echo "      function blockGoogleAnalytics() {\n";
            echo "        if (window.ga) {\n";
            echo "          window.ga('set', 'sendHitTask', null);\n";
            echo "          console.log('[Zizi Analytics Guard] Blocked GA during prerender');\n";
            echo "        }\n";
            echo "        if (window.gtag) {\n";
            echo "          const originalGtag = window.gtag;\n";
            echo "          window.gtag = function() {\n";
            echo "            if (arguments[0] === 'config' && arguments[1] === '{$ga_id}') {\n";
            echo "              arguments[2] = arguments[2] || {};\n";
            echo "              arguments[2].send_page_view = false;\n";
            echo "            }\n";
            echo "            originalGtag.apply(null, arguments);\n";
            echo "          };\n";
            echo "          console.log('[Zizi Analytics Guard] Modified gtag during prerender');\n";
            echo "        }\n";
            echo "      }\n";
            echo "      blockGoogleAnalytics();\n";
        }
        
        // Facebook Pixel
        if (!empty($config['speculative_loading_facebook_pixel_id'])) {
            echo "      // Block Facebook Pixel during prerender\n";
            echo "      if (window.fbq) {\n";
            echo "        window._fbq_original = window.fbq;\n";
            echo "        window.fbq = function(track, event) {\n";
            echo "          console.log('[Zizi Analytics Guard] Blocked Facebook Pixel during prerender');\n";
            echo "          return undefined;\n";
            echo "        };\n";
            echo "      }\n";
        }
        
        // Add any custom guard JavaScript
        if (!empty($config['speculative_loading_custom_guard'])) {
            echo "      // Custom analytics guard code\n";
            echo "      try {\n";
            echo "        " . $config['speculative_loading_custom_guard'] . "\n";
            echo "      } catch(e) { console.error('[Zizi Analytics Guard] Custom guard error:', e); }\n";
        }
        
        echo '    }' . "\n";
        echo '  }' . "\n";
        echo '  guardAnalytics();' . "\n";
        echo '});' . "\n";
        echo '</script>' . "\n";
    }

    /**
     * Determine prerender setting based on mode
     *
     * @param array $config Plugin configuration
     * @return bool
     */
    private static function get_prerender_setting($config)
    {
        $mode = $config['quicklink_mode'] ?? 'prefetch';
        return in_array($mode, ['prerender', 'both']);
    }

    /**
     * Determine prerenderAndPrefetch setting
     *
     * @param array $config Plugin configuration
     * @return bool
     */
    private static function get_prerender_and_prefetch_setting($config)
    {
        $mode = $config['quicklink_mode'] ?? 'prefetch';
        return $mode === 'both';
    }

    /**
     * Sanitize numeric configuration values
     *
     * @param mixed $value Input value
     * @param int $min Minimum allowed value
     * @param int $max Maximum allowed value
     * @return int
     */
    private static function sanitize_number($value, $min, $max)
    {
        $value = intval($value);
        return max($min, min($max, $value));
    }

    /**
     * Sanitize and validate origins list
     *
     * @param array $origins Array of origin hostnames
     * @return array|bool
     */
    private static function sanitize_origins(array $origins)
    {
        if (empty($origins)) {
            // Empty array means same-origin only (quicklink default)
            return [parse_url(home_url(), PHP_URL_HOST)];
        }

        $sanitized = [];
        foreach ($origins as $origin) {
            $origin = sanitize_text_field(trim($origin));
            if (!empty($origin) && self::is_valid_hostname($origin)) {
                $sanitized[] = $origin;
            }
        }

        // Limit to reasonable number
        return array_slice($sanitized, 0, 10);
    }

    /**
     * Sanitize and compile ignore patterns
     *
     * @param array $ignores Array of ignore patterns
     * @return array
     */
    private static function sanitize_ignores(array $ignores)
    {
        $sanitized = [];
        
        // Add default WordPress ignores
        $default_ignores = [
            '/wp-admin/*',
            '/wp-login.php',
            '*.pdf',
            '*.zip',
            '*.doc*',
            '*.xls*',
            '/feed/*'
        ];

        foreach (array_merge($default_ignores, $ignores) as $ignore) {
            $ignore = sanitize_text_field(trim($ignore));
            if (!empty($ignore)) {
                $sanitized[] = $ignore;
            }
        }

        // Convert to JavaScript-compatible patterns
        return self::convert_ignore_patterns($sanitized);
    }

    /**
     * Convert ignore patterns to JavaScript functions/regexes
     *
     * @param array $patterns Array of pattern strings
     * @return array
     */
    private static function convert_ignore_patterns(array $patterns)
    {
        $js_patterns = [];

        foreach ($patterns as $pattern) {
            if (strpos($pattern, '*') !== false) {
                // Convert wildcard to regex
                $regex = str_replace(['*', '/'], ['.*', '\/'], $pattern);
                $js_patterns[] = "uri => /{$regex}/i.test(uri)";
            } elseif (strpos($pattern, '.') === 0) {
                // File extension
                $ext = ltrim($pattern, '.');
                $js_patterns[] = "uri => uri.toLowerCase().endsWith('.{$ext}')";
            } else {
                // Simple string contains
                $js_patterns[] = "uri => uri.includes('{$pattern}')";
            }
        }

        return $js_patterns;
    }

    /**
     * Get ad blocking patterns if enabled
     *
     * @param array $config Plugin configuration
     * @return array
     */
    private static function get_ad_blocking_patterns($config)
    {
        $patterns = [
            // Google Ads
            "uri => uri.includes('googleads.g.doubleclick.net')",
            "uri => uri.includes('googlesyndication.com')",
            "uri => uri.includes('googleadservices.com')",
            
            // Facebook Ads
            "uri => uri.includes('facebook.com/ads')",
            "uri => uri.includes('facebook.com/tr')",
            
            // Common affiliate patterns
            "uri => uri.includes('/affiliate')",
            "uri => uri.includes('/partner')",
            "uri => uri.includes('?ref=')",
            
            // Element-based detection
            "(uri, elem) => elem.hasAttribute('data-ad')",
            "(uri, elem) => elem.classList.contains('affiliate-link')",
            "(uri, elem) => elem.closest('.advertisement')"
        ];

        // Add custom ad patterns if configured
        if (!empty($config['quicklink_ad_patterns'])) {
            foreach ($config['quicklink_ad_patterns'] as $pattern) {
                $pattern = sanitize_text_field(trim($pattern));
                if (!empty($pattern)) {
                    $patterns[] = "uri => uri.includes('{$pattern}')";
                }
            }
        }

        return $patterns;
    }

    /**
     * Check if hostname is valid
     *
     * @param string $hostname Hostname to validate
     * @return bool
     */
    private static function is_valid_hostname($hostname)
    {
        return filter_var($hostname, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME) !== false;
    }

    /**
     * Check conditional execution based on page type
     *
     * @param bool $apply Default filter value
     * @return bool
     */
    public static function check_conditional($apply)
    {
        $config = SysConfig::$config;
        $page_types = isset($config['quicklink_page_types']) ? (array)$config['quicklink_page_types'] : [];

        // If nothing is set, apply everywhere
        if (empty($page_types)) {
            return $apply;
        }

        // Check various page types
        if (in_array('home', $page_types) && is_home()) {
            return true;
        }

        if (in_array('front_page', $page_types) && is_front_page()) {
            return true;
        }

        if (in_array('singular', $page_types) && is_singular()) {
            return true;
        }

        if (in_array('archive', $page_types) && is_archive()) {
            return true;
        }

        if (in_array('search', $page_types) && is_search()) {
            return true;
        }

        return false;
    }

    /**
     * Check if Speculation Rules API is currently active
     *
     * @return bool
     */
    private static function is_speculation_rules_active()
    {
        $config = SysConfig::$config;
        return !empty($config['speculative_loading_mode']) && $config['speculative_loading_mode'] !== 'off';
    }

    /**
     * Determine coexistence mode
     *
     * @return string
     */
    private static function get_coexistence_mode()
    {
        if (self::is_speculation_rules_active()) {
            return 'dual';
        }
        return 'quicklink_only';
    }

    /**
     * Check and log coexistence status
     *
     * @return void
     */
    private static function check_coexistence()
    {
        if (self::is_speculation_rules_active()) {
            $spec_mode = SysConfig::$config['speculative_loading_mode'];
            $quick_mode = SysConfig::$config['quicklink_mode'] ?? 'prefetch';
            
            if ($spec_mode === $quick_mode) {
                CacheSys::writeLog('[QuicklinkIntegration] Notice: Both systems using same mode (' . $spec_mode . '). Consider using different modes for optimal performance.');
            }
            
            // Check page type overlap
            $spec_pages = SysConfig::$config['speculative_loading_page_types'] ?? [];
            $quick_pages = SysConfig::$config['quicklink_page_types'] ?? [];
            $overlap = array_intersect($spec_pages, $quick_pages);
            
            if (!empty($overlap)) {
                CacheSys::writeLog('[QuicklinkIntegration] Notice: Page type overlap detected: ' . implode(', ', $overlap) . '. Monitor for duplicate requests.');
            }
        }
    }

    /**
     * Get default configuration values
     *
     * @return array
     */
    public static function get_default_config()
    {
        return [
            'quicklink_enabled' => false,
            'quicklink_mode' => 'prefetch',
            'quicklink_throttle' => 5,
            'quicklink_limit' => 20,
            'quicklink_delay' => 0,
            'quicklink_timeout' => 2000,
            'quicklink_threshold' => 0,
            'quicklink_priority' => false,
            'quicklink_origins' => [],
            'quicklink_ignores' => [],
            'quicklink_respect_data_saver' => true,
            'quicklink_min_connection' => '3g',
            'quicklink_block_ads' => true,
            'quicklink_ad_patterns' => [],
            'quicklink_conditional' => false,
            'quicklink_page_types' => []
        ];
    }
}
