<?php

namespace ZiziCache;

use FilesystemIterator;

/**
 * Class Purge
 *
 * Handles cache purging for ZiziCache: single URLs, all HTML pages, or the entire cache directory.
 * Provides safe recursive deletion and integration with WordPress hooks.
 *
 * @package ZiziCache
 */
class Purge
{
  /**
   * Purge a list of HTML pages by their URLs.
   *
   * @param array $urls List of URLs to purge from cache.
   * @return void
   */
  public static function purge_urls($urls)
  {
    do_action('zizi_cache_purge_urls:before', $urls);

    foreach ($urls as $url) {
      self::purge_url($url);
    }

    do_action('zizi_cache_purge_urls:after', $urls);
  }
  /**
   * Purge a single HTML page by its URL (non-recursive, only .html/.html.gz files).
   * Enhanced with path traversal protection.
   *
   * @param string $url URL to purge from cache.
   * @return void
   */
  private static function purge_url($url)
  {
    do_action('zizi_cache_purge_url:before', $url);
    
    $parsed = parse_url($url);
    if (!$parsed || !isset($parsed['host'])) {
      \ZiziCache\CacheSys::writeError("Invalid URL for purge: $url", 'Security');
      return;
    }
    
    $host = $parsed['host'];
    $path = $parsed['path'] ?? '/';
    
    // Sanitize and normalize path
    $path = wp_normalize_path(urldecode($path));
    $path = ltrim($path, '/');
    
    // Remove any remaining dangerous sequences after normalization
    if (strpos($path, '..') !== false || strpos($path, "\0") !== false) {
      \ZiziCache\CacheSys::writeError(
        "Path traversal attempt detected in normalized path: $url → $path from IP: " . 
        ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
        'Security'
      );
      return;
    }
    
    // Construct target directory safely
    $base_cache_dir = wp_normalize_path(realpath(ZIZI_CACHE_CACHE_DIR));
    if (!$base_cache_dir) {
      \ZiziCache\CacheSys::writeError("Cache directory not found: " . ZIZI_CACHE_CACHE_DIR, 'Error');
      return;
    }
    
    $target_dir = $base_cache_dir . DIRECTORY_SEPARATOR . $host;
    if (!empty($path)) {
      $target_dir .= DIRECTORY_SEPARATOR . $path;
    }
    $target_dir = wp_normalize_path($target_dir);
    
    // Critical: Verify path is within cache directory
    // First normalize the target directory path for security validation
    $normalized_target = wp_normalize_path($target_dir);
    $normalized_base = wp_normalize_path($base_cache_dir);
    
    // Security check: Ensure target path is within base cache directory
    if (strpos($normalized_target, $normalized_base) !== 0) {
      \ZiziCache\CacheSys::writeError(
        "Path traversal attempt blocked: $url → $target_dir from IP: " . 
        ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
        'Security'
      );
      return;
    }
    
    // Create directory structure if it doesn't exist (after security validation)
    $target_parent = dirname($target_dir);
    if (!is_dir($target_parent)) {
      wp_mkdir_p($target_parent);
    }
    
    // Additional realpath verification now that directory exists
    $real_target_parent = realpath($target_parent);
    if (!$real_target_parent || strpos($real_target_parent, $base_cache_dir) !== 0) {
      \ZiziCache\CacheSys::writeError(
        "Path traversal attempt blocked after directory creation: $url → $target_dir from IP: " . 
        ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
        'Security'
      );
      return;
    }
    
    // Safe file operations with additional validation
    foreach (['.html', '.html.gz'] as $ext) {
      $pattern = $target_dir . DIRECTORY_SEPARATOR . '*' . $ext;
      foreach (glob($pattern) ?: [] as $file) {
        if (is_file($file) && file_exists($file)) {
          // Double-check each file is within cache directory
          $real_file_path = realpath($file);
          if ($real_file_path && strpos($real_file_path, $base_cache_dir) === 0) {
            @unlink($file);
          } else {
            \ZiziCache\CacheSys::writeError(
              "Blocked attempt to delete file outside cache: $file", 
              'Security'
            );
          }
        }
      }
    }
    
    do_action('zizi_cache_purge_url:after', $url);
  }

  /**
   * Purge all HTML pages (recursively) and remove preload list file.
   *
   * @return void
   */
  public static function purge_pages()
  {
    do_action('zizi_cache_purge_pages:before');

    if (file_exists(ZIZI_CACHE_CACHE_DIR . '/preloader-list.txt')) {
      unlink(ZIZI_CACHE_CACHE_DIR . '/preloader-list.txt');
    }

    // Delete all HTML pages including subdirectories
    self::delete_all_pages(ZIZI_CACHE_CACHE_DIR);

    do_action('zizi_cache_purge_pages:after');
  }

  /**
   * Purge the entire cache directory (all files and subdirectories) and remove preload list file.
   *
   * @return void
   */
  public static function purge_everything()
  {
    do_action('zizi_cache_purge_everything:before');

    if (file_exists(ZIZI_CACHE_CACHE_DIR . '/preloader-list.txt')) {
      unlink(ZIZI_CACHE_CACHE_DIR . '/preloader-list.txt');
    }    // Delete all files and subdirectories
    self::delete_directory(ZIZI_CACHE_CACHE_DIR);

    // Create cache directory
    mkdir(ZIZI_CACHE_CACHE_DIR, 0755, true);

    // CRITICAL: Restore security files after cache clearing
    \ZiziCache\SysTool::ensure_cache_security();

    do_action('zizi_cache_purge_everything:after');
  }

  /**
   * Recursively delete all HTML and GZ files in a directory and its subdirectories.
   * Remove directory if empty after deletion.
   *
   * @param string $path Directory path to process.
   * @return void
   */
  private static function delete_all_pages($path)
  {
    if (!file_exists($path) || !is_dir($path)) {
      return;
    }
    $empty = true;
    try {
      $it = new \FilesystemIterator($path, \FilesystemIterator::SKIP_DOTS);
      foreach ($it as $fileinfo) {
        if ($fileinfo->isDir()) {
          self::delete_all_pages($fileinfo->getRealPath());
        } else {
          if (in_array($fileinfo->getExtension(), ['html', 'gz'])) {
            @unlink($fileinfo->getRealPath());
          } else {
            $empty = false;
          }
        }
      }
      // Smaz adresar jen pokud je prazdny
      if ($empty && iterator_count($it) === 0) {
        @rmdir($path);
      }
    } catch (\Exception $e) {
      // Directory does not exist or is not accessible, safely ignore
      return;
    }
  }

  /**
   * Recursively delete all files and subdirectories in a directory, then remove the directory itself.
   *
   * @param string $path Directory path to delete.
   * @return void
   */
  private static function delete_directory($path)
  {
    if (!file_exists($path) || !is_dir($path)) {
      return;
    }
    $it = new FilesystemIterator($path, FilesystemIterator::SKIP_DOTS);
    foreach ($it as $fileinfo) {
      if ($fileinfo->isDir()) {
        self::delete_directory($fileinfo->getRealPath());
      } else {
        @unlink($fileinfo->getRealPath());
      }
    }
    @rmdir($path);
  }
}
