<?php

namespace ZiziCache;

/**
 * PreloadSqlite - Asynchronous worker for preloading expired pages from SQLite
 * 
 * Handles background processing of cache preloading operations using SQLite
 * for storing URLs that need to be preloaded.
 * 
 * @package ZiziCache
 */
class PreloadSqlite
{
    const SQLITE_PATH = ZIZI_CACHE_CACHE_DIR . 'preload.sqlite';
    const TABLE = 'zizi_expired_cache';

    /**
     * Ensures the existence of the table for expired URLs in SQLite.
     * Also checks for the existence of the cache_path column (for older installations) and adds it if missing.
     *
     * @param \PDO $db SQLite database connection
     * @return void
     */
    private static function ensure_table_exists($db) {
        // Create table if it doesn't exist
        $db->exec('CREATE TABLE IF NOT EXISTS ' . self::TABLE . ' (
            url TEXT PRIMARY KEY,
            status TEXT NOT NULL DEFAULT "pending",
            attempts INTEGER NOT NULL DEFAULT 0,
            last_attempt INTEGER NOT NULL DEFAULT 0,
            expired_at INTEGER NOT NULL DEFAULT 0,
            cache_path TEXT
        )');

        // Then check for the existence of the cache_path column (for older installations)
        $columns = $db->query('PRAGMA table_info(' . self::TABLE . ')')->fetchAll(\PDO::FETCH_ASSOC);
        $has_cache_path = false;
        foreach ($columns as $col) {
            if ($col['name'] === 'cache_path') {
                $has_cache_path = true;
                break;
            }
        }
        if (!$has_cache_path) {
            $db->exec('ALTER TABLE ' . self::TABLE . ' ADD COLUMN cache_path TEXT');
        }
    }

    /**
     * Ensures the existence of font intelligence tables in SQLite.
     *
     * @param \PDO $db SQLite database connection
     * @return void
     */
    private static function ensure_font_intelligence_tables($db) {
        // Create font metadata table
        $db->exec('CREATE TABLE IF NOT EXISTS zizi_font_metadata (
            url TEXT PRIMARY KEY,
            font_families TEXT,
            template_type TEXT,
            theme_name TEXT,
            active_plugins TEXT,
            confidence_score REAL,
            performance_impact TEXT,
            analyzed_at INTEGER NOT NULL,
            last_updated INTEGER NOT NULL DEFAULT 0
        )');

        // Create font recommendations table
        $db->exec('CREATE TABLE IF NOT EXISTS zizi_font_recommendations (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            url TEXT NOT NULL,
            font_url TEXT NOT NULL,
            font_family TEXT,
            confidence REAL,
            reasoning TEXT,
            implementation_strategy TEXT,
            priority TEXT,
            created_at INTEGER NOT NULL,
            applied BOOLEAN DEFAULT 0
        )');

        // Create font performance metrics table
        $db->exec('CREATE TABLE IF NOT EXISTS zizi_font_performance (
            url TEXT PRIMARY KEY,
            lcp_fonts TEXT,
            cls_impact_fonts TEXT,
            load_time_metrics TEXT,
            optimization_benefits TEXT,
            measured_at INTEGER NOT NULL
        )');
    }

    /**
     * Ensures the existence of performance monitoring tables in SQLite.
     *
     * @param \PDO $db SQLite database connection
     * @return void
     */
    private static function ensure_performance_tables($db) {
        // Performance metrics table for storing various performance data
        $db->exec('CREATE TABLE IF NOT EXISTS zizi_performance_metrics (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            metric_type TEXT NOT NULL,
            metric_data TEXT NOT NULL,
            recorded_at INTEGER NOT NULL,
            page_url TEXT,
            user_agent TEXT
        )');
        
        // Index for fast lookup by type and time
        $db->exec('CREATE INDEX IF NOT EXISTS idx_performance_type_time 
                   ON zizi_performance_metrics(metric_type, recorded_at)');
        
        // Historical performance trends table
        $db->exec('CREATE TABLE IF NOT EXISTS zizi_performance_trends (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            trend_date TEXT NOT NULL,
            avg_database_queries REAL,
            avg_http_response_time REAL,
            total_slow_queries INTEGER,
            total_http_errors INTEGER,
            memory_peak REAL,
            calculated_at INTEGER NOT NULL
        )');
    }

    /**
     * Determines the batch size and interval for the preload worker based on the number of pending URLs.
     *
     * Can be overridden via wp-config.php constants ZIZI_PRELOAD_BATCH and ZIZI_PRELOAD_INTERVAL.
     *
     * @return array [int $batch_size, int $interval_seconds]
     */
    public static function get_batch_and_interval() {
        // Allow override via wp-config.php
        if (defined('ZIZI_PRELOAD_BATCH') && defined('ZIZI_PRELOAD_INTERVAL')) {
            return [constant('ZIZI_PRELOAD_BATCH'), constant('ZIZI_PRELOAD_INTERVAL')];
        }
        $db = new \PDO('sqlite:' . self::SQLITE_PATH);
        self::ensure_table_exists($db);
        $count = $db->query('SELECT COUNT(*) FROM ' . self::TABLE . ' WHERE status = "pending"')->fetchColumn();
        if ($count <= 500) {
            return [50, 60]; // 50 URLs, 1 minute
        } elseif ($count <= 5000) {
            return [20, 30]; // 20 URLs, 30 seconds
        } else {
            return [5, 10]; // 5 URLs, 10 seconds
        }
    }

    /**
     * Runs the batch worker for preloading expired URLs from SQLite.
     *
     * Loads a batch of pending URLs, attempts to preload each, updates their status, and schedules the next run.
     *
     * @return void
     */
    public static function run_worker() {
        \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Worker started - checking for expired URLs to preload');
        if (!file_exists(self::SQLITE_PATH)) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] SQLite file does not exist. Worker stopped.');
            return;
        }
        list($batch, $interval) = self::get_batch_and_interval();
        \ZiziCache\CacheSys::writeLog("[PreloadSqlite] Configuration: batch={$batch} URLs, interval={$interval}s");
        $db = new \PDO('sqlite:' . self::SQLITE_PATH);
        self::ensure_table_exists($db);
        $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        $stmt = $db->prepare('SELECT url, attempts FROM ' . self::TABLE . ' WHERE status = "pending" AND attempts < 5 ORDER BY last_attempt ASC, expired_at ASC LIMIT ?');
        $stmt->bindValue(1, $batch, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        if (!$rows) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] No pending URLs found for this batch - worker completed');
            return;
        }
        
        $success_count = 0;
        $failed_count = 0;
        
        foreach ($rows as $row) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Processing: ' . $row['url'] . ' (attempt ' . ($row['attempts']+1) . '/5)');
            $success = self::preload_url($row['url']);
            $upd = $db->prepare('UPDATE ' . self::TABLE . ' SET attempts = attempts + 1, last_attempt = ? WHERE url = ?');
            $upd->execute([time(), $row['url']]);
            if ($success) {
                // Delete successfully preloaded URL from database
                $del = $db->prepare('DELETE FROM ' . self::TABLE . ' WHERE url = ?');
                $del->execute([$row['url']]);
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] ✓ Successfully preloaded and removed: ' . $row['url']);
                $success_count++;
            } elseif ($row['attempts'] + 1 >= 5) {
                // Delete failed URL after 5 attempts
                $del = $db->prepare('DELETE FROM ' . self::TABLE . ' WHERE url = ?');
                $del->execute([$row['url']]);
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] ✗ Failed after 5 attempts, removed: ' . $row['url']);
                $failed_count++;
            } else {
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] ⚠ Preload failed, will retry: ' . $row['url']);
                $failed_count++;
            }
        }
        // Dynamic scheduling of the next run
        if (!wp_next_scheduled('zizi_cache_preload_sqlite_worker')) {
            wp_schedule_single_event(time() + $interval, 'zizi_cache_preload_sqlite_worker');
            \ZiziCache\CacheSys::writeLog("[PreloadSqlite] Next worker scheduled in {$interval}s");
        }
        \ZiziCache\CacheSys::writeLog("[PreloadSqlite] Batch completed: {$success_count} success, {$failed_count} failed/retry");
    }

    /**
     * Preloads a single URL (calls WP via HTTP).
     *
     * @param string $url URL to preload
     * @return bool True if preload was successful, false otherwise
     */
    public static function preload_url($url)
    {
        \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Starting preload of URL: ' . $url);
        // Add cache busting parameter and warmed flag, use unique User-Agent
        $url = add_query_arg(['zizi_preload' => '1', 'zizi_warmed' => '1'], $url);
        try {
            $response = wp_remote_get($url, [
                'user-agent' => SysTool::get_user_agent(),
                'timeout' => 30,
                'blocking' => true,
                'sslverify' => false,
                'httpversion' => '2.0',
                'cookies' => [],
                'redirection' => 5,
            ]);
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Successfully loaded: ' . $url);
                return true;
            } else {
                $err = is_wp_error($response) ? $response->get_error_message() : wp_remote_retrieve_response_code($response);
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Error loading ' . $url . ': ' . $err);
                return false;
            }
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Exception during preload of ' . $url . ': ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Inserts (or updates) a URL in the SQLite table zizi_expired_cache.
     *
     * @param string $url URL to insert or update
     * @param int $expired_at Expiration timestamp
     * @param string|null $cache_path Optional cache path
     * @return void
     */
    public static function insert_or_update_url($url, $expired_at = 0, $cache_path = null) {
        $db = new \PDO('sqlite:' . self::SQLITE_PATH);
        self::ensure_table_exists($db);
        $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        // Try INSERT, if it already exists, update status and expired_at
        $stmt = $db->prepare('INSERT INTO ' . self::TABLE . ' (url, status, attempts, last_attempt, expired_at, cache_path) VALUES (?, "pending", 0, 0, ?, ?)
            ON CONFLICT(url) DO UPDATE SET status = "pending", expired_at = excluded.expired_at, cache_path = excluded.cache_path, attempts = 0, last_attempt = 0');
        $stmt->execute([$url, $expired_at, $cache_path]);
    }

    /**
     * Regular cleanup of old records in SQLite (e.g. older than 7 days).
     *
     * @param int $days Number of days after which records are deleted
     * @return void
     */
    public static function cleanup_old_entries($days = 7) {
        try {
            // Ensure cache directory exists
            if (!file_exists(dirname(self::SQLITE_PATH))) {
                wp_mkdir_p(dirname(self::SQLITE_PATH));
            }
            
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
            
            self::ensure_table_exists($db);
            self::ensure_font_intelligence_tables($db);
            self::ensure_performance_tables($db);
            
            $threshold = time() - ($days * 86400);
            $stmt = $db->prepare('DELETE FROM ' . self::TABLE . ' WHERE expired_at < ?');
            $stmt->execute([$threshold]);
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Cleanup of old records in SQLite completed.');
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Cleanup failed: ' . $e->getMessage());
        }
    }

    /**
     * Registers WP-Cron event for regular cleanup.
     *
     * @return void
     */
    public static function schedule_cleanup() {
        if (!wp_next_scheduled('zizi_cache_sqlite_cleanup')) {
            wp_schedule_event(time(), 'daily', 'zizi_cache_sqlite_cleanup');
        }
    }

    /**
     * Registers WP-Cron event for the preload worker.
     *
     * @return void
     */
    public static function schedule_worker()
    {
        if (!wp_next_scheduled('zizi_cache_preload_sqlite_worker')) {
            list($batch, $interval) = self::get_batch_and_interval();
            wp_schedule_single_event(time() + $interval, 'zizi_cache_preload_sqlite_worker');
        }
    }

    /**
     * Store font intelligence data for a URL.
     *
     * @param string $url The URL being analyzed
     * @param array $font_data Font intelligence data
     * @return bool Success status
     */
    public static function store_font_intelligence_data($url, $font_data) {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
            self::ensure_font_intelligence_tables($db);

            // Store font metadata
            $stmt = $db->prepare('INSERT OR REPLACE INTO zizi_font_metadata 
                (url, font_families, template_type, theme_name, active_plugins, confidence_score, performance_impact, analyzed_at, last_updated) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)');
            
            $stmt->execute([
                $url,
                json_encode($font_data['detected_fonts'] ?? []),
                $font_data['template_type'] ?? '',
                $font_data['theme_info']['name'] ?? '',
                json_encode($font_data['active_plugins'] ?? []),
                $font_data['analysis_confidence'] ?? 0,
                $font_data['performance_context']['current_optimization_level'] ?? '',
                time(),
                time()
            ]);

            // Store recommendations
            if (!empty($font_data['recommendations'])) {
                $stmt = $db->prepare('DELETE FROM zizi_font_recommendations WHERE url = ?');
                $stmt->execute([$url]);

                $stmt = $db->prepare('INSERT INTO zizi_font_recommendations 
                    (url, font_url, font_family, confidence, reasoning, implementation_strategy, priority, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)');

                foreach ($font_data['recommendations'] as $rec) {
                    $stmt->execute([
                        $url,
                        $rec['font_url'] ?? '',
                        $rec['font_family'] ?? '',
                        $rec['confidence'] ?? 0,
                        $rec['reasoning'] ?? '',
                        $rec['implementation_strategy'] ?? '',
                        $rec['priority'] ?? 'medium',
                        time()
                    ]);
                }
            }

            // Store performance metrics
            if (!empty($font_data['optimization_benefits'])) {
                $stmt = $db->prepare('INSERT OR REPLACE INTO zizi_font_performance 
                    (url, lcp_fonts, cls_impact_fonts, load_time_metrics, optimization_benefits, measured_at) 
                    VALUES (?, ?, ?, ?, ?, ?)');
                
                $stmt->execute([
                    $url,
                    json_encode($font_data['lcp_fonts'] ?? []),
                    json_encode($font_data['cls_fonts'] ?? []),
                    json_encode($font_data['load_metrics'] ?? []),
                    json_encode($font_data['optimization_benefits'] ?? []),
                    time()
                ]);
            }

            return true;

        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Font intelligence storage failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get font intelligence data for a URL.
     *
     * @param string $url The URL to get data for
     * @return array|null Font intelligence data or null if not found
     */
    public static function get_font_intelligence_data($url) {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            self::ensure_font_intelligence_tables($db);

            // Get metadata
            $stmt = $db->prepare('SELECT * FROM zizi_font_metadata WHERE url = ?');
            $stmt->execute([$url]);
            $metadata = $stmt->fetch(\PDO::FETCH_ASSOC);

            if (!$metadata) {
                return null;
            }

            // Get recommendations
            $stmt = $db->prepare('SELECT * FROM zizi_font_recommendations WHERE url = ? ORDER BY confidence DESC');
            $stmt->execute([$url]);
            $recommendations = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            // Get performance data
            $stmt = $db->prepare('SELECT * FROM zizi_font_performance WHERE url = ?');
            $stmt->execute([$url]);
            $performance = $stmt->fetch(\PDO::FETCH_ASSOC);

            return [
                'metadata' => $metadata,
                'recommendations' => $recommendations,
                'performance' => $performance,
                'last_updated' => $metadata['last_updated']
            ];

        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Font intelligence retrieval failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get font intelligence statistics.
     *
     * @return array Statistics about font intelligence data
     */
    public static function get_font_intelligence_stats() {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            self::ensure_font_intelligence_tables($db);

            $stats = [];

            // Total analyzed URLs
            $stmt = $db->query('SELECT COUNT(*) FROM zizi_font_metadata');
            $stats['analyzed_urls'] = $stmt->fetchColumn();

            // Average confidence score
            $stmt = $db->query('SELECT AVG(confidence_score) FROM zizi_font_metadata');
            $stats['avg_confidence'] = round($stmt->fetchColumn(), 2);

            // Total recommendations
            $stmt = $db->query('SELECT COUNT(*) FROM zizi_font_recommendations');
            $stats['total_recommendations'] = $stmt->fetchColumn();

            // Applied recommendations
            $stmt = $db->query('SELECT COUNT(*) FROM zizi_font_recommendations WHERE applied = 1');
            $stats['applied_recommendations'] = $stmt->fetchColumn();

            // Most common template types
            $stmt = $db->query('SELECT template_type, COUNT(*) as count FROM zizi_font_metadata GROUP BY template_type ORDER BY count DESC LIMIT 5');
            $stats['template_types'] = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            // Most recommended fonts
            $stmt = $db->query('SELECT font_family, COUNT(*) as count FROM zizi_font_recommendations GROUP BY font_family ORDER BY count DESC LIMIT 10');
            $stats['popular_fonts'] = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            return $stats;

        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Font intelligence stats failed: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Store performance metrics in SQLite
     * 
     * @param string $type Metric type (database, http, cache, etc.)
     * @param array $data Performance data to store
     * @param string|null $url Optional page URL
     * @return void
     */
    public static function storePerformanceMetrics($type, $data, $url = null): void
    {
        if (!file_exists(dirname(self::SQLITE_PATH))) {
            wp_mkdir_p(dirname(self::SQLITE_PATH));
        }
        
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
            
            self::ensure_performance_tables($db);
            
            $stmt = $db->prepare('INSERT INTO zizi_performance_metrics 
                                 (metric_type, metric_data, recorded_at, page_url, user_agent) 
                                 VALUES (?, ?, ?, ?, ?)');
            
            $stmt->execute([
                $type,
                json_encode($data),
                time(),
                $url ?? ($_SERVER['REQUEST_URI'] ?? ''),
                $_SERVER['HTTP_USER_AGENT'] ?? ''
            ]);
            
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Performance Storage Error: ' . $e->getMessage());
        }
    }

    /**
     * Get performance history from SQLite
     * 
     * @param string|null $type Optional metric type filter
     * @param int $limit Maximum number of records to return
     * @param int $days Number of days to look back
     * @return array Historical performance data
     */
    public static function getPerformanceHistory($type = null, $limit = 100, $days = 7): array
    {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            self::ensure_performance_tables($db);
            
            $since = time() - ($days * 24 * 60 * 60);
            
            if ($type) {
                $stmt = $db->prepare('SELECT * FROM zizi_performance_metrics 
                                     WHERE metric_type = ? AND recorded_at > ? 
                                     ORDER BY recorded_at DESC LIMIT ?');
                $stmt->execute([$type, $since, $limit]);
            } else {
                $stmt = $db->prepare('SELECT * FROM zizi_performance_metrics 
                                     WHERE recorded_at > ? 
                                     ORDER BY recorded_at DESC LIMIT ?');
                $stmt->execute([$since, $limit]);
            }
            
            $results = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            // Decode JSON data
            foreach ($results as &$result) {
                $result['metric_data'] = json_decode($result['metric_data'], true);
            }
            
            return $results;
            
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Performance History Error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get performance statistics and trends
     * 
     * @return array Performance statistics
     */
    public static function getPerformanceStats(): array
    {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            self::ensure_performance_tables($db);
            
            $stats = [];
            
            // Total metrics recorded
            $stmt = $db->query('SELECT COUNT(*) FROM zizi_performance_metrics');
            $stats['total_metrics'] = $stmt->fetchColumn();
            
            // Metrics by type
            $stmt = $db->query('SELECT metric_type, COUNT(*) as count 
                               FROM zizi_performance_metrics 
                               GROUP BY metric_type 
                               ORDER BY count DESC');
            $stats['metrics_by_type'] = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            // Recent metrics (last 24 hours)
            $since = time() - (24 * 60 * 60);
            $stmt = $db->prepare('SELECT COUNT(*) FROM zizi_performance_metrics WHERE recorded_at > ?');
            $stmt->execute([$since]);
            $stats['recent_metrics'] = $stmt->fetchColumn();
            
            return $stats;
            
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Performance Stats Error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Clean up old performance metrics to prevent database bloat
     * 
     * @param int $days Keep metrics from last N days (default: 30)
     * @return void
     */
    public static function cleanupOldPerformanceMetrics($days = 30): void
    {
        try {
            $db = new \PDO('sqlite:' . self::SQLITE_PATH);
            self::ensure_performance_tables($db);
            
            $cutoff = time() - ($days * 24 * 60 * 60);
            
            $stmt = $db->prepare('DELETE FROM zizi_performance_metrics WHERE recorded_at < ?');
            $deleted = $stmt->execute([$cutoff]);
            
            if ($deleted) {
                \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Cleaned up old performance metrics older than ' . $days . ' days');
            }
            
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[PreloadSqlite] Performance Cleanup Error: ' . $e->getMessage());
        }
    }
}

// Register WP-Cron hook for the preload worker
add_action('zizi_cache_preload_sqlite_worker', ['\\ZiziCache\\PreloadSqlite', 'run_worker']);
// Activate the scheduler when the plugin is loaded
add_action('init', ['\\ZiziCache\\PreloadSqlite', 'schedule_worker']);
// Hook for running cleanup
add_action('zizi_cache_sqlite_cleanup', ['\\ZiziCache\\PreloadSqlite', 'cleanup_old_entries']);
// Schedule cleanup on plugin activation or load
\ZiziCache\PreloadSqlite::schedule_cleanup();
