<?php

namespace ZiziCache;

/**
 * Class Preload
 *
 * Handles batch preloading and caching of WordPress pages. Manages a queue of URLs
 * which are gradually loaded and cached. Supports scheduling further preloads via WP cron,
 * queue management, and extension via filters.
 *
 * @package ZiziCache
 */
class Preload
{
  /**
   * Path to the file holding the queue of URLs for preload.
   * @var string
   */
  private static $preload_file = ZIZI_CACHE_CACHE_DIR . 'preloader-list.txt';

  /**
   * Adds an array of URLs to the preload queue, removes duplicates, and starts batch processing.
   *
   * @param array $urls List of URLs to preload
   * @return void
   */
  public static function preload_urls($urls)
  {
    \ZiziCache\CacheSys::writeLog('[Preload] Starting batch preload of ' . count($urls) . ' URLs.');
    $cache_dir = dirname(self::$preload_file);
    if (!is_dir($cache_dir)) {
      if (!@mkdir($cache_dir, 0755, true)) {
        \ZiziCache\CacheSys::writeLog('[Preload] ERROR: Failed to create cache directory: ' . $cache_dir);
        return;
      }
      \ZiziCache\CacheSys::writeLog('[Preload] Created cache directory: ' . $cache_dir);
      
      // Ensure security files are created for new cache directory
      \ZiziCache\SysTool::ensure_cache_security();
    }
    // Load existing URLs from the queue
    $existing = file_exists(self::$preload_file)
      ? file(self::$preload_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES)
      : [];
    // Merge and remove duplicates
    $all_urls = array_unique(array_merge($existing, array_map('trim', $urls)));
    // Write back to file
    if (false === @file_put_contents(self::$preload_file, implode("\n", $all_urls) . PHP_EOL)) {
      \ZiziCache\CacheSys::writeLog('[Preload] ERROR: Failed to write URLs to preload file.');
      return;
    }
    self::preload_available();
  }

  /**
   * Starts batch cache preloading. If not enabled in config and $force=false, the function exits.
   *
   * @param bool $force Force preload even if disabled in config
   * @return void
   */
  public static function preload_cache($force = false)
  {
    \ZiziCache\CacheSys::writeLog('[Preload] Starting preload_cache (force=' . ($force ? 'true' : 'false') . ')');
    // Use force = true to force preload cache bypassing the config setting
    if ($force === false && !SysConfig::$config['cache_preload']) {
      \ZiziCache\CacheSys::writeLog('[Preload] Preload cache is not enabled in config.');
      return;
    }

    // Fetch URLs to preload
    $urls = self::get_preload_urls();
    \ZiziCache\CacheSys::writeLog('[Preload] Fetched ' . count($urls) . ' URLs to preload.');

    // Ensure cache directory exists before writing file
    $cache_dir = dirname(self::$preload_file);
    if (!is_dir($cache_dir)) {
      \ZiziCache\CacheSys::writeLog('[Preload] Creating cache directory: ' . $cache_dir);
      mkdir($cache_dir, 0755, true);
      
      // Ensure security files are created for new cache directory
      \ZiziCache\SysTool::ensure_cache_security();
    }
    
    // Write URLs to file
    file_put_contents(self::$preload_file, implode("\n", $urls));

    // Preload first URL
    self::preload_available();
  }

  /**
   * Processes a batch of URLs from the queue and schedules the next batch if more URLs remain.
   *
   * @param int $batch_size Number of URLs in one batch (default 5)
   * @return void
   */
  public static function preload_available($batch_size = 5)
  {
    if (!file_exists(self::$preload_file) || !filesize(self::$preload_file)) {
      \ZiziCache\CacheSys::writeLog('[Preload] No URLs to preload.');
      return;
    }
    $urls = file(self::$preload_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$urls) {
      \ZiziCache\CacheSys::writeLog('[Preload] Preload file is empty.');
      @file_put_contents(self::$preload_file, '');
      return;
    }
    $urls = array_unique(array_map('trim', $urls));
    $to_process = array_splice($urls, 0, $batch_size);
    // Write remaining URLs back to file
    if (false === @file_put_contents(self::$preload_file, implode("\n", $urls) . PHP_EOL)) {
      \ZiziCache\CacheSys::writeLog('[Preload] ERROR: Failed to write remaining URLs to preload file.');
      return;
    }
    \ZiziCache\CacheSys::writeLog('[Preload] Preloading batch: ' . implode(', ', $to_process) . ' (remaining ' . count($urls) . ')');
    foreach ($to_process as $url) {
      self::preload_url($url);
    }
    if (!empty($urls)) {
      wp_schedule_single_event(time() + 1, 'zizi_cache_preload_next');
    } else {
      \ZiziCache\CacheSys::writeLog('[Preload] All URLs have been processed.');
    }
  }

  /**
   * Preloads a single URL and saves it to cache if the content is HTML and cacheable.
   *
   * @param string $url URL to preload
   * @return void
   */
  public static function preload_url($url)
  {
    $url = trim($url);
    if (empty($url)) {
      \ZiziCache\CacheSys::writeLog('[Preload] Missing URL for preload.');
      return;
    }
    \ZiziCache\CacheSys::writeLog('[Preload] Starting preload of URL: ' . $url);
    $url = add_query_arg('zizi_preload', '1', $url);
    try {
      $response = wp_remote_get($url, [
        'user-agent' => SysTool::$user_agent,
        'timeout' => 30,
        'blocking' => true,
        'sslverify' => false,
        'httpversion' => '2.0',
        'cookies' => [],
        'redirection' => 3,
      ]);
      if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
        $content = wp_remote_retrieve_body($response);
        if (preg_match('/<!DOCTYPE\s*html\b[^>]*>/i', $content)) {
          if (CacheSys::is_cacheable($content)) {
            $parsed_url = parse_url($url);
            $host = $_SERVER['HTTP_HOST'] ?? (isset($parsed_url['host']) ? $parsed_url['host'] : '');
            $path = isset($parsed_url['path']) ? $parsed_url['path'] : '/';
            $path = urldecode($path);
            $cache_file_path = rtrim(ZIZI_CACHE_CACHE_DIR, '/\\') . "/$host$path/";
            if (!is_dir($cache_file_path) && !@mkdir($cache_file_path, 0755, true)) {
              \ZiziCache\CacheSys::writeLog('[Preload] ERROR: Failed to create cache dir: ' . $cache_file_path);
              return;
            }
            $cache_file_name = 'index.html.gz';
            if (false === @file_put_contents($cache_file_path . $cache_file_name, gzencode($content, 9))) {
              \ZiziCache\CacheSys::writeLog('[Preload] ERROR: Failed to write cache file: ' . $cache_file_path . $cache_file_name);
              return;
            }
            \ZiziCache\CacheSys::writeLog('[Preload] Successfully loaded and saved: ' . $url);
          } else {
            \ZiziCache\CacheSys::writeLog('[Preload] Content is not cacheable: ' . $url);
          }
        } else {
          \ZiziCache\CacheSys::writeLog('[Preload] Not an HTML document: ' . $url);
        }
      } else {
        $err = is_wp_error($response) ? $response->get_error_message() : wp_remote_retrieve_response_code($response);
        \ZiziCache\CacheSys::writeLog('[Preload] Error loading ' . $url . ': ' . $err);
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeLog('[Preload] Exception during preload of ' . $url . ': ' . $e->getMessage());
    }
  }

  /**
   * Returns an array of all URLs that should be preloaded (homepage, posts, taxonomies, authors, ...).
   * Results are cached in a transient and can be extended using the 'zizi_cache_preload_urls' filter.
   *
   * @return array List of URLs to preload
   */
  public static function get_preload_urls()
  {
    // Get from transient if available
    $urls = get_transient('zizi_cache_preload_urls');
    if ($urls) {
      return $urls;
    }

    $urls = [];

    // Add homepage
    $urls[] = home_url();

    // Suspend cache addition to prevent cache writes (memory usage)
    wp_suspend_cache_addition(true);

    // Fetch post type URLs in batches
    $post_types = get_post_types(['public' => true, 'exclude_from_search' => false]);
    $paged = 1;

    do {
      $query = new \WP_Query([
        'post_status' => 'publish',
        'has_password' => false,
        'ignore_sticky_posts' => true,
        'no_found_rows' => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
        'order' => 'DESC',
        'orderby' => 'date',
        'post_type' => $post_types,
        'posts_per_page' => 10000, // Fetch 10k posts at a time
        'paged' => $paged,
        'fields' => 'ids', // Only get post IDs
      ]);

      foreach ($query->posts as $post_id) {
        $urls[] = get_permalink($post_id);
      }
      $paged++;
    } while ($query->have_posts());

    // Fetch taxonomy URLs
    $taxonomies = get_taxonomies(['public' => true, 'rewrite' => true]);
    foreach ($taxonomies as $taxonomy) {
      $query_args = [
        'hide_empty' => true,
        'hierarchical' => false,
        'update_term_meta_cache' => false,
        'taxonomy' => $taxonomy,
      ];

      $terms = get_terms($query_args);

      foreach ($terms as $term) {
        $urls[] = get_term_link($term, $taxonomy);
      }
    }

    // Fetch author URLs
    $user_ids = get_users([
      'role' => 'author',
      'count_total' => false,
      'fields' => 'ID',
    ]);
    foreach ($user_ids as $user_id) {
      $urls[] = get_author_posts_url($user_id);
    }

    // Resume cache addition
    wp_suspend_cache_addition(false);

    // Add additional URLs to preload via filter
    $urls = apply_filters('zizi_cache_preload_urls', $urls);

    // Set transient
    set_transient('zizi_cache_preload_urls', $urls, 60 * 60 * 24);

    return $urls;
  }
}
