<?php
/**
 * Plugin Manager handles all third-party plugin integrations.
 *
 * This central class is responsible for:
 * - Registering all available integrations
 * - Initializing only active integrations to optimize performance
 * - Providing an API to get information about active integrations
 *
 * @package ZiziCache\Plugins
 * @author ZiziCache Team
 * @since 0.2.0
 */
namespace ZiziCache\Plugins;

class PluginManager {
    /**
     * Collection of registered integration class names
     * 
     * @var array<string> List of fully qualified class names
     */
    private static $integrations = [];

    /**
     * Registers a new integration class
     *
     * This method prevents duplicate registrations for better performance.
     * 
     * @param string $className Fully qualified class name of the integration
     * @return void
     */
    public static function register(string $className): void {
        if (!in_array($className, self::$integrations, true)) {
            self::$integrations[] = $className;
        }
    }

    /**
     * Initializes all registered integrations that are available
     *
     * Only initializes integrations where isAvailable() returns true
     * to optimize performance and avoid unnecessary processing.
     * 
     * @return void
     */
    public static function initAll(): void {
        foreach (self::$integrations as $className) {
            if ($className::isAvailable()) {
                $className::init();
            }
        }
    }

    /**
     * Returns a list of active integrations
     *
     * Used for debugging and information purposes.
     * 
     * @return array<string> Array of active integration class names
     */
    public static function getActiveIntegrations(): array {
        return array_filter(self::$integrations, function($className) {
            return $className::isAvailable();
        });
    }
}
