<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Core\CacheSys;
use ZiziCache\Purge;
use ZiziCache\Preload;
use ZiziCache\AutomaticallyClear;

/**
 * WooCommerce E-commerce Integration
 * 
 * Enterprise-grade integration for WooCommerce plugin with comprehensive
 * cache management, product-aware purging, and enhanced e-commerce features.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @since 2.0.0
 * @version 2.0.0
 */
class WooCommerce extends PluginBase
{
    /**
     * Plugin configuration
     * @var array
     */
    private static $config = [
        'cache_timeout' => 600,
        'max_retries' => 3,
        'enable_logging' => true,
        'validate_products' => true,
        'include_variations' => true,
        'cache_product_attributes' => true
    ];

    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'init_time' => 0,
        'purge_operations' => 0,
        'urls_processed' => 0,
        'attributes_cached' => 0,
        'error_count' => 0,
        'last_operation' => null
    ];

    /**
     * Check WooCommerce plugin availability with comprehensive validation
     * 
     * @return bool True if WooCommerce is available and properly configured
     */
    protected static function checkAvailability(): bool
    {
        $start_time = microtime(true);
        
        try {
            // Core WooCommerce dependencies check
            $dependencies = [
                'WooCommerce' => class_exists('WooCommerce'),
                'wc_get_page_id' => function_exists('wc_get_page_id'),
                'wc_get_attribute_taxonomies' => function_exists('wc_get_attribute_taxonomies'),
                'is_cart' => function_exists('is_cart'),
                'is_checkout' => function_exists('is_checkout'),
                'is_account_page' => function_exists('is_account_page')
            ];
            
            $missing_dependencies = array_filter($dependencies, function($available) {
                return !$available;
            });
            
            if (!empty($missing_dependencies)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WooCommerce integration dependencies missing: ' . implode(', ', array_keys($missing_dependencies)));
                }
                return false;
            }
            
            // WooCommerce version compatibility check
            if (class_exists('WooCommerce') && defined('WC_VERSION')) {
                if (version_compare(WC_VERSION, '5.0.0', '<')) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WooCommerce version ' . WC_VERSION . ' is not supported. Minimum version: 5.0.0');
                    }
                    return false;
                }
            }
            
            // WooCommerce pages validation
            if (function_exists('wc_get_page_id')) {
                try {
                    $required_pages = ['shop', 'cart', 'checkout', 'myaccount'];
                    foreach ($required_pages as $page) {
                        $page_id = wc_get_page_id($page);
                        if ($page_id <= 0) {
                            if (self::$config['enable_logging']) {
                                CacheSys::writeError("WooCommerce {$page} page not configured properly (ID: {$page_id})");
                            }
                        }
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WooCommerce pages validation failed: ' . $e->getMessage());
                    }
                    return false;
                }
            }
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce integration successfully validated in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
            return true;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce availability check failed: ' . $e->getMessage());
            }
            return false;
        } catch (\Error $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce availability check error: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Initialize WooCommerce integration with comprehensive hooks and monitoring
     * 
     * @return void
     */
    public function init(): void
    {
        $start_time = microtime(true);
        
        try {
            // Core WooCommerce cache integration hooks
            add_filter('zizi_cache_is_cacheable', [$this, 'isCacheable'], 10, 1);
            add_filter('zizi_cache_cache_include_queries', [$this, 'cacheIncludeQueries'], 10, 1);
            
            // Product stock and update hooks
            add_action('woocommerce_product_set_stock', [$this, 'purgeProduct'], 10, 1);
            add_action('woocommerce_variation_set_stock', [$this, 'purgeProduct'], 10, 1);
            add_action('woocommerce_rest_insert_product_object', [$this, 'purgeProduct'], 10, 1);
            
            // Additional WooCommerce-specific hooks for comprehensive cache management
            add_action('woocommerce_update_product', [$this, 'purgeProduct'], 10, 1);
            add_action('woocommerce_delete_product', [$this, 'purgeProduct'], 10, 1);
            add_action('woocommerce_product_duplicate', [$this, 'handleProductDuplicate'], 10, 2);
            
            // Order hooks for cart and checkout cache coordination
            add_action('woocommerce_new_order', [$this, 'handleNewOrder'], 10, 1);
            add_action('woocommerce_order_status_changed', [$this, 'handleOrderStatusChange'], 10, 3);
            
            // Admin integration hooks
            add_action('admin_init', [$this, 'setupAdminHooks']);
            add_action('wp_ajax_woocommerce_cache_purge', [$this, 'handleAjaxCachePurge']);
            
            // Performance monitoring hooks
            add_action('shutdown', [$this, 'logPerformanceMetrics']);
            
            // Security hooks
            add_action('init', [$this, 'validateWooCommerceParameters'], 1);
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'init';
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce integration initialized successfully in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce integration initialization failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Determine if current page is cacheable with comprehensive WooCommerce validation
     * 
     * @param bool $is_cacheable Current cacheable status
     * @return bool Modified cacheable status
     */
    public function isCacheable($is_cacheable): bool
    {
        $start_time = microtime(true);
        
        try {
            // Validate input
            if (!is_bool($is_cacheable)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid cacheable status provided: ' . gettype($is_cacheable));
                }
                return false;
            }
            
            // Check WooCommerce availability
            if (!class_exists('WooCommerce')) {
                return $is_cacheable;
            }
            
            // Validate required WooCommerce functions
            $required_functions = ['is_cart', 'is_checkout', 'is_account_page'];
            foreach ($required_functions as $function) {
                if (!function_exists($function)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError("Required WooCommerce function {$function} not available");
                    }
                    return $is_cacheable;
                }
            }
            
            // Check WooCommerce-specific uncacheable pages
            try {
                $uncacheable_conditions = [
                    'cart' => is_cart(),
                    'checkout' => is_checkout(), 
                    'account' => is_account_page()
                ];
                
                foreach ($uncacheable_conditions as $page => $condition) {
                    if ($condition) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("WooCommerce {$page} page detected - excluding from cache");
                        }
                        return false;
                    }
                }
                
                // Additional enterprise checks for uncacheable scenarios
                if (is_admin() && isset($_GET['page']) && strpos($_GET['page'], 'wc-') === 0) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WooCommerce admin page detected - excluding from cache');
                    }
                    return false;
                }
                
                // Check for WooCommerce AJAX requests
                if (defined('DOING_AJAX') && DOING_AJAX && isset($_POST['action']) && strpos($_POST['action'], 'woocommerce') === 0) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WooCommerce AJAX request detected - excluding from cache');
                    }
                    return false;
                }
                
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WooCommerce cacheable check failed: ' . $e->getMessage());
                }
                return $is_cacheable;
            }
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce cacheable check completed in ' . round($execution_time * 1000, 2) . 'ms');
            }
            
            return $is_cacheable;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce cacheable determination failed: ' . $e->getMessage());
            }
            return $is_cacheable;
        }
    }

    /**
     * Purge product cache with comprehensive validation and error handling
     * Enhanced implementation following competitor pattern but with enterprise features
     * 
     * @param mixed $product Product object or ID
     * @return void
     */
    public function purgeProduct($product): void
    {
        $start_time = microtime(true);
        
        try {
            // Validate and normalize product input
            $product_id = null;
            $product_object = null;
            
            if (is_numeric($product)) {
                $product_id = (int) $product;
                if (function_exists('wc_get_product')) {
                    try {
                        $product_object = wc_get_product($product_id);
                    } catch (\Exception $e) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Failed to get product object for ID {$product_id}: " . $e->getMessage());
                        }
                        return;
                    }
                }
            } elseif (is_object($product) && method_exists($product, 'get_id')) {
                $product_object = $product;
                try {
                    $product_id = $product->get_id();
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Failed to get product ID from object: ' . $e->getMessage());
                    }
                    return;
                }
            } else {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid product provided for purging: ' . gettype($product));
                }
                return;
            }
            
            if (empty($product_id) || $product_id <= 0) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid product ID for purging: ' . $product_id);
                }
                return;
            }
            
            $urls_to_purge = [];
            
            // Add product URL (following competitor pattern)
            try {
                $product_url = get_permalink($product_id);
                if (!empty($product_url) && $product_url !== false) {
                    $urls_to_purge[] = $product_url;
                } else {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError("Failed to get permalink for product ID {$product_id}");
                    }
                }
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("Error getting product permalink: " . $e->getMessage());
                }
            }
            
            // Add shop page URL (following competitor pattern)
            if (function_exists('wc_get_page_id')) {
                try {
                    $shop_page_id = wc_get_page_id('shop');
                    if ($shop_page_id > 0) {
                        $shop_url = get_permalink($shop_page_id);
                        if (!empty($shop_url) && $shop_url !== false) {
                            $urls_to_purge[] = $shop_url;
                        }
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Error getting shop page URL: ' . $e->getMessage());
                    }
                }
            }
            
            // Add taxonomy URLs (following competitor pattern)
            if (class_exists('ZiziCache\AutomaticallyClear')) {
                try {
                    $taxonomy_urls = AutomaticallyClear::get_post_taxonomy_urls($product_id);
                    if (is_array($taxonomy_urls)) {
                        $urls_to_purge = [...$urls_to_purge, ...$taxonomy_urls];
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Error getting taxonomy URLs: ' . $e->getMessage());
                    }
                }
            }
            
            // Enterprise enhancement: Add related product URLs
            if (self::$config['include_variations'] && $product_object && method_exists($product_object, 'get_children')) {
                try {
                    $variation_ids = $product_object->get_children();
                    foreach ($variation_ids as $variation_id) {
                        $variation_url = get_permalink($variation_id);
                        if (!empty($variation_url) && $variation_url !== false) {
                            $urls_to_purge[] = $variation_url;
                        }
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Error getting product variations: ' . $e->getMessage());
                    }
                }
            }
            
            // Filter and validate URLs
            $urls_to_purge = array_filter($urls_to_purge, function($url) {
                return !empty($url) && is_string($url) && filter_var($url, FILTER_VALIDATE_URL);
            });
            
            $urls_to_purge = array_unique($urls_to_purge);
            
            if (!empty($urls_to_purge)) {
                // Purge URLs with error handling
                try {
                    if (class_exists('ZiziCache\Purge')) {
                        Purge::purge_urls($urls_to_purge);
                        self::$metrics['purge_operations']++;
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Cache purge failed: ' . $e->getMessage());
                    }
                }
                
                // Preload URLs with error handling (following competitor pattern)
                try {
                    if (class_exists('ZiziCache\Preload')) {
                        Preload::preload_urls($urls_to_purge);
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Cache preload failed: ' . $e->getMessage());
                    }
                }
                
                self::$metrics['urls_processed'] += count($urls_to_purge);
            }
            
            self::$metrics['last_operation'] = 'product_purge';
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                $url_count = count($urls_to_purge);
                CacheSys::writeError("Product {$product_id} cache purged: {$url_count} URLs in " . round($execution_time * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Product purge process failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Include WooCommerce-specific query parameters for cache
     * Enhanced implementation following competitor pattern with enterprise features
     * 
     * @param array $queries Existing query parameters
     * @return array Modified query parameters with WooCommerce attributes
     */
    public function cacheIncludeQueries($queries): array
    {
        $start_time = microtime(true);
        
        try {
            // Validate input
            if (!is_array($queries)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid queries array provided: ' . gettype($queries));
                }
                return is_array($queries) ? $queries : [];
            }
            
            // Check WooCommerce availability
            if (!class_exists('WooCommerce') || !function_exists('wc_get_attribute_taxonomies')) {
                return $queries;
            }
            
            $attribute_filters = [];
            
            // Get all product attributes (following Flying Press competitor pattern)
            try {
                $product_attributes = wc_get_attribute_taxonomies();
                
                if (empty($product_attributes) || !is_array($product_attributes)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('No WooCommerce product attributes found');
                    }
                    return $queries;
                }
                
                // Build the available query parameters (competitor pattern)
                foreach ($product_attributes as $product_attribute) {
                    try {
                        if (is_object($product_attribute) && isset($product_attribute->attribute_name)) {
                            $attribute_name = sanitize_text_field($product_attribute->attribute_name);
                            if (!empty($attribute_name)) {
                                $attribute_filters[] = 'filter_' . $attribute_name;
                            }
                        }
                    } catch (\Exception $e) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError('Error processing product attribute: ' . $e->getMessage());
                        }
                        continue;
                    }
                }
                
                // Enterprise enhancement: Add additional WooCommerce query parameters
                if (self::$config['cache_product_attributes']) {
                    $additional_params = [
                        'orderby',
                        'order', 
                        'min_price',
                        'max_price',
                        'product_cat',
                        'product_tag',
                        'rating_filter',
                        'on_sale',
                        'in_stock'
                    ];
                    
                    $attribute_filters = [...$attribute_filters, ...$additional_params];
                }
                
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Failed to get WooCommerce product attributes: ' . $e->getMessage());
                }
                return $queries;
            }
            
            // Append to existing queries (following competitor pattern)
            $queries = [...$queries, ...$attribute_filters];
            
            // Remove duplicates and validate
            $queries = array_unique(array_filter($queries, function($query) {
                return is_string($query) && !empty($query);
            }));
            
            self::$metrics['attributes_cached'] = count($attribute_filters);
            self::$metrics['last_operation'] = 'query_inclusion';
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                $total_queries = count($queries);
                $new_attributes = count($attribute_filters);
                CacheSys::writeError("WooCommerce query parameters included: {$new_attributes} attributes, {$total_queries} total queries in " . round($execution_time * 1000, 2) . 'ms');
            }
            
            return $queries;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce query inclusion failed: ' . $e->getMessage());
            }
            return $queries;
        }
    }

    /**
     * Handle product duplication with cache coordination
     * 
     * @param int $duplicate_id New product ID
     * @param object $original_product Original product object
     * @return void
     */
    public function handleProductDuplicate($duplicate_id, $original_product): void
    {
        try {
            if (!empty($duplicate_id) && is_numeric($duplicate_id)) {
                $this->purgeProduct($duplicate_id);
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("Product duplication handled for new product ID: {$duplicate_id}");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Product duplication handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Handle new order creation with cache considerations
     * 
     * @param int $order_id New order ID
     * @return void
     */
    public function handleNewOrder($order_id): void
    {
        try {
            if (!empty($order_id) && is_numeric($order_id) && function_exists('wc_get_order')) {
                $order = wc_get_order($order_id);
                
                if ($order && method_exists($order, 'get_items')) {
                    foreach ($order->get_items() as $item) {
                        if (method_exists($item, 'get_product_id')) {
                            $product_id = $item->get_product_id();
                            if (!empty($product_id)) {
                                $this->purgeProduct($product_id);
                            }
                        }
                    }
                }
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("New order cache handling completed for order ID: {$order_id}");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('New order handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Handle order status changes with cache coordination
     * 
     * @param int $order_id Order ID
     * @param string $old_status Previous status
     * @param string $new_status New status
     * @return void
     */
    public function handleOrderStatusChange($order_id, $old_status, $new_status): void
    {
        try {
            // Only handle significant status changes that might affect product availability
            $significant_statuses = ['completed', 'cancelled', 'refunded', 'failed'];
            
            if (in_array($new_status, $significant_statuses) || in_array($old_status, $significant_statuses)) {
                $this->handleNewOrder($order_id);
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("Order status change handled: {$old_status} → {$new_status} for order {$order_id}");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Order status change handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Setup admin interface hooks
     * 
     * @return void
     */
    public function setupAdminHooks(): void
    {
        try {
            if (!is_admin()) {
                return;
            }
            
            add_action('admin_menu', [$this, 'addAdminMenuItems']);
            add_action('admin_notices', [$this, 'showAdminNotices']);
            add_filter('plugin_action_links', [$this, 'addPluginActionLinks'], 10, 2);
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce admin hooks initialized');
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Admin hooks setup failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Log performance metrics for monitoring
     * 
     * @return void
     */
    public function logPerformanceMetrics(): void
    {
        try {
            if (!self::$config['enable_logging']) {
                return;
            }
            
            $metrics_summary = sprintf(
                'WooCommerce Performance Metrics - Init: %sms, Purges: %d, URLs: %d, Attributes: %d, Errors: %d, Last: %s',
                round(self::$metrics['init_time'] * 1000, 2),
                self::$metrics['purge_operations'],
                self::$metrics['urls_processed'],
                self::$metrics['attributes_cached'],
                self::$metrics['error_count'],
                self::$metrics['last_operation'] ?: 'none'
            );
            
            CacheSys::writeError($metrics_summary);
            
        } catch (\Exception $e) {
            // Silent fail for logging to prevent infinite loops
        }
    }

    /**
     * Validate WooCommerce parameters for security
     * 
     * @return void
     */
    public function validateWooCommerceParameters(): void
    {
        try {
            // Validate WooCommerce-specific parameters
            $wc_params = ['add-to-cart', 'remove_item', 'quantity', 'variation_id', 'product_id'];
            
            foreach ($wc_params as $param) {
                if (isset($_GET[$param])) {
                    $value = sanitize_text_field($_GET[$param]);
                    
                    // Validate numeric parameters
                    if (in_array($param, ['quantity', 'variation_id', 'product_id', 'add-to-cart']) && !is_numeric($value)) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Invalid WooCommerce parameter {$param} detected: {$value}");
                        }
                        unset($_GET[$param]);
                    }
                }
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WooCommerce parameter validation failed: ' . $e->getMessage());
            }
        }
    }
}
