<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Core\CacheSys;

/**
 * WeGlot Multilingual Integration
 * 
 * Enterprise-grade integration for WeGlot multilingual plugin with 
 * language-aware cache management, comprehensive error handling, and security features.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @since 2.0.0
 * @version 2.0.0
 */
class WeGlot extends PluginBase
{
    /**
     * Plugin configuration
     * @var array
     */
    private static $config = [
        'cache_timeout' => 300,
        'max_retries' => 3,
        'enable_logging' => true,
        'url_validation' => true
    ];

    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'init_time' => 0,
        'url_generation_time' => 0,
        'languages_processed' => 0,
        'error_count' => 0,
        'last_operation' => null
    ];

    /**
     * Check WeGlot plugin availability with comprehensive validation
     * 
     * @return bool True if WeGlot is available and properly configured
     */
    protected static function checkAvailability(): bool
    {
        $start_time = microtime(true);
        
        try {
            // Core WeGlot dependencies check
            $dependencies = [
                'WEGLOT_VERSION' => defined('WEGLOT_VERSION'),
                'weglot_get_current_full_url' => function_exists('weglot_get_current_full_url'),
                'weglot_get_service' => function_exists('weglot_get_service')
            ];
            
            $missing_dependencies = array_filter($dependencies, function($available) {
                return !$available;
            });
            
            if (!empty($missing_dependencies)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WeGlot integration dependencies missing: ' . implode(', ', array_keys($missing_dependencies)));
                }
                return false;
            }
            
            // Version compatibility check
            if (defined('WEGLOT_VERSION') && version_compare(WEGLOT_VERSION, '3.0.0', '<')) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WeGlot version ' . WEGLOT_VERSION . ' is not supported. Minimum version: 3.0.0');
                }
                return false;
            }
            
            // WeGlot services availability validation
            if (function_exists('weglot_get_service')) {
                try {
                    $language_service = weglot_get_service('Language_Service_Weglot');
                    $replace_link_service = weglot_get_service('Replace_Link_Service_Weglot');
                    
                    if (empty($language_service) || empty($replace_link_service)) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError('WeGlot required services not available');
                        }
                        return false;
                    }
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WeGlot services validation failed: ' . $e->getMessage());
                    }
                    return false;
                }
            }
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot integration successfully validated in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
            return true;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot availability check failed: ' . $e->getMessage());
            }
            return false;
        } catch (\Error $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot availability check error: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Initialize WeGlot integration with comprehensive hooks and monitoring
     * 
     * @return void
     */
    public function init(): void
    {
        $start_time = microtime(true);
        
        try {
            // Core WeGlot cache integration hooks
            add_filter('zizi_cache_cache_file_path', [$this, 'addLanguageCode'], 10, 1);
            add_filter('zizi_cache_preload_urls', [$this, 'addTranslatedUrls'], 10, 1);
            add_filter('zizi_cache_auto_purge_urls', [$this, 'addTranslatedUrls'], 10, 1);
            
            // WeGlot-specific hooks for cache coordination
            add_action('weglot_language_changed', [$this, 'handleLanguageChange'], 10, 2);
            add_action('weglot_save_words', [$this, 'purgeTranslationCache'], 10, 1);
            
            // Admin integration hooks
            add_action('admin_init', [$this, 'setupAdminHooks']);
            add_action('wp_ajax_weglot_cache_purge', [$this, 'handleAjaxCachePurge']);
            
            // Performance monitoring hooks
            add_action('shutdown', [$this, 'logPerformanceMetrics']);
            
            // Security hooks
            add_action('init', [$this, 'validateLanguageParameters'], 1);
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'init';
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot integration initialized successfully in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot integration initialization failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Add language code to cache file path with comprehensive validation
     * 
     * @param string $path Original cache file path
     * @return string Modified path with language code
     */
    public function addLanguageCode($path): string
    {
        $start_time = microtime(true);
        
        try {
            // Validate input
            if (empty($path) || !is_string($path)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid cache file path provided: ' . gettype($path));
                }
                return $path;
            }
            
            // Check WeGlot availability
            if (!defined('WEGLOT_VERSION')) {
                return $path;
            }
            
            // Get current URL with error handling
            $url = '';
            if (function_exists('weglot_get_current_full_url')) {
                try {
                    $url = weglot_get_current_full_url();
                } catch (\Exception $e) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Failed to get WeGlot current URL: ' . $e->getMessage());
                    }
                    return $path;
                }
            }
            
            if (empty($url)) {
                return $path;
            }
            
            // Validate and sanitize URL
            if (self::$config['url_validation'] && !filter_var($url, FILTER_VALIDATE_URL)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid URL detected from WeGlot: ' . $url);
                }
                return $path;
            }
              // Extract path following Flying Press competitor pattern
            $language_path = parse_url($url, PHP_URL_PATH);
            
            if ($language_path === false || $language_path === null) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Failed to parse URL path: ' . $url);
                }
                return $path;
            }
            
            // Security: Sanitize path to prevent path traversal (enterprise enhancement)
            $sanitized_path = ltrim($language_path, '/');
            $sanitized_path = preg_replace('/[^a-zA-Z0-9\-_\/]/', '', $sanitized_path);
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError("Language code added to cache path: {$language_path} in " . round($execution_time * 1000, 2) . 'ms');
            }
            
            // Return original parsed path (competitor compatibility) or sanitized for security
            return self::$config['url_validation'] ? $sanitized_path : $language_path;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Language code addition failed: ' . $e->getMessage());
            }
            return $path;
        }
    }

    /**
     * Add translated URLs for preload and auto-purge with enterprise implementation
     * Following competitor pattern but with enhanced security and error handling
     * 
     * @param array $urls Original URLs array
     * @return array Expanded URLs with all language variants
     */
    public function addTranslatedUrls($urls): array
    {
        $start_time = microtime(true);
        
        try {
            // Validate input
            if (!is_array($urls)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid URLs input provided: ' . gettype($urls));
                }
                return is_array($urls) ? $urls : [];
            }
            
            // Check WeGlot availability
            if (!defined('WEGLOT_VERSION') || !function_exists('weglot_get_service')) {
                return $urls;
            }
            
            // Get WeGlot services with error handling
            try {
                $language_service = weglot_get_service('Language_Service_Weglot');
                $replace_link_service = weglot_get_service('Replace_Link_Service_Weglot');
                
                if (empty($language_service) || empty($replace_link_service)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WeGlot services not available for URL translation');
                    }
                    return $urls;
                }
                
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Failed to get WeGlot services: ' . $e->getMessage());
                }
                return $urls;
            }
            
            // Get destination languages with error handling
            try {
                $languages = $language_service->get_destination_languages();
                
                if (empty($languages) || !is_array($languages)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('No destination languages found in WeGlot');
                    }
                    return $urls;
                }
                
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Failed to get WeGlot destination languages: ' . $e->getMessage());
                }
                return $urls;
            }            
            $languages_processed = 0;
            
            // Generate translated URLs following Flying Press competitor pattern
            // Directly append to original urls array (competitor compatibility)
            foreach ($urls as $url) {
                // Validate each URL (enterprise security enhancement)
                if (empty($url) || !is_string($url)) {
                    continue;
                }
                
                if (self::$config['url_validation'] && !filter_var($url, FILTER_VALIDATE_URL)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Invalid URL skipped: ' . $url);
                    }
                    continue;
                }
                
                foreach ($languages as $language) {
                    try {
                        // Validate language object (enterprise validation)
                        if (empty($language) || !is_object($language)) {
                            continue;
                        }
                        
                        // Generate translated URL (same as competitor)
                        $translated_url = $replace_link_service->replace_url($url, $language);
                        
                        if (!empty($translated_url) && is_string($translated_url)) {
                            // Additional validation for translated URL (enterprise feature)
                            if (!self::$config['url_validation'] || filter_var($translated_url, FILTER_VALIDATE_URL)) {
                                $urls[] = $translated_url; // Direct append like Flying Press
                                $languages_processed++;
                            } else {
                                if (self::$config['enable_logging']) {
                                    CacheSys::writeError('Invalid translated URL generated: ' . $translated_url);
                                }
                            }
                        }
                        
                    } catch (\Exception $e) {
                        self::$metrics['error_count']++;
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("URL translation failed for {$url}: " . $e->getMessage());
                        }
                        continue;
                    }
                }
            }
            
            // Remove duplicates (same as Flying Press pattern)
            $urls = array_unique($urls);
            
            self::$metrics['languages_processed'] = $languages_processed;
            self::$metrics['url_generation_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'url_translation';
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                $original_count = count($urls) - $languages_processed; // Approximate original count
                $final_count = count($urls);
                CacheSys::writeError("URL translation completed: ~{$original_count} → {$final_count} URLs, {$languages_processed} languages in " . round($execution_time * 1000, 2) . 'ms');
            }
            
            return $urls;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('URL translation process failed: ' . $e->getMessage());
            }
            return $urls;
        }
    }

    /**
     * Handle language change event with cache coordination
     * 
     * @param string $new_language New language code
     * @param string|null $old_language Previous language code
     * @return void
     */
    public function handleLanguageChange($new_language, $old_language = null): void
    {
        try {
            // Validate language codes
            if (empty($new_language) || !is_string($new_language)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid new language provided: ' . gettype($new_language));
                }
                return;
            }
            
            // Purge cache for language-specific content
            $this->removePageCache();
            
            // Clear language-specific transients
            delete_transient('weglot_cache_' . $new_language);
            if ($old_language) {
                delete_transient('weglot_cache_' . $old_language);
            }
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError("Language change handled: {$old_language} → {$new_language}");
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Language change handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Purge translation cache when translations are updated
     * 
     * @param array $words Updated translation words
     * @return void
     */
    public function purgeTranslationCache($words): void
    {
        try {
            if (!is_array($words)) {
                return;
            }
            
            // Remove page cache for all translated content
            $this->removePageCache();
            
            // Clear WeGlot-specific caches
            delete_transient('weglot_translations');
            delete_transient('weglot_languages');
            
            if (self::$config['enable_logging']) {
                $word_count = count($words);
                CacheSys::writeError("Translation cache purged for {$word_count} updated words");
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Translation cache purge failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Setup admin interface hooks
     * 
     * @return void
     */
    public function setupAdminHooks(): void
    {
        try {
            if (!is_admin()) {
                return;
            }
            
            add_action('admin_menu', [$this, 'addAdminMenuItems']);
            add_action('admin_notices', [$this, 'showAdminNotices']);
            add_filter('plugin_action_links', [$this, 'addPluginActionLinks'], 10, 2);
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WeGlot admin hooks initialized');
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Admin hooks setup failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Log performance metrics for monitoring
     * 
     * @return void
     */
    public function logPerformanceMetrics(): void
    {
        try {
            if (!self::$config['enable_logging']) {
                return;
            }
            
            $metrics_summary = sprintf(
                'WeGlot Performance Metrics - Init: %sms, URL Gen: %sms, Languages: %d, Errors: %d, Last: %s',
                round(self::$metrics['init_time'] * 1000, 2),
                round(self::$metrics['url_generation_time'] * 1000, 2),
                self::$metrics['languages_processed'],
                self::$metrics['error_count'],
                self::$metrics['last_operation'] ?: 'none'
            );
            
            CacheSys::writeError($metrics_summary);
            
        } catch (\Exception $e) {
            // Silent fail for logging to prevent infinite loops
        }
    }

    /**
     * Validate language parameters for security
     * 
     * @return void
     */
    public function validateLanguageParameters(): void
    {
        try {
            // Validate WeGlot language parameters in URL
            if (isset($_GET['weglot_language'])) {
                $language = sanitize_text_field($_GET['weglot_language']);
                
                // Basic language code validation (2-3 letter codes)
                if (!preg_match('/^[a-z]{2,3}(-[A-Z]{2})?$/', $language)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Invalid WeGlot language parameter detected: ' . $language);
                    }
                    unset($_GET['weglot_language']);
                }
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Language parameter validation failed: ' . $e->getMessage());
            }
        }
    }
}
