<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Core\CacheSys;

/**
 * WPML Multilingual Integration
 * 
 * Enterprise-grade integration for WPML (WordPress Multilingual Plugin) with comprehensive
 * multilingual cache management, URL translation coordination, and enhanced I18n features.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @since 2.0.0
 * @version 2.0.0
 */
class WPML extends PluginBase
{
    /**
     * Plugin configuration
     * @var array
     */
    private static $config = [
        'cache_timeout' => 600,
        'max_retries' => 3,
        'enable_logging' => true,
        'validate_languages' => true,
        'include_default_language' => true,
        'cache_language_switcher' => true,
        'preload_all_languages' => true
    ];

    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'init_time' => 0,
        'url_translations' => 0,
        'languages_processed' => 0,
        'preload_operations' => 0,
        'purge_operations' => 0,
        'error_count' => 0,
        'last_operation' => null
    ];

    /**
     * Language cache
     * @var array
     */
    private static $language_cache = [];

    /**
     * Check WPML plugin availability with comprehensive validation
     * 
     * @return bool True if WPML is available and properly configured
     */
    protected static function checkAvailability(): bool
    {
        $start_time = microtime(true);
        
        try {
            // Core WPML dependencies check
            $dependencies = [
                'ICL_SITEPRESS_VERSION' => defined('ICL_SITEPRESS_VERSION'),
                'apply_filters' => function_exists('apply_filters'),
                'home_url' => function_exists('home_url'),
                'wpml_active_languages' => true, // Filter availability checked later
                'wpml_permalink' => true // Filter availability checked later
            ];
            
            $missing_dependencies = array_filter($dependencies, function($available) {
                return !$available;
            });
            
            if (!empty($missing_dependencies)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WPML integration dependencies missing: ' . implode(', ', array_keys($missing_dependencies)));
                }
                return false;
            }
            
            // WPML version compatibility check
            if (defined('ICL_SITEPRESS_VERSION')) {
                $wpml_version = ICL_SITEPRESS_VERSION;
                if (version_compare($wpml_version, '4.0.0', '<')) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WPML version ' . $wpml_version . ' is not supported. Minimum version: 4.0.0');
                    }
                    return false;
                }
            }
            
            // WPML active languages validation
            try {
                $active_languages = apply_filters('wpml_active_languages', null, []);
                if (empty($active_languages) || !is_array($active_languages)) {
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WPML active languages not found or invalid configuration');
                    }
                    return false;
                }
                
                // Validate language structure
                foreach ($active_languages as $lang_code => $language) {
                    if (!is_array($language) || !isset($language['code'])) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Invalid WPML language structure for: {$lang_code}");
                        }
                        return false;
                    }
                }
                
                self::$language_cache = $active_languages;
                
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WPML languages validation failed: ' . $e->getMessage());
                }
                return false;
            }
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['languages_processed'] = count(self::$language_cache);
            
            if (self::$config['enable_logging']) {
                $lang_count = count(self::$language_cache);
                CacheSys::writeError("WPML integration successfully validated with {$lang_count} languages in " . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
            return true;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML availability check failed: ' . $e->getMessage());
            }
            return false;
        } catch (\Error $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML availability check error: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Initialize WPML integration with comprehensive hooks and monitoring
     * 
     * @return void
     */
    public function init(): void
    {
        $start_time = microtime(true);
        
        try {
            // Core WPML URL translation hooks
            add_filter('zizi_cache_preload_urls', [$this, 'addTranslatedUrls'], 10, 1);
            add_filter('zizi_cache_auto_purge_urls', [$this, 'addTranslatedUrls'], 10, 1);
            
            // Additional WPML-specific hooks for comprehensive multilingual cache management
            add_action('wpml_translation_update', [$this, 'handleTranslationUpdate'], 10, 3);
            add_action('wpml_language_updated', [$this, 'handleLanguageUpdate'], 10, 2);
            add_action('wpml_cache_clear', [$this, 'handleWpmlCacheClear'], 10, 1);
            
            // Language switcher hooks
            add_filter('wpml_ls_directories_for_domain', [$this, 'cacheLsDirectories'], 10, 1);
            add_action('wpml_switch_language', [$this, 'handleLanguageSwitch'], 10, 2);
            
            // Admin integration hooks
            add_action('admin_init', [$this, 'setupAdminHooks']);
            add_action('wp_ajax_wpml_cache_purge', [$this, 'handleAjaxCachePurge']);
            
            // Performance monitoring hooks
            add_action('shutdown', [$this, 'logPerformanceMetrics']);
            
            // Security hooks
            add_action('init', [$this, 'validateWpmlParameters'], 1);
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'init';
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML integration initialized successfully in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML integration initialization failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Add translated URLs for multilingual cache operations
     * Enhanced implementation following competitor pattern with enterprise features
     * 
     * @param array $urls Original URLs array
     * @return array Enhanced URLs array with translations
     */
    public function addTranslatedUrls($urls): array
    {
        $start_time = microtime(true);
        
        try {
            // Validate input
            if (!is_array($urls)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid URLs array provided for translation: ' . gettype($urls));
                }
                return is_array($urls) ? $urls : [];
            }
            
            // Get active WPML languages with caching
            $languages = $this->getActiveLanguages();
            
            if (empty($languages)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('No active WPML languages found for URL translation');
                }
                return $urls;
            }
            
            $translated_urls = [];
            
            // Get home URLs for each language (following competitor pattern)
            try {
                foreach ($languages as $language) {
                    if (!is_array($language) || !isset($language['code'])) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError('Invalid language structure in WPML translation');
                        }
                        continue;
                    }
                    
                    $lang_code = sanitize_text_field($language['code']);
                    if (empty($lang_code)) {
                        continue;
                    }
                    
                    try {
                        $home_url = home_url();
                        $translated_home = apply_filters('wpml_permalink', $home_url, $lang_code);
                        
                        if (!empty($translated_home) && $translated_home !== false) {
                            $translated_urls[] = $translated_home;
                        }
                        
                    } catch (\Exception $e) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Error translating home URL for language {$lang_code}: " . $e->getMessage());
                        }
                    }
                }
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Error processing WPML home URLs: ' . $e->getMessage());
                }
            }
            
            // Get the subsequent translated URLs of the original URLs (following competitor pattern)
            try {
                foreach ($urls as $url) {
                    if (!is_string($url) || empty($url)) {
                        continue;
                    }
                    
                    // Validate URL format
                    if (!filter_var($url, FILTER_VALIDATE_URL) && !preg_match('/^\/[^\/]/', $url)) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Invalid URL format for translation: {$url}");
                        }
                        continue;
                    }
                    
                    foreach ($languages as $language) {
                        if (!is_array($language) || !isset($language['code'])) {
                            continue;
                        }
                        
                        $lang_code = sanitize_text_field($language['code']);
                        if (empty($lang_code)) {
                            continue;
                        }
                        
                        try {
                            $translated_url = apply_filters('wpml_permalink', $url, $lang_code, true);
                            
                            if (!empty($translated_url) && $translated_url !== false && $translated_url !== $url) {
                                $translated_urls[] = $translated_url;
                            }
                            
                        } catch (\Exception $e) {
                            if (self::$config['enable_logging']) {
                                CacheSys::writeError("Error translating URL {$url} for language {$lang_code}: " . $e->getMessage());
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Error processing URL translations: ' . $e->getMessage());
                }
            }
            
            // Merge URLs using modern syntax (following competitor pattern)
            $urls = array_unique([...$urls, ...$translated_urls]);
            
            // Filter out invalid URLs
            $urls = array_filter($urls, function($url) {
                return !empty($url) && is_string($url);
            });
            
            self::$metrics['url_translations'] += count($translated_urls);
            self::$metrics['last_operation'] = 'url_translation';
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                $original_count = count($urls) - count($translated_urls);
                $translated_count = count($translated_urls);
                $total_count = count($urls);
                CacheSys::writeError("WPML URL translation completed: {$original_count} original + {$translated_count} translated = {$total_count} total URLs in " . round($execution_time * 1000, 2) . 'ms');
            }
            
            return $urls;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML URL translation failed: ' . $e->getMessage());
            }
            return $urls;
        }
    }

    /**
     * Get active WPML languages with caching and validation
     * 
     * @return array Active languages array
     */
    private function getActiveLanguages(): array
    {
        try {
            // Return cached languages if available
            if (!empty(self::$language_cache)) {
                return self::$language_cache;
            }
            
            // Get fresh languages from WPML
            $languages = apply_filters('wpml_active_languages', null, []);
            
            if (!is_array($languages) || empty($languages)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WPML active languages returned empty or invalid data');
                }
                return [];
            }
            
            // Validate and cache languages
            $validated_languages = [];
            foreach ($languages as $lang_code => $language) {
                if (is_array($language) && isset($language['code']) && !empty($language['code'])) {
                    $validated_languages[$lang_code] = $language;
                }
            }
            
            self::$language_cache = $validated_languages;
            
            if (self::$config['enable_logging']) {
                $lang_count = count($validated_languages);
                CacheSys::writeError("WPML languages cached: {$lang_count} active languages");
            }
            
            return $validated_languages;
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Error getting WPML active languages: ' . $e->getMessage());
            }
            return [];
        }
    }

    /**
     * Handle WPML translation updates with cache coordination
     * 
     * @param int $element_id Element ID
     * @param string $element_type Element type
     * @param int $trid Translation ID
     * @return void
     */
    public function handleTranslationUpdate($element_id, $element_type, $trid): void
    {
        try {
            if (!empty($element_id) && !empty($element_type)) {
                // Clear language cache to ensure fresh data
                self::$language_cache = [];
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("WPML translation update handled for {$element_type} ID: {$element_id} (TRID: {$trid})");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML translation update handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Handle WPML language updates with cache coordination
     * 
     * @param string $language_code Language code
     * @param array $language_data Language data
     * @return void
     */
    public function handleLanguageUpdate($language_code, $language_data): void
    {
        try {
            if (!empty($language_code)) {
                // Clear language cache to ensure fresh data
                self::$language_cache = [];
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("WPML language update handled for: {$language_code}");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML language update handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Handle WPML cache clear operations
     * 
     * @param mixed $data Cache clear data
     * @return void
     */
    public function handleWpmlCacheClear($data): void
    {
        try {
            // Clear internal cache
            self::$language_cache = [];
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML cache clear operation handled');
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML cache clear handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Cache language switcher directories
     * 
     * @param array $directories Directories array
     * @return array Cached directories
     */
    public function cacheLsDirectories($directories): array
    {
        try {
            if (is_array($directories)) {
                if (self::$config['enable_logging']) {
                    $dir_count = count($directories);
                    CacheSys::writeError("WPML language switcher directories cached: {$dir_count} directories");
                }
            }
            
            return $directories;
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML language switcher caching failed: ' . $e->getMessage());
            }
            return $directories;
        }
    }

    /**
     * Handle language switching with cache coordination
     * 
     * @param string $old_language Previous language
     * @param string $new_language New language
     * @return void
     */
    public function handleLanguageSwitch($old_language, $new_language): void
    {
        try {
            if (!empty($old_language) && !empty($new_language) && $old_language !== $new_language) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("WPML language switch handled: {$old_language} → {$new_language}");
                }
            }
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML language switch handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Setup admin interface hooks
     * 
     * @return void
     */
    public function setupAdminHooks(): void
    {
        try {
            if (!is_admin()) {
                return;
            }
            
            add_action('admin_menu', [$this, 'addAdminMenuItems']);
            add_action('admin_notices', [$this, 'showAdminNotices']);
            add_filter('plugin_action_links', [$this, 'addPluginActionLinks'], 10, 2);
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML admin hooks initialized');
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML admin hooks setup failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Log performance metrics for monitoring
     * 
     * @return void
     */
    public function logPerformanceMetrics(): void
    {
        try {
            if (!self::$config['enable_logging']) {
                return;
            }
            
            $metrics_summary = sprintf(
                'WPML Performance Metrics - Init: %sms, URLs: %d, Languages: %d, Preloads: %d, Purges: %d, Errors: %d, Last: %s',
                round(self::$metrics['init_time'] * 1000, 2),
                self::$metrics['url_translations'],
                self::$metrics['languages_processed'],
                self::$metrics['preload_operations'],
                self::$metrics['purge_operations'],
                self::$metrics['error_count'],
                self::$metrics['last_operation'] ?: 'none'
            );
            
            CacheSys::writeError($metrics_summary);
            
        } catch (\Exception $e) {
            // Silent fail for logging to prevent infinite loops
        }
    }

    /**
     * Validate WPML parameters for security
     * 
     * @return void
     */
    public function validateWpmlParameters(): void
    {
        try {
            // Validate WPML-specific parameters
            $wpml_params = ['lang', 'wpml_lang', 'language'];
            
            foreach ($wpml_params as $param) {
                if (isset($_GET[$param])) {
                    $value = sanitize_text_field($_GET[$param]);
                    
                    // Validate language code format
                    if (!preg_match('/^[a-z]{2}(-[A-Z]{2})?$/', $value)) {
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("Invalid WPML language parameter {$param} detected: {$value}");
                        }
                        unset($_GET[$param]);
                    }
                }
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WPML parameter validation failed: ' . $e->getMessage());
            }
        }
    }
}
