<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Core\CacheSys;

/**
 * WooCommerce Multilingual (WCML) Integration
 * 
 * Enterprise-grade integration for WooCommerce Multilingual plugin with 
 * multi-currency cache management, comprehensive error handling, and security features.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @since 2.0.0
 * @version 2.0.0
 */
class WCML extends PluginBase
{
    /**
     * Plugin configuration
     * @var array
     */
    private static $config = [
        'currency_cookie_name' => 'wcml_client_currency',
        'cache_timeout' => 300,
        'max_retries' => 3,
        'enable_logging' => true
    ];

    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'init_time' => 0,
        'purge_count' => 0,
        'error_count' => 0,
        'last_operation' => null
    ];

    /**
     * Check WCML plugin availability with comprehensive validation
     * 
     * @return bool True if WCML is available and properly configured
     */
    protected static function checkAvailability(): bool
    {
        $start_time = microtime(true);
        
        try {
            // Primary check: Function-based WCML API (standard approach)
            if (function_exists('wcml_is_multi_currency_on')) {
                $function_available = true;
                CacheSys::writeError('WCML function-based API detected', 'info');
            } else {
                $function_available = false;
                CacheSys::writeError('WCML function-based API not available', 'warning');
            }

            // Secondary check: Class-based WCML API (legacy/enterprise)
            $class_available = false;
            if (class_exists('\\WCML\\Multi_Currency')) {
                $class_available = true;
                CacheSys::writeError('WCML class-based API detected', 'info');
            }

            // Third check: Plugin file existence
            $plugin_file_exists = false;
            $wcml_paths = [
                'wpml-woocommerce/wpml-woocommerce.php',
                'woocommerce-multilingual/wpml-woocommerce.php'
            ];

            foreach ($wcml_paths as $path) {
                if (defined('WP_PLUGIN_DIR') && file_exists(WP_PLUGIN_DIR . '/' . $path)) {
                    $plugin_file_exists = true;
                    CacheSys::writeError("WCML plugin file found: {$path}", 'info');
                    break;
                }
            }

            // Hybrid availability: Function-based API is preferred
            $is_available = $function_available || $class_available || $plugin_file_exists;

            // Log availability status
            if ($is_available) {
                CacheSys::writeError('WCML hybrid availability confirmed', 'info');
                
                // Test multi-currency status if function is available
                if ($function_available && function_exists('wcml_is_multi_currency_on')) {
                    $multicurrency_status = wcml_is_multi_currency_on();
                    CacheSys::writeError("WCML multi-currency status: " . ($multicurrency_status ? 'enabled' : 'disabled'), 'info');
                }
            } else {
                CacheSys::writeError('WCML not available - no API detected', 'warning');
            }

            // Performance tracking
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError("WCML availability check completed in {$execution_time}s", 'info');

            return $is_available;

        } catch (\Exception $e) {
            CacheSys::writeError("WCML availability check failed: " . $e->getMessage(), 'error');
            return false;
        } catch (\Error $e) {
            CacheSys::writeError("WCML availability check fatal error: " . $e->getMessage(), 'error');
            return false;
        }
    }

    /**
     * Initialize WCML integration with comprehensive hooks and monitoring
     * 
     * @return void
     */
    public function init(): void
    {
        $start_time = microtime(true);
        
        try {
            // Setup integration with delay for proper WCML initialization
            add_action('init', [$this, 'setupIntegration'], 10);
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'init';
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML integration init hooks registered in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML integration initialization failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Setup WCML integration after WordPress init (following competitor pattern)
     * 
     * @return void
     */
    public function setupIntegration(): void
    {
        $start_time = microtime(true);
        
        try {
            // Verify WCML function availability before proceeding
            if (!function_exists('\WCML\functions\getClientCurrency')) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WCML getClientCurrency function not available during setup');
                }
                return;
            }
            
            // Set initial currency cookie
            $this->setCurrencyCookie();
            
            // Core currency switching hooks
            add_action('wcml_switch_currency', [$this, 'setCurrencyCookie'], 2, 1);
            add_action('wp_set_current_user', [$this, 'handleUserCurrencyChange'], 10, 1);
            
            // WooCommerce currency hooks
            add_action('woocommerce_currency_changed', [$this, 'purgeCache'], 10);
            add_filter('wcml_client_currency', [$this, 'validateClientCurrency'], 10, 1);
            
            // Plugin deactivation hook (following competitor pattern)
            add_action('deactivate_plugin', [$this, 'removeCurrencyCookie'], 5, 1);
            
            // Cache coordination hooks
            add_filter('zizi_cache_include_cookies', [$this, 'includeCurrencyCookie'], 10, 1);
            
            // Admin integration hooks
            add_action('admin_init', [$this, 'setupAdminHooks']);
            add_action('wp_ajax_wcml_cache_purge', [$this, 'handleAjaxCachePurge']);
            
            // Performance monitoring hooks
            add_action('shutdown', [$this, 'logPerformanceMetrics']);
            
            // Security hooks
            add_action('init', [$this, 'validateCurrencyCookies'], 1);
            add_action('wp_login', [$this, 'clearUserCurrencyCache'], 10, 2);
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML integration setup completed in ' . round($execution_time * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML integration setup failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Purge cache when currency changes with comprehensive validation
     * 
     * @param string|null $new_currency New currency code
     * @param string|null $old_currency Previous currency code
     * @return void
     */
    public function purgeCache($new_currency = null, $old_currency = null): void
    {
        $start_time = microtime(true);
        
        try {
            // Validate currency parameters
            if ($new_currency && !$this->isValidCurrency($new_currency)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid new currency provided: ' . $new_currency);
                }
                return;
            }
            
            if ($old_currency && !$this->isValidCurrency($old_currency)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid old currency provided: ' . $old_currency);
                }
                return;
            }
            
            // Remove page cache with currency context
            $this->removePageCache();
            
            // Clear currency-specific caches
            $this->clearCurrencySpecificCaches($new_currency, $old_currency);
            
            // Update currency cookie securely
            $this->updateCurrencyCookie($new_currency);
            
            // Clear related transients
            $this->clearCurrencyTransients($new_currency, $old_currency);
            
            self::$metrics['purge_count']++;
            self::$metrics['last_operation'] = 'purge';
            
            $execution_time = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError("WCML cache purged successfully (currency: {$new_currency}) in " . round($execution_time * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML cache purge failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Handle user currency change with security validation
     * 
     * @param int $user_id User ID
     * @return void
     */
    public function handleUserCurrencyChange($user_id): void
    {
        try {
            if (!is_numeric($user_id) || $user_id <= 0) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid user ID provided for currency change: ' . $user_id);
                }
                return;
            }
            
            // Get user's preferred currency
            $user_currency = get_user_meta($user_id, 'wcml_user_currency', true);
            
            if ($user_currency && $this->isValidCurrency($user_currency)) {
                $this->purgeCache($user_currency);
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("User currency change processed for user {$user_id}: {$user_currency}");
                }
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('User currency change handling failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Validate client currency with security checks
     * 
     * @param string $currency Currency code to validate
     * @return string Validated currency code
     */
    public function validateClientCurrency($currency): string
    {
        try {
            // Sanitize input
            $currency = sanitize_text_field($currency);
            
            if (empty($currency)) {
                return $this->getDefaultCurrency();
            }
            
            // Validate currency format and availability
            if (!$this->isValidCurrency($currency)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('Invalid client currency detected: ' . $currency);
                }
                return $this->getDefaultCurrency();
            }
            
            return $currency;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Client currency validation failed: ' . $e->getMessage());
            }
            return $this->getDefaultCurrency();
        }
    }

    /**
     * Remove and update currency cookie with security measures
     * 
     * @param string|null $new_currency New currency to set
     * @return void
     */
    public function updateCurrencyCookie($new_currency = null): void
    {
        try {
            $cookie_name = self::$config['currency_cookie_name'];
            
            // Remove existing currency cookie
            if (isset($_COOKIE[$cookie_name])) {
                unset($_COOKIE[$cookie_name]);
                setcookie($cookie_name, '', time() - 3600, '/', '', is_ssl(), true);
            }
            
            // Set new currency cookie if provided
            if ($new_currency && $this->isValidCurrency($new_currency)) {
                $secure = is_ssl();
                $httponly = true;
                $samesite = 'Lax';
                
                setcookie(
                    $cookie_name, 
                    sanitize_text_field($new_currency), 
                    time() + self::$config['cache_timeout'], 
                    '/', 
                    '', 
                    $secure, 
                    $httponly
                );
                
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("Currency cookie updated: {$new_currency}");
                }
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Currency cookie update failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Set currency cookie using standard WCML API (following competitor pattern)
     * 
     * @param string $currency Optional currency code
     * @return void
     */
    public function setCurrencyCookie($currency = ''): void
    {
        $start_time = microtime(true);
        
        try {
            // Check if multi-currency is enabled using standard WCML function
            if (!function_exists('wcml_is_multi_currency_on') || !wcml_is_multi_currency_on()) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('WCML multi-currency not enabled, skipping cookie setup');
                }
                return;
            }
            
            $cookie_name = 'wcml_currency'; // Using standard WCML cookie name
            $expiry = time() + 14 * DAY_IN_SECONDS;
            
            // Set cookie if not exists and no currency provided
            if (!isset($_COOKIE[$cookie_name]) && empty($currency)) {
                if (function_exists('\WCML\functions\getClientCurrency')) {
                    $currency = \WCML\functions\getClientCurrency();
                    $currency = sanitize_text_field($currency);
                    
                    if ($this->isValidCurrency($currency)) {
                        setcookie($cookie_name, $currency, $expiry, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
                        
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError("WCML currency cookie set: {$currency}");
                        }
                    }
                }
            }
            
            // Update cookie if currency changed
            if (!empty($currency) && isset($_COOKIE[$cookie_name]) && $currency !== $_COOKIE[$cookie_name]) {
                $currency = sanitize_text_field($currency);
                
                if ($this->isValidCurrency($currency)) {
                    setcookie($cookie_name, $currency, $expiry, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
                    
                    // Trigger cache purge for currency change
                    $this->purgeCache($currency, $_COOKIE[$cookie_name]);
                    
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError("WCML currency cookie updated: {$currency}");
                    }
                }
            }
            
            $execution_time = microtime(true) - $start_time;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML currency cookie setup failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Remove currency cookie when WCML plugin is deactivated (following competitor pattern)
     * 
     * @param string $plugin Plugin path being deactivated
     * @return void
     */
    public function removeCurrencyCookie($plugin): void
    {
        try {
            // Check if it's the WCML plugin being deactivated
            if ($plugin === 'woocommerce-multilingual/wpml-woocommerce.php') {
                $cookie_name = 'wcml_currency';
                
                if (isset($_COOKIE[$cookie_name])) {
                    // Remove from $_COOKIE superglobal
                    unset($_COOKIE[$cookie_name]);
                    
                    // Set expired cookie to remove from browser
                    setcookie($cookie_name, '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
                    
                    // Purge all currency-related cache
                    $this->purgeCache();
                    
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('WCML currency cookie removed due to plugin deactivation');
                    }
                }
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('WCML currency cookie removal failed: ' . $e->getMessage());
            }
        }
    }    /**
     * Include currency in cache key (compatibility method for cache systems)
     * Following Flying Press pattern for cache integration
     * 
     * @param array $cookies Current cookies for cache key
     * @return array Updated cookies with currency if applicable
     */
    public static function includeCurrencyCookie(array $cookies = []): array
    {
        try {
            // Check if multi-currency is enabled using function-based API
            if (function_exists('wcml_is_multi_currency_on') && !wcml_is_multi_currency_on()) {
                CacheSys::writeError("Multi-currency disabled, currency cookie not included", 'info');
                return $cookies;
            }

            // Get current currency using function-based API
            $current_currency = '';
            if (function_exists('\\WCML\\functions\\getClientCurrency')) {
                $current_currency = \WCML\functions\getClientCurrency();
                CacheSys::writeError("Currency obtained via function API: {$current_currency}", 'info');
            }
            // Fallback to standard WCML cookie
            elseif (isset($_COOKIE['wcml_currency'])) {
                $current_currency = sanitize_text_field($_COOKIE['wcml_currency']);
                CacheSys::writeError("Currency obtained from cookie: {$current_currency}", 'info');
            }

            // Include currency in cache key if available
            if (!empty($current_currency) && preg_match('/^[A-Z]{3}$/', $current_currency)) {
                $cookies['wcml_currency'] = $current_currency;
                CacheSys::writeError("Currency included in cache key: {$current_currency}", 'info');
            } else {
                CacheSys::writeError("No valid currency for cache inclusion", 'warning');
            }

            return $cookies;

        } catch (\Exception $e) {
            CacheSys::writeError("Currency inclusion failed: " . $e->getMessage(), 'error');
            return $cookies;
        }
    }

    /**
     * Log performance metrics for monitoring
     * 
     * @return void
     */
    public function logPerformanceMetrics(): void
    {
        try {
            if (!self::$config['enable_logging']) {
                return;
            }
            
            $metrics_summary = sprintf(
                'WCML Performance Metrics - Init: %sms, Purges: %d, Errors: %d, Last: %s',
                round(self::$metrics['init_time'] * 1000, 2),
                self::$metrics['purge_count'],
                self::$metrics['error_count'],
                self::$metrics['last_operation'] ?: 'none'
            );
            
            CacheSys::writeError($metrics_summary);
            
        } catch (\Exception $e) {
            // Silent fail for logging to prevent infinite loops
        }
    }

    /**
     * Validate currency cookies for security
     * 
     * @return void
     */
    public function validateCurrencyCookies(): void
    {
        try {
            $cookie_name = self::$config['currency_cookie_name'];
            
            if (isset($_COOKIE[$cookie_name])) {
                $currency = sanitize_text_field($_COOKIE[$cookie_name]);
                
                if (!$this->isValidCurrency($currency)) {
                    $this->updateCurrencyCookie(null); // Remove invalid cookie
                    
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('Invalid currency cookie detected and removed: ' . $currency);
                    }
                }
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('Currency cookie validation failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Clear user-specific currency cache on login
     * 
     * @param string $user_login User login name
     * @param \WP_User $user User object
     * @return void
     */
    public function clearUserCurrencyCache($user_login, $user): void
    {
        try {
            if (!$user instanceof \WP_User) {
                return;
            }
            
            // Clear user-specific currency cache
            delete_user_meta($user->ID, 'wcml_user_currency_cache');
            wp_cache_delete("wcml_user_currency_{$user->ID}", 'wcml');
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError("User currency cache cleared for user: {$user->user_login}");
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('User currency cache clearing failed: ' . $e->getMessage());
            }
        }
    }
}
