<?php
/**
 * Varnish HTTP Cache Integration for ZiziCache
 * 
 * Enterprise-grade integration with Varnish HTTP cache server for coordinated
 * cache purging operations. Provides comprehensive PURGE functionality, security
 * hardening, and robust error handling for production environments.
 * 
 * @package ZiziCache
 * @subpackage Plugins\Integrations
 * @version 2.0.0
 * @since 1.0.0
 * @author ZiziCache Team
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\Helpers\CacheSys;

/**
 * Varnish HTTP Cache Integration Class
 * 
 * Handles coordination between ZiziCache and Varnish HTTP cache server including:
 * - Automatic Varnish server detection via HTTP headers analysis
 * - Secure PURGE operations for specific URLs and full cache
 * - Network error handling and timeout protection
 * - Performance monitoring and structured logging
 * - Security hardening against SSRF and injection attacks
 */
class Varnish extends PluginBase {
    use PluginTrait;

    /**
     * Configuration constants
     */
    private const VARNISH_HEADERS = ['X-Varnish', 'Via', 'X-Cache', 'X-Cache-Status'];
    private const PURGE_METHOD = 'PURGE';
    private const DEFAULT_TIMEOUT = 5; // seconds
    private const MAX_RETRY_ATTEMPTS = 3;
    private const BATCH_SIZE = 20;
    private const MAX_EXECUTION_TIME = 10.0; // seconds
    private const DEFAULT_VARNISH_PORT = 80;

    /**
     * State management
     */
    private static $varnish_servers = [];
    private static $is_initialized = false;
    private static $detection_cache = null;
    private static $performance_metrics = [];
    private static $failed_servers = [];
    private static $purge_statistics = [
        'total_requests' => 0,
        'successful_purges' => 0,
        'failed_purges' => 0,
        'avg_response_time' => 0
    ];

    /**
     * Check if Varnish cache server is available and properly configured
     * 
     * Implements abstract method from PluginBase with comprehensive Varnish detection
     * including HTTP headers analysis, server connectivity testing, and security validation.
     * 
     * @return bool True if Varnish is available and functional
     */
    protected static function checkAvailability(): bool {
        $start_time = microtime(true);
        
        try {
            // Use cached detection result if available
            if (self::$detection_cache !== null) {
                $execution_time = microtime(true) - $start_time;
                CacheSys::writeError('Varnish: Using cached availability result (' . (self::$detection_cache ? 'available' : 'unavailable') . ') in ' . round($execution_time * 1000, 2) . 'ms');
                return self::$detection_cache;
            }

            // Check if we're in a web context (not CLI)
            if (!isset($_SERVER['HTTP_HOST']) || php_sapi_name() === 'cli') {
                CacheSys::writeError('Varnish: Not in web context, skipping detection');
                self::$detection_cache = false;
                return false;
            }

            // Detect Varnish through multiple methods
            $detection_methods = [
                'headers' => self::detectVarnishByHeaders(),
                'response' => self::detectVarnishByResponse(),
                'config' => self::detectVarnishByConfig()
            ];

            $varnish_detected = false;
            foreach ($detection_methods as $method => $result) {
                if ($result) {
                    CacheSys::writeError('Varnish: Detected via ' . $method . ' method');
                    $varnish_detected = true;
                    break;
                }
            }

            // Initialize Varnish servers list if detected
            if ($varnish_detected) {
                self::initializeVarnishServers();
            }

            self::$detection_cache = $varnish_detected;
            $execution_time = microtime(true) - $start_time;
            
            CacheSys::writeError('Varnish: Availability detection completed - ' . 
                ($varnish_detected ? 'Available' : 'Not available') . 
                ' (' . count(self::$varnish_servers) . ' servers) in ' . 
                round($execution_time * 1000, 2) . 'ms');

            return $varnish_detected;

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('Varnish: Availability check failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            self::$detection_cache = false;
            return false;
        }
    }

    /**
     * Initialize Varnish integration
     * 
     * Sets up comprehensive integration including hook registration,
     * server configuration, and performance monitoring setup.
     * 
     * @return void
     */
    public static function init(): void {
        $start_time = microtime(true);
        
        try {
            if (!static::checkAvailability()) {
                CacheSys::writeError('Varnish: Initialization skipped - Varnish not available');
                return;
            }

            if (self::$is_initialized) {
                CacheSys::writeError('Varnish: Already initialized, skipping duplicate initialization');
                return;
            }

            // Register purge hooks with comprehensive error handling
            static::addAction('zizi_cache_purge_url:before', 'purgeVarnishByUrl', 10, 1);
            static::addAction('zizi_cache_purge_pages:before', 'purgeVarnishCache', 10, 0);
            static::addAction('zizi_cache_purge_everything:before', 'purgeVarnishCache', 10, 0);
            
            // Register maintenance and monitoring hooks
            static::addAction('wp_loaded', 'validateVarnishIntegration', 15);
            static::addAction('admin_init', 'adminIntegration', 10);
            static::addAction('wp_footer', 'performanceReporting', 999);
            
            // Register periodic health checks
            static::addAction('zizi_cache_maintenance', 'performHealthCheck', 10);

            // Initialize performance metrics
            self::$performance_metrics = [
                'init_time' => microtime(true) - $start_time,
                'purge_requests' => 0,
                'network_errors' => 0,
                'timeouts' => 0
            ];

            self::$is_initialized = true;
            $execution_time = microtime(true) - $start_time;
            
            CacheSys::writeError('Varnish: Integration initialized successfully with ' . 
                count(self::$varnish_servers) . ' servers in ' . 
                round($execution_time * 1000, 2) . 'ms');

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('Varnish: Initialization failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
        }
    }

    /**
     * Detect Varnish by analyzing HTTP response headers
     * 
     * @return bool True if Varnish headers detected
     */
    private static function detectVarnishByHeaders(): bool {
        try {
            // Check current response headers if available
            if (function_exists('headers_list')) {
                $headers = headers_list();
                foreach ($headers as $header) {
                    foreach (self::VARNISH_HEADERS as $varnish_header) {
                        if (stripos($header, $varnish_header . ':') === 0) {
                            CacheSys::writeError('Varnish: Detected via response header - ' . $header);
                            return true;
                        }
                    }
                }
            }

            // Check server environment variables
            foreach (self::VARNISH_HEADERS as $header) {
                $env_key = 'HTTP_' . str_replace('-', '_', strtoupper($header));
                if (!empty($_SERVER[$env_key])) {
                    CacheSys::writeError('Varnish: Detected via environment variable - ' . $env_key . ': ' . $_SERVER[$env_key]);
                    return true;
                }
            }

            return false;

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Header detection failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Detect Varnish by making test request to current URL
     * 
     * @return bool True if Varnish detected in response
     */
    private static function detectVarnishByResponse(): bool {
        try {
            if (!isset($_SERVER['HTTP_HOST']) || !isset($_SERVER['REQUEST_URI'])) {
                return false;
            }

            $test_url = 'http://' . $_SERVER['HTTP_HOST'] . '/';
            $context = stream_context_create([
                'http' => [
                    'method' => 'HEAD',
                    'timeout' => 3,
                    'ignore_errors' => true
                ]
            ]);

            $response = @file_get_contents($test_url, false, $context);
            
            if ($http_response_header) {
                foreach ($http_response_header as $header) {
                    foreach (self::VARNISH_HEADERS as $varnish_header) {
                        if (stripos($header, $varnish_header . ':') === 0) {
                            CacheSys::writeError('Varnish: Detected via test request header - ' . $header);
                            return true;
                        }
                    }
                }
            }

            return false;

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Response detection failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Detect Varnish by configuration or environment settings
     * 
     * @return bool True if Varnish configured
     */
    private static function detectVarnishByConfig(): bool {
        try {
            // Check for Varnish-specific environment variables
            $varnish_env_vars = ['VARNISH_SERVER', 'VARNISH_HOST', 'CACHE_SERVER'];
            
            foreach ($varnish_env_vars as $env_var) {
                if (!empty($_SERVER[$env_var])) {
                    CacheSys::writeError('Varnish: Detected via environment configuration - ' . $env_var);
                    return true;
                }
            }

            // Check for WordPress-specific Varnish plugins
            if (defined('VARNISH_PURGE_URL') || defined('VARNISH_SERVER')) {
                CacheSys::writeError('Varnish: Detected via WordPress configuration constants');
                return true;
            }

            return false;

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Configuration detection failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Initialize Varnish servers configuration
     * 
     * @return void
     */
    private static function initializeVarnishServers(): void {
        try {
            self::$varnish_servers = [];

            // Add localhost as default Varnish server
            $default_server = [
                'host' => 'localhost',
                'port' => self::DEFAULT_VARNISH_PORT,
                'timeout' => self::DEFAULT_TIMEOUT,
                'active' => true
            ];

            // Check for custom server configuration
            if (defined('VARNISH_SERVER')) {
                $custom_server = self::parseVarnishServer(VARNISH_SERVER);
                if ($custom_server) {
                    self::$varnish_servers[] = $custom_server;
                    CacheSys::writeError('Varnish: Added custom server - ' . VARNISH_SERVER);
                }
            } else {
                self::$varnish_servers[] = $default_server;
            }

            // Initialize failed servers tracking
            self::$failed_servers = [];

            CacheSys::writeError('Varnish: Initialized ' . count(self::$varnish_servers) . ' server(s)');

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Server initialization failed - ' . $e->getMessage());
            self::$varnish_servers = [];
        }
    }

    /**
     * Parse Varnish server configuration string
     * 
     * @param string $server_config Server configuration (host:port format)
     * @return array|null Parsed server configuration or null on failure
     */
    private static function parseVarnishServer($server_config): ?array {
        try {
            if (!is_string($server_config) || empty($server_config)) {
                return null;
            }

            // Basic format validation
            if (!preg_match('/^[a-zA-Z0-9.-]+(?::\d+)?$/', $server_config)) {
                CacheSys::writeError('Varnish: Invalid server format - ' . $server_config);
                return null;
            }

            $parts = explode(':', $server_config);
            $host = $parts[0];
            $port = isset($parts[1]) ? (int)$parts[1] : self::DEFAULT_VARNISH_PORT;

            // Validate host and port
            if (empty($host) || $port <= 0 || $port > 65535) {
                return null;
            }

            return [
                'host' => $host,
                'port' => $port,
                'timeout' => self::DEFAULT_TIMEOUT,
                'active' => true
            ];

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Server parsing failed - ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Purge specific URL from Varnish cache
     * 
     * Performs secure PURGE operation for specific URL with comprehensive
     * error handling, security validation, and performance monitoring.
     * 
     * @param string $url URL to purge from cache
     * @return void
     */
    public static function purgeVarnishByUrl($url): void {
        $start_time = microtime(true);
        
        try {
            if (!self::validateUrl($url)) {
                CacheSys::writeError('Varnish: Invalid URL provided for purge - ' . (is_string($url) ? $url : gettype($url)));
                return;
            }

            if (empty(self::$varnish_servers)) {
                CacheSys::writeError('Varnish: No Varnish servers configured for URL purge');
                return;
            }

            $successful_purges = 0;
            $total_servers = count(self::$varnish_servers);

            foreach (self::$varnish_servers as $server_id => $server) {
                if (!$server['active']) {
                    continue;
                }

                try {
                    $purge_success = self::executePurgeRequest($server, $url);
                    
                    if ($purge_success) {
                        $successful_purges++;
                        self::$purge_statistics['successful_purges']++;
                    } else {
                        self::$purge_statistics['failed_purges']++;
                        self::markServerAsFailed($server_id);
                    }

                } catch (\Exception $e) {
                    CacheSys::writeError('Varnish: PURGE failed for server ' . $server['host'] . ':' . $server['port'] . ' - ' . $e->getMessage());
                    self::$purge_statistics['failed_purges']++;
                    self::markServerAsFailed($server_id);
                }
            }

            self::$performance_metrics['purge_requests']++;
            self::$purge_statistics['total_requests']++;
            
            $execution_time = microtime(true) - $start_time;
            self::updateAverageResponseTime($execution_time);

            if ($successful_purges > 0) {
                CacheSys::writeError('Varnish: Successfully purged URL "' . $url . '" from ' . $successful_purges . '/' . $total_servers . ' servers in ' . round($execution_time * 1000, 2) . 'ms');
            } else {
                CacheSys::writeError('Varnish: Failed to purge URL "' . $url . '" from any server (' . $total_servers . ' attempted) in ' . round($execution_time * 1000, 2) . 'ms');
            }

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('Varnish: URL purge operation failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            self::$performance_metrics['network_errors']++;
        }
    }

    /**
     * Purge entire Varnish cache
     * 
     * Performs full cache purge with batch processing and performance optimization.
     * 
     * @return void
     */
    public static function purgeVarnishCache(): void {
        $start_time = microtime(true);
        
        try {
            if (empty(self::$varnish_servers)) {
                CacheSys::writeError('Varnish: No Varnish servers configured for full cache purge');
                return;
            }

            $successful_purges = 0;
            $total_servers = count(self::$varnish_servers);

            // Purge from all active servers
            foreach (self::$varnish_servers as $server_id => $server) {
                if (!$server['active']) {
                    continue;
                }

                try {
                    // Use wildcard purge for full cache
                    $purge_success = self::executePurgeRequest($server, '.*', true);
                    
                    if ($purge_success) {
                        $successful_purges++;
                        self::$purge_statistics['successful_purges']++;
                    } else {
                        self::$purge_statistics['failed_purges']++;
                        self::markServerAsFailed($server_id);
                    }

                } catch (\Exception $e) {
                    CacheSys::writeError('Varnish: Full cache purge failed for server ' . $server['host'] . ':' . $server['port'] . ' - ' . $e->getMessage());
                    self::$purge_statistics['failed_purges']++;
                    self::markServerAsFailed($server_id);
                }
            }

            self::$performance_metrics['purge_requests']++;
            self::$purge_statistics['total_requests']++;
            
            $execution_time = microtime(true) - $start_time;
            self::updateAverageResponseTime($execution_time);

            if ($successful_purges > 0) {
                CacheSys::writeError('Varnish: Successfully purged full cache from ' . $successful_purges . '/' . $total_servers . ' servers in ' . round($execution_time * 1000, 2) . 'ms');
            } else {
                CacheSys::writeError('Varnish: Failed to purge full cache from any server (' . $total_servers . ' attempted) in ' . round($execution_time * 1000, 2) . 'ms');
            }

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('Varnish: Full cache purge operation failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            self::$performance_metrics['network_errors']++;
        }
    }

    /**
     * Execute PURGE request to Varnish server
     * 
     * @param array $server Server configuration
     * @param string $url URL to purge
     * @param bool $wildcard Whether to use wildcard purge
     * @return bool True on successful purge
     */
    private static function executePurgeRequest($server, $url, $wildcard = false): bool {
        try {
            $purge_url = 'http://' . $server['host'] . ':' . $server['port'] . ($wildcard ? '.*' : $url);
            
            $context = stream_context_create([
                'http' => [
                    'method' => self::PURGE_METHOD,
                    'timeout' => $server['timeout'],
                    'ignore_errors' => true,
                    'header' => [
                        'Host: ' . ($_SERVER['HTTP_HOST'] ?? 'localhost'),
                        'User-Agent: ZiziCache/2.0.0 Varnish Integration'
                    ]
                ]
            ]);

            $response = @file_get_contents($purge_url, false, $context);
            
            // Check HTTP response code
            if (isset($http_response_header) && is_array($http_response_header)) {
                $status_line = $http_response_header[0];
                if (strpos($status_line, '200') !== false || strpos($status_line, '204') !== false) {
                    return true;
                }
                
                CacheSys::writeError('Varnish: PURGE request returned non-success status - ' . $status_line);
            }

            return false;

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: PURGE request execution failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Validate URL for security and format
     * 
     * @param mixed $url URL to validate
     * @return bool True if URL is valid and safe
     */
    private static function validateUrl($url): bool {
        try {
            if (!is_string($url) || empty($url)) {
                return false;
            }

            // Basic format validation
            if (!filter_var($url, FILTER_VALIDATE_URL) && !preg_match('/^\/[^\s]*$/', $url)) {
                return false;
            }

            // Security validation - prevent SSRF
            $parsed_url = parse_url($url);
            if ($parsed_url && isset($parsed_url['host'])) {
                $host = $parsed_url['host'];
                
                // Block private IP ranges
                if (filter_var($host, FILTER_VALIDATE_IP)) {
                    if (!filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                        CacheSys::writeError('Varnish: Blocked private/reserved IP address - ' . $host);
                        return false;
                    }
                }
            }

            // Length validation
            if (strlen($url) > 2048) {
                CacheSys::writeError('Varnish: URL too long for purge - ' . strlen($url) . ' characters');
                return false;
            }

            return true;

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: URL validation failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Mark server as failed and implement circuit breaker
     * 
     * @param int $server_id Server identifier
     * @return void
     */
    private static function markServerAsFailed($server_id): void {
        try {
            if (!isset(self::$failed_servers[$server_id])) {
                self::$failed_servers[$server_id] = 0;
            }

            self::$failed_servers[$server_id]++;

            // Implement circuit breaker - disable server after multiple failures
            if (self::$failed_servers[$server_id] >= self::MAX_RETRY_ATTEMPTS) {
                self::$varnish_servers[$server_id]['active'] = false;
                CacheSys::writeError('Varnish: Server ' . self::$varnish_servers[$server_id]['host'] . ' disabled after ' . self::MAX_RETRY_ATTEMPTS . ' failures');
            }

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Failed to mark server as failed - ' . $e->getMessage());
        }
    }

    /**
     * Update average response time metrics
     * 
     * @param float $execution_time Execution time in seconds
     * @return void
     */
    private static function updateAverageResponseTime($execution_time): void {
        try {
            $current_avg = self::$purge_statistics['avg_response_time'];
            $total_requests = self::$purge_statistics['total_requests'];
            
            if ($total_requests > 1) {
                self::$purge_statistics['avg_response_time'] = (($current_avg * ($total_requests - 1)) + $execution_time) / $total_requests;
            } else {
                self::$purge_statistics['avg_response_time'] = $execution_time;
            }

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Failed to update response time metrics - ' . $e->getMessage());
        }
    }

    /**
     * Validate Varnish integration health
     * 
     * @return void
     */
    public static function validateVarnishIntegration(): void {
        try {
            if (!self::$is_initialized) {
                return;
            }

            $health_status = [
                'servers_configured' => count(self::$varnish_servers),
                'active_servers' => count(array_filter(self::$varnish_servers, function($server) { return $server['active']; })),
                'failed_servers' => count(self::$failed_servers),
                'total_purge_requests' => self::$purge_statistics['total_requests'],
                'success_rate' => self::$purge_statistics['total_requests'] > 0 ? 
                    round((self::$purge_statistics['successful_purges'] / self::$purge_statistics['total_requests']) * 100, 2) : 0
            ];

            CacheSys::writeError('Varnish: Integration health check - ' . json_encode($health_status));

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Health validation failed - ' . $e->getMessage());
        }
    }

    /**
     * Perform periodic health check
     * 
     * @return void
     */
    public static function performHealthCheck(): void {
        try {
            foreach (self::$varnish_servers as $server_id => $server) {
                if (!$server['active']) {
                    continue;
                }

                // Simple connectivity test
                $connection = @fsockopen($server['host'], $server['port'], $errno, $errstr, 3);
                if ($connection) {
                    fclose($connection);
                    // Reset failure count on successful connection
                    if (isset(self::$failed_servers[$server_id])) {
                        unset(self::$failed_servers[$server_id]);
                    }
                } else {
                    self::markServerAsFailed($server_id);
                    CacheSys::writeError('Varnish: Health check failed for server ' . $server['host'] . ':' . $server['port'] . ' - ' . $errstr);
                }
            }

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Health check failed - ' . $e->getMessage());
        }
    }

    /**
     * Admin integration and user notices
     * 
     * @return void
     */
    public static function adminIntegration(): void {
        try {
            if (!current_user_can('manage_options')) {
                return;
            }

            // Add admin notice if no active Varnish servers
            $active_servers = array_filter(self::$varnish_servers, function($server) { return $server['active']; });
            if (empty($active_servers) && !empty(self::$varnish_servers)) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-warning"><p>';
                    echo '<strong>ZiziCache Varnish Integration:</strong> All Varnish servers are currently unavailable. Cache purging may not work properly.';
                    echo '</p></div>';
                });
            }

        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Admin integration failed - ' . $e->getMessage());
        }
    }

    /**
     * Performance reporting
     * 
     * @return void
     */
    public static function performanceReporting(): void {
        try {
            if (!empty(self::$performance_metrics) || !empty(self::$purge_statistics)) {
                $report = array_merge(self::$performance_metrics, self::$purge_statistics);
                CacheSys::writeError('Varnish: Performance metrics - ' . json_encode($report));
            }
        } catch (\Exception $e) {
            CacheSys::writeError('Varnish: Performance reporting failed - ' . $e->getMessage());
        }
    }

    /**
     * Legacy method compatibility (deprecated)
     * 
     * @deprecated Use checkAvailability() instead
     * @return bool
     */
    public static function is_varnish_enabled(): bool {
        CacheSys::writeError('Varnish: Warning - is_varnish_enabled() is deprecated, use checkAvailability() instead');
        return static::checkAvailability();
    }
}
