<?php
/**
 * TranslatePress Integration for ZiziCache
 * 
 * Enterprise-grade integration with TranslatePress plugin for multilingual cache management.
 * Provides comprehensive URL handling, language-specific cache coordination, and robust
 * error handling for production environments.
 * 
 * @package ZiziCache
 * @subpackage Plugins\Integrations
 * @version 2.0.0
 * @since 1.0.0
 * @author ZiziCache Team
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\Helpers\CacheSys;

/**
 * TranslatePress Integration Class
 * 
 * Handles multilingual cache management for TranslatePress plugin including:
 * - Language-specific URL generation and cache coordination
 * - Request URI conversion for proper cache key generation
 * - Cache purging coordination with language switching
 * - Performance monitoring and structured logging
 */
class TranslatePress extends PluginBase {
    use PluginTrait;

    /**
     * Configuration constants
     */
    private const PLUGIN_CLASS = 'TRP_Translate_Press';
    private const URL_CONVERTER_COMPONENT = 'url_converter';
    private const MAX_LANGUAGE_PROCESSING_TIME = 2.0; // seconds
    private const URL_BATCH_SIZE = 50;

    /**
     * Internal state management
     */
    private static $url_converter = null;
    private static $trp_language = null;
    private static $is_initialized = false;
    private static $supported_languages = [];
    private static $performance_metrics = [];

    /**
     * Check if TranslatePress plugin is available and properly configured
     * 
     * Implements abstract method from PluginBase with comprehensive validation
     * including plugin existence, required components, and configuration state.
     * 
     * @return bool True if TranslatePress is available and functional
     */
    protected static function checkAvailability(): bool {
        $start_time = microtime(true);
        
        try {
            // Basic class existence check
            if (!class_exists(self::PLUGIN_CLASS)) {
                CacheSys::writeError('TranslatePress: Plugin class not found - ' . self::PLUGIN_CLASS);
                return false;
            }

            // Check if TranslatePress is properly initialized
            if (!function_exists('trp_get_languages')) {
                CacheSys::writeError('TranslatePress: Core functions not available');
                return false;
            }

            // Verify required components availability
            try {
                $trp_instance = \TRP_Translate_Press::get_trp_instance();
                if (!$trp_instance) {
                    CacheSys::writeError('TranslatePress: Failed to get plugin instance');
                    return false;
                }

                $url_converter = $trp_instance->get_component(self::URL_CONVERTER_COMPONENT);
                if (!$url_converter) {
                    CacheSys::writeError('TranslatePress: URL converter component not available');
                    return false;
                }
            } catch (\Exception $e) {
                CacheSys::writeError('TranslatePress: Component validation failed - ' . $e->getMessage());
                return false;
            }

            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: Availability check completed successfully in ' . round($execution_time * 1000, 2) . 'ms');
            
            return true;

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: Availability check failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Initialize TranslatePress integration
     * 
     * Sets up comprehensive integration including hook registration,
     * component initialization, and performance monitoring setup.
     * 
     * @return void
     */
    public static function init(): void {
        $start_time = microtime(true);
        
        try {
            if (!static::checkAvailability()) {
                CacheSys::writeError('TranslatePress: Initialization skipped - plugin not available');
                return;
            }

            if (self::$is_initialized) {
                CacheSys::writeError('TranslatePress: Already initialized, skipping duplicate initialization');
                return;
            }

            // Register integration hooks with priority management
            static::addAction('init', 'setupIntegration', 20); // Late init for proper TranslatePress setup
            static::addAction('wp_loaded', 'validateIntegration', 10);
            static::addAction('admin_init', 'adminIntegration', 10);
            static::addAction('wp_footer', 'performanceReporting', 999);

            // Register cache coordination hooks
            static::addAction('trp_language_switched', 'onLanguageSwitched', 10);
            
            self::$is_initialized = true;
            $execution_time = microtime(true) - $start_time;
            
            CacheSys::writeError('TranslatePress: Integration initialized successfully in ' . round($execution_time * 1000, 2) . 'ms');

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: Initialization failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
        }
    }

    /**
     * Setup comprehensive TranslatePress integration
     * 
     * Initializes URL converter, language detection, and cache coordination
     * with comprehensive error handling and performance monitoring.
     * 
     * @return void
     */
    public static function setupIntegration(): void {
        $start_time = microtime(true);
        
        try {
            global $TRP_LANGUAGE;

            // Initialize TranslatePress components with validation
            $trp_instance = \TRP_Translate_Press::get_trp_instance();
            if (!$trp_instance) {
                throw new \Exception('Failed to get TranslatePress instance');
            }

            self::$url_converter = $trp_instance->get_component(self::URL_CONVERTER_COMPONENT);
            if (!self::$url_converter) {
                throw new \Exception('Failed to get URL converter component');
            }

            // Language detection and validation
            self::$trp_language = $TRP_LANGUAGE ?? null;
            if (!self::$trp_language) {
                CacheSys::writeError('TranslatePress: Current language not detected, using default');
            }

            // Initialize supported languages cache
            self::initializeSupportedLanguages();

            // Register cache coordination filters with comprehensive error handling
            add_filter('zizi_cache_preload_urls', [__CLASS__, 'addTranslatedUrls'], 10, 1);
            add_filter('zizi_cache_auto_purge_urls', [__CLASS__, 'addTranslatedUrls'], 10, 1);
            add_filter('zizi_cache_request_uri', [__CLASS__, 'convertRequestUri'], 10, 1);
            add_filter('zizi_cache_cache_key', [__CLASS__, 'modifyCacheKey'], 10, 2);

            // Performance monitoring setup
            self::$performance_metrics['setup_time'] = microtime(true) - $start_time;
            self::$performance_metrics['url_conversions'] = 0;
            self::$performance_metrics['cache_hits'] = 0;

            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: Integration setup completed successfully in ' . round($execution_time * 1000, 2) . 'ms with ' . count(self::$supported_languages) . ' languages detected');

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: Integration setup failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            
            // Graceful degradation - disable integration on failure
            self::disableIntegration();
        }
    }

    /**
     * Initialize supported languages cache
     * 
     * @return void
     */
    private static function initializeSupportedLanguages(): void {
        try {
            if (function_exists('trp_get_languages')) {
                $languages = trp_get_languages();
                self::$supported_languages = is_array($languages) ? $languages : [];
                CacheSys::writeError('TranslatePress: Initialized ' . count(self::$supported_languages) . ' supported languages');
            } else {
                CacheSys::writeError('TranslatePress: trp_get_languages function not available');
                self::$supported_languages = [];
            }
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Failed to initialize supported languages - ' . $e->getMessage());
            self::$supported_languages = [];
        }
    }

    /**
     * Add translated URLs to cache preload and purge operations
     * 
     * Generates language-specific URLs for comprehensive cache management
     * with batch processing and performance optimization.
     * 
     * @param array $urls Original URLs to process
     * @return array Enhanced URLs with language variations
     */
    public static function addTranslatedUrls($urls): array {
        $start_time = microtime(true);
        
        try {
            if (!is_array($urls)) {
                CacheSys::writeError('TranslatePress: Invalid URLs parameter type - expected array, got ' . gettype($urls));
                return [];
            }

            if (!self::$url_converter || empty(self::$supported_languages)) {
                CacheSys::writeError('TranslatePress: URL converter or languages not available for URL translation');
                return $urls;
            }

            $translated_urls = [];
            $processed_count = 0;
            $batch_start_time = microtime(true);

            // Process URLs in batches for performance
            $url_batches = array_chunk($urls, self::URL_BATCH_SIZE);
            
            foreach ($url_batches as $batch) {
                foreach ($batch as $url) {
                    if (!is_string($url) || empty($url)) {
                        continue;
                    }

                    // Add original URL
                    $translated_urls[] = $url;

                    // Generate translated variations
                    foreach (self::$supported_languages as $language_code) {
                        try {
                            $translated_url = self::$url_converter->get_url_for_language($language_code, $url);
                            if ($translated_url && $translated_url !== $url) {
                                $translated_urls[] = $translated_url;
                            }
                        } catch (\Exception $e) {
                            CacheSys::writeError('TranslatePress: Failed to translate URL "' . $url . '" for language "' . $language_code . '" - ' . $e->getMessage());
                        }
                    }

                    $processed_count++;
                }

                // Performance monitoring for batch processing
                $batch_time = microtime(true) - $batch_start_time;
                if ($batch_time > self::MAX_LANGUAGE_PROCESSING_TIME) {
                    CacheSys::writeError('TranslatePress: Batch processing exceeded time limit (' . round($batch_time, 2) . 's), stopping early to prevent performance issues');
                    break;
                }
            }

            // Remove duplicates and sort for consistency
            $translated_urls = array_unique($translated_urls);
            sort($translated_urls);

            self::$performance_metrics['url_conversions'] += $processed_count;
            $execution_time = microtime(true) - $start_time;

            CacheSys::writeError('TranslatePress: Successfully processed ' . $processed_count . ' URLs into ' . count($translated_urls) . ' translated variants in ' . round($execution_time * 1000, 2) . 'ms');

            return $translated_urls;

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: URL translation failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            return is_array($urls) ? $urls : [];
        }
    }

    /**
     * Convert request URI for language-specific cache key generation
     * 
     * Ensures proper cache isolation between languages with fallback handling
     * and performance optimization.
     * 
     * @param string $uri Original request URI
     * @return string Converted URI with language context
     */
    public static function convertRequestUri($uri): string {
        $start_time = microtime(true);
        
        try {
            if (!is_string($uri)) {
                CacheSys::writeError('TranslatePress: Invalid URI parameter type - expected string, got ' . gettype($uri));
                return '';
            }

            if (!self::$url_converter) {
                CacheSys::writeError('TranslatePress: URL converter not available for URI conversion');
                return $uri;
            }

            // Get current language context
            $current_language = self::getCurrentLanguage();
            if (!$current_language) {
                CacheSys::writeError('TranslatePress: Current language not detected for URI conversion');
                return $uri;
            }

            // Convert URI with language context
            $converted_uri = self::$url_converter->get_abs_home() . $uri;
            $language_specific_uri = self::$url_converter->get_url_for_language($current_language, $converted_uri);

            if ($language_specific_uri) {
                // Extract relative path from absolute URL
                $parsed_url = parse_url($language_specific_uri);
                $final_uri = $parsed_url['path'] ?? $uri;
                
                // Add query parameters if present
                if (!empty($parsed_url['query'])) {
                    $final_uri .= '?' . $parsed_url['query'];
                }
            } else {
                $final_uri = $uri;
                CacheSys::writeError('TranslatePress: Failed to convert URI, using original - ' . $uri);
            }

            $execution_time = microtime(true) - $start_time;
            
            if ($final_uri !== $uri) {
                CacheSys::writeError('TranslatePress: URI converted from "' . $uri . '" to "' . $final_uri . '" for language "' . $current_language . '" in ' . round($execution_time * 1000, 2) . 'ms');
            }

            return $final_uri;

        } catch (\Exception $e) {
            $execution_time = microtime(true) - $start_time;
            CacheSys::writeError('TranslatePress: URI conversion failed after ' . round($execution_time * 1000, 2) . 'ms - ' . $e->getMessage());
            return is_string($uri) ? $uri : '';
        }
    }

    /**
     * Modify cache key to include language context
     * 
     * @param string $cache_key Original cache key
     * @param string $uri Request URI
     * @return string Modified cache key with language suffix
     */
    public static function modifyCacheKey($cache_key, $uri): string {
        try {
            $current_language = self::getCurrentLanguage();
            if ($current_language && $current_language !== 'en') { // Don't modify default language
                $cache_key .= '_lang_' . $current_language;
                self::$performance_metrics['cache_hits']++;
            }
            return $cache_key;
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Cache key modification failed - ' . $e->getMessage());
            return $cache_key;
        }
    }

    /**
     * Handle language switching events
     * 
     * @return void
     */
    public static function onLanguageSwitched(): void {
        try {
            CacheSys::writeError('TranslatePress: Language switched detected, updating cache coordination');
            
            // Update current language cache
            global $TRP_LANGUAGE;
            self::$trp_language = $TRP_LANGUAGE;
            
            // Trigger cache purge if necessary
            do_action('zizi_cache_language_switched', self::$trp_language);
            
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Language switch handling failed - ' . $e->getMessage());
        }
    }

    /**
     * Get current language with fallback handling
     * 
     * @return string|null Current language code or null if not available
     */
    private static function getCurrentLanguage(): ?string {
        try {
            // Try multiple methods to get current language
            global $TRP_LANGUAGE;
            
            if (self::$trp_language) {
                return self::$trp_language;
            }
            
            if ($TRP_LANGUAGE) {
                return $TRP_LANGUAGE;
            }
            
            if (function_exists('trp_get_current_language')) {
                return trp_get_current_language();
            }
            
            return null;
            
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Failed to get current language - ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Validate integration health
     * 
     * @return void
     */
    public static function validateIntegration(): void {
        try {
            if (!self::$is_initialized) {
                CacheSys::writeError('TranslatePress: Integration validation skipped - not initialized');
                return;
            }

            $health_status = [
                'url_converter_available' => (bool) self::$url_converter,
                'languages_detected' => count(self::$supported_languages),
                'current_language' => self::getCurrentLanguage(),
                'hooks_registered' => has_filter('zizi_cache_preload_urls', [__CLASS__, 'addTranslatedUrls'])
            ];

            CacheSys::writeError('TranslatePress: Integration health check - ' . json_encode($health_status));

        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Integration validation failed - ' . $e->getMessage());
        }
    }

    /**
     * Admin integration and user notices
     * 
     * @return void
     */
    public static function adminIntegration(): void {
        try {
            if (!current_user_can('manage_options')) {
                return;
            }

            // Add admin notices for configuration issues
            if (self::checkAvailability() && empty(self::$supported_languages)) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-warning"><p>';
                    echo '<strong>ZiziCache TranslatePress Integration:</strong> No languages detected. Please configure TranslatePress languages for optimal cache performance.';
                    echo '</p></div>';
                });
            }

        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Admin integration failed - ' . $e->getMessage());
        }
    }

    /**
     * Performance reporting
     * 
     * @return void
     */
    public static function performanceReporting(): void {
        try {
            if (!empty(self::$performance_metrics)) {
                CacheSys::writeError('TranslatePress: Performance metrics - ' . json_encode(self::$performance_metrics));
            }
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Performance reporting failed - ' . $e->getMessage());
        }
    }

    /**
     * Disable integration on critical failures
     * 
     * @return void
     */
    private static function disableIntegration(): void {
        try {
            remove_all_filters('zizi_cache_preload_urls');
            remove_all_filters('zizi_cache_auto_purge_urls');
            remove_all_filters('zizi_cache_request_uri');
            remove_all_filters('zizi_cache_cache_key');
            
            self::$is_initialized = false;
            CacheSys::writeError('TranslatePress: Integration disabled due to critical failures');
            
        } catch (\Exception $e) {
            CacheSys::writeError('TranslatePress: Failed to disable integration - ' . $e->getMessage());
        }
    }
}
