<?php
/**
 * ShortPixel Adaptive Images Integration - Enterprise Grade
 * 
 * Comprehensive integration with ShortPixel Adaptive Images providing:
 * - Coordinated lazy loading management
 * - Image optimization conflict resolution
 * - Performance monitoring and reporting
 * - Enterprise-grade error handling and logging
 * 
 * @package ZiziCache\Plugins\Integrations
 * @version 2.0.0
 * @author ZiziCache Team
 * @since 2.0.0
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\CacheSys;

class ShortPixelAI extends PluginBase {
    use PluginTrait;

    /**
     * ShortPixel AI settings that conflict with ZiziCache
     */
    private const CONFLICTING_SETTINGS = [
        'eager_selectors_path' => ['settings', 'exclusions'],
        'alter2wh_path' => ['settings', 'behaviour'],
        'alter2wh_value' => 0
    ];

    /**
     * Check if ShortPixel Adaptive Images plugin is available
     * 
     * @return bool True if ShortPixel AI is available, false otherwise
     */
    protected static function checkAvailability(): bool {
        return defined('SHORTPIXEL_AI_VERSION');
    }

    /**
     * Initialize the ShortPixel AI integration
     * 
     * @return void
     */
    public static function init(): void {
        try {
            // Register hooks using the parent's secure method
            static::addAction('zizi_cache_update_config:after', 'disable_conflicting_settings');
            
            // Track initialization for monitoring
            $version = defined('SHORTPIXEL_AI_VERSION') ? SHORTPIXEL_AI_VERSION : 'unknown';
            
            CacheSys::writeError("ShortPixel AI integration initialized successfully", 'ShortPixelAI', [
                'shortpixel_version' => $version,
                'integration_version' => static::getVersion()
            ]);
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to initialize ShortPixel AI integration: " . $e->getMessage(), 'ShortPixelAI', [
                'error_code' => $e->getCode(),
                'error_file' => $e->getFile(),
                'error_line' => $e->getLine()
            ]);
        }
    }

    /**
     * Disable conflicting settings between ZiziCache and ShortPixel AI
     * 
     * This method prevents conflicts with lazy loading and image dimension 
     * alterations by configuring ShortPixel AI to work harmoniously with ZiziCache.
     * 
     * @return void
     */
    public static function disable_conflicting_settings(): void {
        $start_time = microtime(true);
        
        try {
            // Early return if ShortPixel AI is not available
            if (!defined('SHORTPIXEL_AI_VERSION')) {
                CacheSys::writeWarning("ShortPixel AI not detected, skipping configuration", 'ShortPixelAI');
                return;
            }

            // Validate ShortPixel AI Options class availability
            if (!class_exists('ShortPixel\\AI\\Options')) {
                CacheSys::writeWarning("ShortPixel AI Options class not found", 'ShortPixelAI', [
                    'shortpixel_version' => SHORTPIXEL_AI_VERSION
                ]);
                return;
            }

            // Get ShortPixel AI options instance with validation
            $options = \ShortPixel\AI\Options::_();
            
            if (!$options) {
                CacheSys::writeError("Failed to get ShortPixel AI Options instance", 'ShortPixelAI');
                return;
            }

            // Validate required methods exist
            if (!method_exists($options, 'set')) {
                CacheSys::writeError("ShortPixel AI Options 'set' method not found", 'ShortPixelAI', [
                    'options_class' => get_class($options)
                ]);
                return;
            }

            // Apply configuration changes with error handling
            static::applyConfigurationChanges($options);
            
            // Track performance
            $execution_time = (microtime(true) - $start_time) * 1000;
            CacheSys::writeError("ShortPixel AI conflicting settings disabled successfully", 'ShortPixelAI', [
                'execution_time_ms' => round($execution_time, 2),
                'settings_applied' => array_keys(self::CONFLICTING_SETTINGS)
            ]);
            
        } catch (\Throwable $e) {
            $execution_time = (microtime(true) - $start_time) * 1000;
            CacheSys::writeError("Failed to disable ShortPixel AI conflicting settings: " . $e->getMessage(), 'ShortPixelAI', [
                'error_code' => $e->getCode(),
                'error_file' => $e->getFile(),
                'error_line' => $e->getLine(),
                'execution_time_ms' => round($execution_time, 2)
            ]);
        }
    }

    /**
     * Apply specific configuration changes to ShortPixel AI
     * 
     * @param object $options ShortPixel AI Options instance
     * @return void
     * @throws \Exception If configuration application fails
     */
    private static function applyConfigurationChanges($options): void {
        try {
            // Disable lazy loading conflicts
            $options->set('img', 'eager_selectors', self::CONFLICTING_SETTINGS['eager_selectors_path']);
            CacheSys::writeDebug("Applied eager_selectors configuration", 'ShortPixelAI');

            // Disable image dimension alterations
            $options->set(
                self::CONFLICTING_SETTINGS['alter2wh_value'], 
                'alter2wh', 
                self::CONFLICTING_SETTINGS['alter2wh_path']
            );
            CacheSys::writeDebug("Applied alter2wh configuration", 'ShortPixelAI');
            
        } catch (\Throwable $e) {
            throw new \Exception("Configuration application failed: " . $e->getMessage(), 0, $e);
        }
    }

    /**
     * Get the integration version
     * 
     * @return string Version number
     */
    public static function getVersion(): string {
        return '2.0.0';
    }

    /**
     * Get integration description
     * 
     * @return string Description text
     */
    public static function getDescription(): string {
        return 'Enterprise-grade integration with ShortPixel Adaptive Images plugin for optimal image optimization coordination and conflict prevention.';
    }

    /**
     * Get comprehensive metadata about the integration
     * 
     * @return array Metadata including name, version, description and status
     */
    public static function getMetadata(): array {
        $metadata = parent::getMetadata();
        
        $metadata['shortpixel_version'] = defined('SHORTPIXEL_AI_VERSION') ? SHORTPIXEL_AI_VERSION : null;
        $metadata['conflicts_managed'] = array_keys(self::CONFLICTING_SETTINGS);
        $metadata['enterprise_features'] = [
            'structured_logging',
            'error_handling', 
            'performance_monitoring',
            'configuration_validation'
        ];
        
        return $metadata;
    }
}
