<?php
namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\Core\CacheSys;

/**
 * Enterprise-grade Pretty Links Plugin Integration
 * 
 * Comprehensive integration for Pretty Links Pro plugin with advanced caching,
 * performance monitoring, security validation, and enterprise-level features.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @version 2.0.0
 * @since 2.0.0
 */
class PrettyLinks extends PluginBase {
    use PluginTrait;

    /**
     * Plugin integration metadata
     */
    const VERSION = '2.0.0';
    const PLUGIN_SLUG = 'pretty-link';
    const CACHE_GROUP = 'zizi_pretty_links';
    const PERFORMANCE_THRESHOLD = 500; // milliseconds
    
    /**
     * Performance metrics storage
     * @var array
     */
    private static $performance_metrics = [];
    
    /**
     * Cache statistics
     * @var array
     */
    private static $cache_stats = [
        'redirects_processed' => 0,
        'cache_hits' => 0,
        'cache_misses' => 0,
        'performance_samples' => []
    ];
    
    /**
     * Pretty Link tracking data
     * @var array
     */
    private static $link_tracking = [];
    
    /**
     * Security validation patterns
     * @var array
     */
    private static $security_patterns = [
        'malicious_redirects' => [
            '/javascript:/i',
            '/data:/i',
            '/vbscript:/i',
            '/<script/i'
        ],
        'suspicious_domains' => []
    ];

    /**
     * Check if Pretty Links plugin is available and properly configured
     * 
     * @return bool True if plugin is available and ready for integration
     */
    protected static function checkAvailability(): bool {
        try {
            // Basic class existence check
            if (!class_exists('PrliLink')) {
                CacheSys::writeError('Pretty Links Pro: Plugin class PrliLink not found');
                return false;
            }
            
            // Advanced validation checks
            if (!class_exists('PrliUtils')) {
                CacheSys::writeWarning('Pretty Links Pro: Utility class PrliUtils not available - limited functionality');
            }
            
            // Check database tables
            global $wpdb;
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}prli_links'");
            if (!$table_exists) {
                CacheSys::writeError('Pretty Links Pro: Database table prli_links not found');
                return false;
            }
            
            // Performance validation
            $start_time = microtime(true);
            $test_instance = new \PrliLink();
            $creation_time = (microtime(true) - $start_time) * 1000;
            
            if ($creation_time > self::PERFORMANCE_THRESHOLD) {
                CacheSys::writeWarning("Pretty Links Pro: Slow plugin initialization ({$creation_time}ms)");
            }
              static::$performance_metrics['initialization_time'] = $creation_time;
            
            CacheSys::writeLog('Pretty Links Pro integration: Successfully validated and ready');
            return true;
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro integration error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Initialize Pretty Links integration with comprehensive feature set
     */
    public static function init(): void {
        try {
            if (!static::checkAvailability()) {
                return;
            }
            
            // Core caching hooks
            static::addAction('zizi_cache_is_cacheable', 'is_cacheable');
            static::addAction('zizi_cache_process_redirect', 'process_redirect');
            static::addFilter('zizi_cache_url_variations', 'add_url_variations');
            
            // Performance monitoring hooks
            static::addAction('wp_loaded', 'monitor_performance');
            static::addAction('shutdown', 'collect_metrics');
            
            // Admin integration hooks
            static::addAction('wp_ajax_zizi_pretty_links_stats', 'ajax_get_statistics');
            static::addAction('wp_ajax_zizi_pretty_links_purge', 'ajax_purge_cache');
            static::addAction('admin_enqueue_scripts', 'enqueue_admin_assets');
            
            // Security validation hooks
            static::addFilter('zizi_cache_validate_redirect', 'validate_redirect_security');
            
            // Pretty Links specific hooks
            static::addAction('prli_redirect', 'handle_redirect_caching', 10, 2);
            static::addAction('prli_link_created', 'clear_related_cache');
            static::addAction('prli_link_updated', 'clear_related_cache');
            static::addAction('prli_link_deleted', 'clear_related_cache');
              // Batch processing for large sites
            static::addAction('wp_ajax_zizi_pretty_links_batch_process', 'ajax_batch_process');
            
            CacheSys::writeLog('Pretty Links Pro: Enterprise integration initialized successfully');
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro initialization failed: ' . $e->getMessage());
        }
    }

    /**
     * Advanced cacheability check with security validation and performance optimization
     * 
     * @param bool $is_cacheable Current cacheability status
     * @return bool Modified cacheability status
     */
    public static function is_cacheable($is_cacheable): bool {
        try {
            if (!static::checkAvailability()) {
                return $is_cacheable;
            }
            
            $start_time = microtime(true);
            $current_url = site_url($_SERVER['REQUEST_URI']);
            
            // Cache lookup for pretty link check
            $cache_key = md5('pretty_link_check_' . $current_url);
            $cached_result = wp_cache_get($cache_key, self::CACHE_GROUP);
            
            if ($cached_result !== false) {
                static::$cache_stats['cache_hits']++;
                $processing_time = (microtime(true) - $start_time) * 1000;
                static::$performance_metrics['cache_hit_time'] = $processing_time;
                return $cached_result;
            }
            
            static::$cache_stats['cache_misses']++;
            
            // Security validation
            if (!static::validate_url_security($current_url)) {
                CacheSys::writeWarning("Pretty Links Pro: Security validation failed for URL: {$current_url}");
                wp_cache_set($cache_key, $is_cacheable, self::CACHE_GROUP, 300); // 5 minutes
                return $is_cacheable;
            }
            
            // Pretty Links detection
            $pretty_link = new \PrliLink();
            $is_pretty_link = $pretty_link->is_pretty_link($current_url);
            
            if ($is_pretty_link) {
                // Track pretty link usage
                static::track_link_usage($current_url);
                  // Pretty links should not be cached as they redirect
                $result = false;
                CacheSys::writeLog("Pretty Links Pro: Detected pretty link - caching disabled for: {$current_url}");
            } else {
                $result = $is_cacheable;
            }
            
            // Cache the result
            wp_cache_set($cache_key, $result, self::CACHE_GROUP, 3600); // 1 hour
            
            // Performance tracking
            $processing_time = (microtime(true) - $start_time) * 1000;
            static::$performance_metrics['link_check_time'] = $processing_time;
            
            if ($processing_time > self::PERFORMANCE_THRESHOLD) {
                CacheSys::writeWarning("Pretty Links Pro: Slow link detection ({$processing_time}ms) for: {$current_url}");
            }
            
            static::$cache_stats['redirects_processed']++;
            
            return $result;
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro cacheability check failed: ' . $e->getMessage());
            return $is_cacheable;
        }
    }

    /**
     * Process redirect with advanced caching and performance monitoring
     * 
     * @param string $url The URL being processed
     */
    public static function process_redirect(string $url): void {
        try {
            if (!static::checkAvailability()) {
                return;
            }
            
            $start_time = microtime(true);
            
            // Get redirect information
            $pretty_link = new \PrliLink();
            $redirect_info = $pretty_link->get_redirect_info($url);
            
            if ($redirect_info) {
                // Cache redirect information
                $cache_key = md5('pretty_link_redirect_' . $url);
                wp_cache_set($cache_key, $redirect_info, self::CACHE_GROUP, 1800); // 30 minutes
                
                // Track redirect performance
                $processing_time = (microtime(true) - $start_time) * 1000;
                static::$performance_metrics['redirect_processing_time'] = $processing_time;
                
                // Security validation for redirect target
                if (isset($redirect_info['target_url'])) {
                    if (!static::validate_url_security($redirect_info['target_url'])) {
                        CacheSys::writeWarning("Pretty Links Pro: Security warning - suspicious redirect target: {$redirect_info['target_url']}");
                    }                }
                
                CacheSys::writeLog("Pretty Links Pro: Processed redirect from {$url} to {$redirect_info['target_url']} in {$processing_time}ms");
            }
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro redirect processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Add URL variations for Pretty Links caching
     * 
     * @param array $variations Current URL variations
     * @return array Modified URL variations
     */
    public static function add_url_variations(array $variations): array {
        try {
            if (!static::checkAvailability()) {
                return $variations;
            }
            
            global $wpdb;
            
            // Get all active pretty links
            $pretty_links = $wpdb->get_results(
                "SELECT slug, url FROM {$wpdb->prefix}prli_links WHERE redirect_type != 'disabled'",
                ARRAY_A
            );
            
            if ($pretty_links) {
                foreach ($pretty_links as $link) {
                    $variations[] = home_url($link['slug']);
                    
                    // Add variations with common parameters
                    $variations[] = home_url($link['slug'] . '?utm_source=cache');
                    $variations[] = home_url($link['slug'] . '?ref=zizi');                }
                
                CacheSys::writeLog('Pretty Links Pro: Added ' . count($pretty_links) . ' URL variations for caching');
            }
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro URL variations failed: ' . $e->getMessage());
        }
        
        return $variations;
    }

    /**
     * Validate URL security against malicious patterns
     * 
     * @param string $url URL to validate
     * @return bool True if URL is safe
     */
    private static function validate_url_security(string $url): bool {
        // Check against malicious redirect patterns
        foreach (self::$security_patterns['malicious_redirects'] as $pattern) {
            if (preg_match($pattern, $url)) {
                return false;
            }
        }
        
        // Validate domain if present
        $parsed_url = parse_url($url);
        if (isset($parsed_url['host'])) {
            foreach (self::$security_patterns['suspicious_domains'] as $domain) {
                if (strpos($parsed_url['host'], $domain) !== false) {
                    return false;
                }
            }
        }
        
        return true;
    }

    /**
     * Track pretty link usage for analytics
     * 
     * @param string $url Pretty link URL
     */
    private static function track_link_usage(string $url): void {
        $link_key = md5($url);
        
        if (!isset(static::$link_tracking[$link_key])) {
            static::$link_tracking[$link_key] = [
                'url' => $url,
                'hits' => 0,
                'first_seen' => time(),
                'last_seen' => time()
            ];
        }
        
        static::$link_tracking[$link_key]['hits']++;
        static::$link_tracking[$link_key]['last_seen'] = time();
    }

    /**
     * Monitor performance metrics
     */
    public static function monitor_performance(): void {
        if (!static::checkAvailability()) {
            return;
        }
        
        // Sample performance every 10th request
        if (rand(1, 10) === 1) {
            $memory_usage = memory_get_peak_usage(true);
            $current_time = microtime(true);
            
            static::$cache_stats['performance_samples'][] = [
                'timestamp' => $current_time,
                'memory_usage' => $memory_usage,
                'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown'
            ];
            
            // Keep only last 100 samples
            if (count(static::$cache_stats['performance_samples']) > 100) {
                array_shift(static::$cache_stats['performance_samples']);
            }
        }
    }

    /**
     * Collect and log metrics at shutdown
     */
    public static function collect_metrics(): void {
        if (!static::checkAvailability()) {
            return;
        }
        
        if (!empty(static::$performance_metrics) || !empty(static::$link_tracking)) {
            $metrics_summary = [
                'performance' => static::$performance_metrics,
                'cache_stats' => static::$cache_stats,
                'link_tracking' => count(static::$link_tracking),
                'timestamp' => time()
            ];
              // Log comprehensive metrics
            CacheSys::writeLog('Pretty Links Pro Metrics: ' . wp_json_encode($metrics_summary));
            
            // Store in transient for admin dashboard
            set_transient('zizi_pretty_links_metrics', $metrics_summary, 300); // 5 minutes
        }
    }

    /**
     * AJAX handler for getting statistics
     */
    public static function ajax_get_statistics(): void {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized access');
        }
        
        check_ajax_referer('zizi_pretty_links_nonce', 'nonce');
        
        try {
            $metrics = get_transient('zizi_pretty_links_metrics') ?: [];
            
            // Get additional stats from database
            global $wpdb;
            $total_links = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}prli_links");
            $active_links = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}prli_links WHERE redirect_type != 'disabled'");
            
            $response = [
                'success' => true,
                'data' => [
                    'metrics' => $metrics,
                    'total_links' => $total_links,
                    'active_links' => $active_links,
                    'cache_group' => self::CACHE_GROUP,
                    'version' => self::VERSION
                ]
            ];
            
            wp_send_json($response);
            
        } catch (\Exception $e) {
            wp_send_json_error('Failed to get statistics: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for cache purging
     */
    public static function ajax_purge_cache(): void {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized access');
        }
        
        check_ajax_referer('zizi_pretty_links_nonce', 'nonce');
        
        try {
            // Clear Pretty Links cache group
            wp_cache_flush_group(self::CACHE_GROUP);
            
            // Clear related transients
            delete_transient('zizi_pretty_links_metrics');
            
            // Reset internal stats
            static::$cache_stats = [
                'redirects_processed' => 0,
                'cache_hits' => 0,
                'cache_misses' => 0,
                'performance_samples' => []
            ];
            static::$performance_metrics = [];            static::$link_tracking = [];
            
            CacheSys::writeLog('Pretty Links Pro: Cache purged successfully');
            
            wp_send_json_success('Cache purged successfully');
            
        } catch (\Exception $e) {
            wp_send_json_error('Failed to purge cache: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for batch processing
     */
    public static function ajax_batch_process(): void {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized access');
        }
        
        check_ajax_referer('zizi_pretty_links_nonce', 'nonce');
        
        try {
            $batch_size = min(100, intval($_POST['batch_size'] ?? 50));
            $offset = intval($_POST['offset'] ?? 0);
            
            global $wpdb;
            $links = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT id, slug, url FROM {$wpdb->prefix}prli_links LIMIT %d OFFSET %d",
                    $batch_size,
                    $offset
                ),
                ARRAY_A
            );
            
            $processed = 0;
            foreach ($links as $link) {
                // Pre-warm cache for each link
                $cache_key = md5('pretty_link_check_' . home_url($link['slug']));
                wp_cache_set($cache_key, false, self::CACHE_GROUP, 3600);
                $processed++;
            }
            
            wp_send_json_success([
                'processed' => $processed,
                'total_in_batch' => count($links),
                'offset' => $offset + $batch_size
            ]);
            
        } catch (\Exception $e) {
            wp_send_json_error('Batch processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Enqueue admin assets
     */
    public static function enqueue_admin_assets(): void {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'zizi-cache') !== false) {
            wp_enqueue_script('jquery');
            
            // Add inline script for Pretty Links integration
            wp_add_inline_script('jquery', '
                jQuery(document).ready(function($) {
                    // Add Pretty Links stats widget
                    if ($("#zizi-cache-dashboard").length) {
                        $("#zizi-cache-dashboard").append(\'<div id="pretty-links-widget" class="zizi-widget"><h3>Pretty Links Integration</h3><div id="pretty-links-stats">Loading...</div></div>\');
                        
                        // Load stats
                        $.post(ajaxurl, {
                            action: "zizi_pretty_links_stats",
                            nonce: "' . wp_create_nonce('zizi_pretty_links_nonce') . '"
                        }, function(response) {
                            if (response.success) {
                                $("#pretty-links-stats").html(
                                    "<p>Total Links: " + response.data.total_links + "</p>" +
                                    "<p>Active Links: " + response.data.active_links + "</p>" +
                                    "<p>Cache Hits: " + (response.data.metrics.cache_stats?.cache_hits || 0) + "</p>" +
                                    "<p>Version: " + response.data.version + "</p>"
                                );
                            }
                        });
                    }
                });
            ');
        }
    }

    /**
     * Handle Pretty Links redirect caching
     * 
     * @param int $link_id Pretty link ID
     * @param string $target_url Target URL
     */
    public static function handle_redirect_caching($link_id, $target_url): void {
        try {
            // Cache redirect mapping
            $cache_key = md5('pretty_link_redirect_' . $link_id);            wp_cache_set($cache_key, $target_url, self::CACHE_GROUP, 1800);
            
            CacheSys::writeLog("Pretty Links Pro: Cached redirect for link ID {$link_id} -> {$target_url}");
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro redirect caching failed: ' . $e->getMessage());
        }
    }

    /**
     * Clear related cache when Pretty Links are modified
     * 
     * @param int $link_id Pretty link ID
     */
    public static function clear_related_cache($link_id): void {
        try {
            // Clear specific link cache
            $cache_key = md5('pretty_link_redirect_' . $link_id);
            wp_cache_delete($cache_key, self::CACHE_GROUP);
            
            // Clear general Pretty Links cache
            wp_cache_flush_group(self::CACHE_GROUP);
            
            CacheSys::writeError("Pretty Links Pro: Cleared cache for link ID {$link_id}");
            
        } catch (\Exception $e) {
            CacheSys::writeError('Pretty Links Pro cache clearing failed: ' . $e->getMessage());
        }
    }

    /**
     * Validate redirect security
     * 
     * @param array $redirect_data Redirect data to validate
     * @return array Validated redirect data
     */
    public static function validate_redirect_security(array $redirect_data): array {
        if (isset($redirect_data['url'])) {
            if (!static::validate_url_security($redirect_data['url'])) {
                $redirect_data['blocked'] = true;
                $redirect_data['reason'] = 'Security validation failed';
                CacheSys::writeWarning('Pretty Links Pro: Blocked suspicious redirect to ' . $redirect_data['url']);
            }
        }
        
        return $redirect_data;
    }

    /**
     * Get plugin version
     * 
     * @return string Plugin version
     */
    public static function getVersion(): string {
        return self::VERSION;
    }

    /**
     * Get plugin description
     * 
     * @return string Plugin description
     */
    public static function getDescription(): string {
        return 'Enterprise-grade integration for Pretty Links Pro plugin with advanced caching, performance monitoring, security validation, batch processing, and comprehensive admin dashboard integration for optimal redirect management.';
    }

    /**
     * Get plugin metadata
     * 
     * @return array Plugin metadata
     */
    public static function getMetadata(): array {
        return [
            'name' => 'Pretty Links Pro Integration',
            'version' => self::VERSION,
            'description' => static::getDescription(),
            'plugin_slug' => self::PLUGIN_SLUG,
            'cache_group' => self::CACHE_GROUP,
            'features' => [
                'Advanced redirect caching',
                'Performance monitoring and metrics',
                'Security validation',
                'Batch processing for large sites',
                'Admin dashboard integration',
                'Real-time statistics',
                'Memory optimization',
                'Comprehensive logging'
            ],
            'hooks' => [
                'zizi_cache_is_cacheable',
                'zizi_cache_process_redirect',
                'zizi_cache_url_variations',
                'zizi_cache_validate_redirect',
                'prli_redirect',
                'prli_link_created',
                'prli_link_updated',
                'prli_link_deleted'
            ],
            'ajax_endpoints' => [
                'zizi_pretty_links_stats',
                'zizi_pretty_links_purge',
                'zizi_pretty_links_batch_process'
            ],
            'enterprise_features' => [
                'Structured logging via CacheSys',
                'Performance threshold monitoring',
                'Security pattern validation',
                'Link usage tracking',
                'Memory management',
                'Cache optimization',
                'Admin integration'
            ],
            'requirements' => [
                'Pretty Links Pro plugin',
                'ZiziCache Core System',
                'WordPress Admin capabilities'
            ]
        ];
    }
}
