<?php
namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\CacheSys;

/**
 * Polylang Enterprise Integration v2.0.0
 * 
 * Enterprise-grade integration providing comprehensive multilingual cache management
 * with advanced URL translation, intelligent cache coordination, and real-time monitoring.
 * 
 * @package ZiziCache\Plugins\Integrations
 * @version 2.0.0
 * @since 2.0.0
 */
class Polylang extends PluginBase {
    use PluginTrait;

    /**
     * Enterprise configuration settings
     */
    private static $enterprise_config = [
        'batch_size' => 1000,
        'max_translations_per_url' => 50,
        'cache_translations_ttl' => 3600,
        'performance_monitoring' => true,
        'monitoring_sample_rate' => 5, // 5% sampling
    ];

    /**
     * Translation cache storage
     */
    private static $translation_cache = [];

    /**
     * Performance metrics
     */
    private static $performance_metrics = [
        'urls_processed' => 0,
        'translations_generated' => 0,
        'cache_hits' => 0,
        'execution_time' => 0.0,
        'memory_usage' => 0,
    ];

    /**
     * Check plugin availability - architectural compliance
     * 
     * @return bool True if Polylang is available and functional
     */
    protected static function checkAvailability(): bool {
        if (!defined('POLYLANG_VERSION')) {
            return false;
        }

        // Verify essential Polylang functions exist
        $required_functions = [
            'pll_languages_list',
            'pll_home_url', 
            'pll_get_post_translations',
            'pll_get_term_translations'
        ];

        foreach ($required_functions as $function) {
            if (!function_exists($function)) {
                CacheSys::writeError("Polylang integration: Required function '{$function}' not found");
                return false;
            }
        }

        return true;
    }    /**
     * Get enterprise version information
     * 
     * @return string Version string
     */
    public static function getVersion(): string {
        return '2.0.0';
    }

    /**
     * Get enterprise description
     * 
     * @return string Description
     */
    public static function getDescription(): string {
        return 'Enterprise-grade Polylang integration with advanced multilingual cache management, intelligent URL translation, and real-time performance monitoring.';
    }

    /**
     * Get enterprise metadata
     * 
     * @return array Plugin metadata
     */
    public static function getMetadata(): array {
        return [
            'name' => 'Polylang Enterprise Integration',
            'version' => static::getVersion(),
            'description' => static::getDescription(),
            'features' => [
                'intelligent_url_translation',
                'batch_processing',
                'performance_monitoring',
                'cache_optimization',
                'translation_caching',
                'admin_integration'
            ],
            'polylang_compatibility' => [
                'min_version' => '2.8.0',
                'tested_up_to' => '3.7.2',
                'pro_support' => true
            ]
        ];
    }

    /**
     * Initialize enterprise integration
     * 
     * @return void
     */
    public static function init(): void {
        if (!static::checkAvailability()) {
            return;
        }

        try {
            // Core URL translation filters
            add_filter('zizi_cache_preload_urls', [__CLASS__, 'addTranslatedUrls'], 10, 1);
            add_filter('zizi_cache_auto_purge_urls', [__CLASS__, 'addTranslatedUrls'], 10, 1);

            // Enterprise cache management
            add_filter('zizi_cache_cache_file_path', [__CLASS__, 'addLanguageCodeToPath'], 10, 2);
            add_action('pll_save_post', [__CLASS__, 'handleTranslationUpdate'], 10, 3);
            add_action('pll_delete_post', [__CLASS__, 'handleTranslationDeletion'], 10, 2);

            // Admin integration hooks
            if (is_admin() && current_user_can('manage_options')) {
                add_action('wp_dashboard_setup', [__CLASS__, 'addDashboardWidget']);
                add_action('admin_bar_menu', [__CLASS__, 'addAdminBarInfo'], 100);
                add_action('wp_ajax_polylang_integration_status', [__CLASS__, 'handleAjaxStatus']);
                add_action('wp_ajax_polylang_cache_stats', [__CLASS__, 'handleAjaxCacheStats']);
            }

            // Performance monitoring initialization
            if (static::isPerformanceMonitoringEnabled()) {
                add_action('init', [__CLASS__, 'initPerformanceMonitoring'], 1);
            }

            CacheSys::writeError('Polylang enterprise integration initialized successfully');

        } catch (\Exception $e) {
            CacheSys::writeError('Polylang integration initialization failed: ' . $e->getMessage());
        }
    }

    /**
     * Add translated URLs with enterprise features
     * 
     * @param array $urls Original URLs
     * @return array Enhanced URLs with translations
     */
    public static function addTranslatedUrls($urls) {
        if (!static::checkAvailability()) {
            return $urls;
        }

        $start_time = microtime(true);
        $initial_memory = memory_get_usage();

        try {
            $translated_urls = [];
            $batch_count = 0;
            $total_urls = count($urls);

            // Get available languages with caching
            $languages = static::getCachedLanguages();
            if (empty($languages)) {
                CacheSys::writeWarning('Polylang: No languages found');
                return $urls;
            }

            // Add home page translations
            foreach ($languages as $language) {
                try {
                    $home_url = \pll_home_url($language);
                    if ($home_url && $home_url !== home_url()) {
                        $translated_urls[] = $home_url;
                    }
                } catch (\Exception $e) {
                    CacheSys::writeWarning("Polylang: Failed to get home URL for language '{$language}': " . $e->getMessage());
                }
            }

            // Process URLs in batches for better performance
            $url_batches = array_chunk($urls, static::$enterprise_config['batch_size']);
            
            foreach ($url_batches as $batch) {
                $batch_count++;
                $batch_translated = static::processBatch($batch, $languages);
                $translated_urls = array_merge($translated_urls, $batch_translated);

                // Memory management for large batches
                if ($batch_count % 10 === 0) {
                    if (function_exists('gc_collect_cycles')) {
                        gc_collect_cycles();
                    }
                }
            }

            // Performance metrics collection
            $execution_time = microtime(true) - $start_time;
            $memory_used = memory_get_usage() - $initial_memory;
            
            static::updatePerformanceMetrics([
                'urls_processed' => $total_urls,
                'translations_generated' => count($translated_urls) - $total_urls,
                'execution_time' => $execution_time,
                'memory_usage' => $memory_used
            ]);

            $final_urls = array_unique(array_merge($urls, $translated_urls));
            
            if (static::isPerformanceMonitoringEnabled()) {
                static::logPerformanceMetrics($total_urls, count($final_urls));
            }

            return $final_urls;

        } catch (\Exception $e) {
            CacheSys::writeError('Polylang: URL translation failed - ' . $e->getMessage());
            return $urls;
        }
    }

    /**
     * Process URL batch with enhanced error handling
     * 
     * @param array $urls URL batch to process
     * @param array $languages Available languages
     * @return array Translated URLs
     */
    private static function processBatch($urls, $languages) {
        $translated_urls = [];

        foreach ($urls as $url) {
            try {
                $postid = \url_to_postid($url);
                
                if ($postid > 0) {
                    // Process post translations
                    foreach (\pll_get_post_translations($postid) as $translation) {
                        if ($translation && $translation !== $postid) {
                            $permalink = \get_permalink($translation);
                            if ($permalink && !is_wp_error($permalink)) {
                                $translated_urls[] = $permalink;
                            }
                        }
                    }
                    
                    // Process taxonomy term translations
                    static::processTermTranslations($postid, $translated_urls);
                }

            } catch (\Exception $e) {
                CacheSys::writeWarning("Polylang: Failed to process URL '{$url}': " . $e->getMessage());
            }
        }

        return $translated_urls;
    }

    /**
     * Process taxonomy term translations for a post
     * 
     * @param int $post_id Post ID
     * @param array &$translated_urls Reference to translated URLs array
     * @return void
     */
    private static function processTermTranslations($post_id, &$translated_urls) {
        try {
            $post_type = get_post_type($post_id);
            if (!$post_type) {
                return;
            }

            foreach (get_object_taxonomies($post_type) as $taxonomy) {
                $terms = get_the_terms($post_id, $taxonomy);
                if (is_array($terms) && !is_wp_error($terms)) {
                    foreach ($terms as $term) {
                        $term_translations = \pll_get_term_translations($term->term_id);
                        if (is_array($term_translations)) {
                            foreach ($term_translations as $translation) {
                                if ($translation && $translation !== $term->term_id) {
                                    $term_link = \get_term_link($translation);
                                    if ($term_link && !is_wp_error($term_link)) {
                                        $translated_urls[] = $term_link;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            CacheSys::writeWarning("Polylang: Failed to process term translations for post {$post_id}: " . $e->getMessage());
        }
    }

    /**
     * Get cached languages list
     * 
     * @return array Available languages
     */
    private static function getCachedLanguages() {
        $cache_key = 'polylang_languages_' . get_current_blog_id();
        
        if (isset(static::$translation_cache[$cache_key])) {
            static::$performance_metrics['cache_hits']++;
            return static::$translation_cache[$cache_key];
        }

        try {
            $languages = \pll_languages_list();
            if (is_array($languages)) {
                static::$translation_cache[$cache_key] = $languages;
                return $languages;
            }
        } catch (\Exception $e) {
            CacheSys::writeError('Polylang: Failed to get languages list: ' . $e->getMessage());
        }

        return [];
    }

    /**
     * Add language code to cache file path
     * 
     * @param string $path Cache file path
     * @param string $url URL being cached
     * @return string Modified path with language code
     */
    public static function addLanguageCodeToPath($path, $url) {
        if (!static::checkAvailability()) {
            return $path;
        }

        try {
            $current_language = \pll_current_language('slug');
            if ($current_language) {
                $path = str_replace('.html', "_{$current_language}.html", $path);
            }
        } catch (\Exception $e) {
            CacheSys::writeWarning('Polylang: Failed to add language code to path: ' . $e->getMessage());
        }

        return $path;
    }

    /**
     * Handle translation update
     * 
     * @param int $post_id Post ID
     * @param \WP_Post $post Post object
     * @param array $translations Translation data
     * @return void
     */
    public static function handleTranslationUpdate($post_id, $post, $translations) {
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        try {
            // Clear related translation caches
            static::clearTranslationCache($post_id);
            
            CacheSys::writeError("Polylang: Translation updated for post {$post_id}");
        } catch (\Exception $e) {
            CacheSys::writeError('Polylang: Failed to handle translation update: ' . $e->getMessage());
        }
    }

    /**
     * Handle translation deletion
     * 
     * @param int $post_id Post ID
     * @param string $language Language code
     * @return void
     */
    public static function handleTranslationDeletion($post_id, $language) {
        if (!current_user_can('delete_post', $post_id)) {
            return;
        }

        try {
            static::clearTranslationCache($post_id);
            CacheSys::writeError("Polylang: Translation deleted for post {$post_id} in language {$language}");
        } catch (\Exception $e) {
            CacheSys::writeError('Polylang: Failed to handle translation deletion: ' . $e->getMessage());
        }
    }

    /**
     * Clear translation cache for post
     * 
     * @param int $post_id Post ID
     * @return void
     */
    private static function clearTranslationCache($post_id) {
        $cache_key = 'polylang_languages_' . get_current_blog_id();
        unset(static::$translation_cache[$cache_key]);
    }

    /**
     * Add dashboard widget
     * 
     * @return void
     */
    public static function addDashboardWidget() {
        if (!current_user_can('manage_options')) {
            return;
        }

        wp_add_dashboard_widget(
            'polylang_cache_stats',
            'Polylang Cache Statistics',
            [__CLASS__, 'renderDashboardWidget']
        );
    }

    /**
     * Render dashboard widget
     * 
     * @return void
     */
    public static function renderDashboardWidget() {
        $stats = static::$performance_metrics;
        $languages = static::getCachedLanguages();
        
        echo '<div class="polylang-cache-stats">';
        echo '<p><strong>Languages:</strong> ' . count($languages) . '</p>';
        echo '<p><strong>URLs Processed:</strong> ' . $stats['urls_processed'] . '</p>';
        echo '<p><strong>Translations Generated:</strong> ' . $stats['translations_generated'] . '</p>';
        echo '<p><strong>Cache Hits:</strong> ' . $stats['cache_hits'] . '</p>';
        echo '</div>';
    }

    /**
     * Add admin bar information
     * 
     * @param \WP_Admin_Bar $wp_admin_bar WordPress admin bar
     * @return void
     */
    public static function addAdminBarInfo($wp_admin_bar) {
        if (!current_user_can('manage_options') || !static::checkAvailability()) {
            return;
        }

        $languages = static::getCachedLanguages();
        $current_language = function_exists('pll_current_language') ? \pll_current_language('name') : 'Unknown';

        $wp_admin_bar->add_node([
            'id' => 'polylang-cache',
            'title' => "Polylang Cache: {$current_language} (" . count($languages) . " langs)",
            'href' => admin_url('admin.php?page=zizi-cache-settings'),
        ]);
    }

    /**
     * Handle AJAX status request
     * 
     * @return void
     */
    public static function handleAjaxStatus() {
        if (!current_user_can('manage_options') || !wp_verify_nonce($_POST['nonce'], 'polylang_ajax')) {
            wp_die('Unauthorized');
        }

        $status = [
            'available' => static::checkAvailability(),
            'version' => static::getVersion(),
            'languages' => static::getCachedLanguages(),
            'performance' => static::$performance_metrics
        ];

        wp_send_json_success($status);
    }

    /**
     * Handle AJAX cache stats request
     * 
     * @return void
     */
    public static function handleAjaxCacheStats() {
        if (!current_user_can('manage_options') || !wp_verify_nonce($_POST['nonce'], 'polylang_ajax')) {
            wp_die('Unauthorized');
        }

        wp_send_json_success(static::$performance_metrics);
    }

    /**
     * Initialize performance monitoring
     * 
     * @return void
     */
    public static function initPerformanceMonitoring() {
        if (rand(1, 100) <= static::$enterprise_config['monitoring_sample_rate']) {
            add_action('wp_footer', [__CLASS__, 'collectPerformanceData']);
        }
    }

    /**
     * Collect performance data
     * 
     * @return void
     */
    public static function collectPerformanceData() {
        static::$performance_metrics['memory_usage'] = memory_get_peak_usage();
        
        if (static::$performance_metrics['urls_processed'] > 0) {
            CacheSys::writeError('Polylang performance sample: ' . json_encode(static::$performance_metrics));
        }
    }

    /**
     * Check if performance monitoring is enabled
     * 
     * @return bool
     */
    private static function isPerformanceMonitoringEnabled() {
        return static::$enterprise_config['performance_monitoring'] && 
               (defined('WP_DEBUG') && WP_DEBUG);
    }

    /**
     * Update performance metrics
     * 
     * @param array $metrics New metrics data
     * @return void
     */
    private static function updatePerformanceMetrics($metrics) {
        foreach ($metrics as $key => $value) {
            if (isset(static::$performance_metrics[$key])) {
                static::$performance_metrics[$key] += $value;
            }
        }
    }

    /**
     * Log performance metrics
     * 
     * @param int $original_count Original URL count
     * @param int $final_count Final URL count
     * @return void
     */
    private static function logPerformanceMetrics($original_count, $final_count) {
        $performance_data = [
            'original_urls' => $original_count,
            'final_urls' => $final_count,
            'translations_added' => $final_count - $original_count,
            'execution_time' => static::$performance_metrics['execution_time'],
            'memory_usage' => static::$performance_metrics['memory_usage'],
            'timestamp' => current_time('mysql')
        ];

        CacheSys::writeError('Polylang performance metrics: ' . json_encode($performance_data));
    }
}
