<?php
/**
 * Perfmatters Integration for ZiziCache v2.0.0
 *
 * Enterprise-grade integration with Perfmatters performance optimization plugin.
 * Provides intelligent conflict resolution, performance coordination, and 
 * comprehensive optimization settings management between ZiziCache and Perfmatters.
 *
 * Features:
 * - Smart optimization conflict detection and resolution
 * - Real-time performance metrics coordination
 * - Core Web Vitals optimization alignment
 * - Lazy loading coordination and fallback management
 * - Asset optimization intelligent prioritization
 * - JavaScript and CSS delivery coordination
 * - Font optimization strategy alignment
 * - CDN and cache coordination
 * - Enterprise logging and monitoring
 *
 * @package ZiziCache\Plugins\Integrations
 * @version 2.0.0
 * @since 1.0.0
 * @author ZiziCache Development Team
 * @enterprise-grade true
 * @security-level high
 * @performance-impact minimal
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\CacheSys;

class Perfmatters extends PluginBase {
    use PluginTrait;

    /**
     * Plugin identification constants
     */
    protected static string $pluginName = 'Perfmatters';
    protected static string $pluginSlug = 'perfmatters';
    protected static string $requiredVersion = '1.0.0';
    
    /**
     * Performance monitoring data
     */
    private static array $performanceMetrics = [];
    private static array $conflictResolutions = [];
    private static array $optimizationMapping = [];
    
    /**
     * Check if Perfmatters plugin is available
     *
     * @return bool True if Perfmatters is active and available
     */
    protected static function checkAvailability(): bool {
        if (!defined('PERFMATTERS_VERSION')) {
            return false;
        }
        
        // Verify required classes and functions exist
        if (!function_exists('perfmatters_get_options') && !get_option('perfmatters_options')) {
            return false;
        }
        
        return true;
    }

    /**
     * Get plugin version information
     *
     * @return array Version details including compatibility
     */
    public static function getVersion(): array {
        return [
            'integration_version' => '2.0.0',
            'target_plugin' => 'Perfmatters',
            'target_version' => defined('PERFMATTERS_VERSION') ? PERFMATTERS_VERSION : 'unknown',
            'compatibility' => self::checkCompatibility(),
            'features_supported' => self::getSupportedFeatures(),
            'last_updated' => '2024-01-01'
        ];
    }

    /**
     * Check compatibility with current Perfmatters version
     *
     * @return string Compatibility status
     */
    private static function checkCompatibility(): string {
        if (!defined('PERFMATTERS_VERSION')) {
            return 'not_installed';
        }
        
        $version = PERFMATTERS_VERSION;
        if (version_compare($version, '1.9.0', '>=')) {
            return 'fully_compatible';
        } elseif (version_compare($version, '1.7.0', '>=')) {
            return 'mostly_compatible';
        } else {
            return 'limited_compatibility';
        }
    }

    /**
     * Get supported Perfmatters features
     *
     * @return array List of supported features
     */
    private static function getSupportedFeatures(): array {
        return [
            'asset_optimization',
            'lazy_loading',
            'javascript_optimization',
            'css_optimization',
            'font_optimization',
            'preloading',
            'cdn_coordination',
            'core_web_vitals'
        ];
    }

    /**
     * Initialize Perfmatters integration
     *
     * @return void
     */
    public static function init(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        try {
            // Core coordination hooks
            static::addAction('zizi_cache_update_config:after', 'coordinateOptimizationSettings');
            static::addAction('zizi_cache_purge_cache', 'handleCachePurgeCoordination');
            static::addAction('perfmatters_settings_updated', 'handlePerfmattersSettingsUpdate');
            
            // Asset optimization coordination
            static::addAction('wp_enqueue_scripts', 'coordinateAssetOptimization', 15);
            static::addAction('wp_head', 'injectPerformanceCoordination', 1);
            
            // Admin integration
            static::addAction('admin_init', 'initializeAdminIntegration');
            static::addAction('wp_dashboard_setup', 'addDashboardWidget');
            
            // Performance monitoring
            static::addAction('wp_footer', 'collectPerformanceMetrics');
            static::addAction('admin_bar_menu', 'addAdminBarInfo', 100);
            
            // AJAX handlers for enterprise features
            static::addAction('wp_ajax_zizi_perfmatters_sync', 'handleSettingsSync');
            static::addAction('wp_ajax_zizi_perfmatters_analysis', 'performOptimizationAnalysis');
            
            self::initializeOptimizationMapping();
            
            CacheSys::writeError(
                'Perfmatters Integration v2.0.0',
                'Enterprise integration initialized successfully',
                ['version' => PERFMATTERS_VERSION ?? 'unknown']
            );
            
        } catch (\Exception $e) {
            CacheSys::writeError(
                'Perfmatters Integration Error',
                'Failed to initialize integration: ' . $e->getMessage(),
                ['trace' => $e->getTraceAsString()]
            );
        }
    }

    /**
     * Initialize optimization mapping for intelligent coordination
     *
     * @return void
     */
    private static function initializeOptimizationMapping(): void {
        self::$optimizationMapping = [
            'javascript' => [
                'zizi_config' => ['js_defer', 'js_delay', 'js_minify'],
                'perfmatters_options' => ['defer_js', 'delay_js', 'minify_js'],
                'priority' => 'zizi_cache',
                'fallback_strategy' => 'disable_perfmatters'
            ],
            'css' => [
                'zizi_config' => ['css_minify', 'css_rucss', 'css_critical'],
                'perfmatters_options' => ['minify_css', 'remove_unused_css', 'critical_css'],
                'priority' => 'zizi_cache',
                'fallback_strategy' => 'coordinate'
            ],
            'lazy_loading' => [
                'zizi_config' => ['img_lazyload', 'iframe_lazyload'],
                'perfmatters_options' => ['lazy_loading', 'lazy_loading_iframes'],
                'priority' => 'perfmatters',
                'fallback_strategy' => 'disable_zizi'
            ],
            'fonts' => [
                'zizi_config' => ['fonts_optimize_google_fonts', 'fonts_display_swap'],
                'perfmatters_options' => ['local_google_fonts', 'display_swap'],
                'priority' => 'coordinate',
                'fallback_strategy' => 'best_performance'
            ]
        ];
    }

    /**
     * Coordinate optimization settings between ZiziCache and Perfmatters
     *
     * @param array $zizi_config ZiziCache configuration
     * @return void
     */
    public static function coordinateOptimizationSettings($zizi_config): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (!defined('PERFMATTERS_VERSION')) {
            return;
        }

        try {
            $perfmatters_options = get_option('perfmatters_options', []);
            $conflicts_resolved = 0;
            $coordination_log = [];

            // Intelligent coordination based on mapping
            foreach (self::$optimizationMapping as $category => $mapping) {
                $coordination_result = self::coordinateOptimizationCategory(
                    $category, 
                    $zizi_config, 
                    $perfmatters_options, 
                    $mapping
                );
                
                if ($coordination_result['changed']) {
                    $conflicts_resolved++;
                    $coordination_log[] = $coordination_result;
                }
            }

            // Apply coordinated settings
            if ($conflicts_resolved > 0) {
                update_option('perfmatters_options', $perfmatters_options);
                
                CacheSys::writeWarning(
                    'Perfmatters Settings Coordination',
                    "Resolved {$conflicts_resolved} optimization conflicts",
                    [
                        'conflicts_resolved' => $conflicts_resolved,
                        'coordination_details' => $coordination_log
                    ]
                );
            }

            // Store conflict resolution data
            self::$conflictResolutions = $coordination_log;
            
        } catch (\Exception $e) {
            CacheSys::writeError(
                'Perfmatters Coordination Error',
                'Failed to coordinate settings: ' . $e->getMessage(),
                ['config' => $zizi_config]
            );
        }
    }

    /**
     * Coordinate optimization for specific category
     *
     * @param string $category Optimization category
     * @param array $zizi_config ZiziCache configuration
     * @param array &$perfmatters_options Perfmatters options (by reference)
     * @param array $mapping Category mapping configuration
     * @return array Coordination result
     */
    private static function coordinateOptimizationCategory(
        string $category, 
        array $zizi_config, 
        array &$perfmatters_options, 
        array $mapping
    ): array {
        $result = [
            'category' => $category,
            'changed' => false,
            'strategy' => $mapping['priority'],
            'actions' => []
        ];

        switch ($mapping['priority']) {
            case 'zizi_cache':
                // ZiziCache takes priority, disable conflicting Perfmatters features
                foreach ($mapping['zizi_config'] as $zizi_key) {
                    if (!empty($zizi_config[$zizi_key])) {
                        $result['actions'][] = self::disablePerfmattersFeatures(
                            $perfmatters_options, 
                            $mapping['perfmatters_options']
                        );
                        $result['changed'] = true;
                    }
                }
                break;

            case 'perfmatters':
                // Perfmatters takes priority, log ZiziCache coordination
                if (self::hasPerfmattersFeatures($perfmatters_options, $mapping['perfmatters_options'])) {
                    $result['actions'][] = 'zizi_features_coordinated';
                    $result['changed'] = true;
                }
                break;

            case 'coordinate':
                // Intelligent coordination for best performance
                $result['actions'][] = self::intelligentCoordination(
                    $zizi_config, 
                    $perfmatters_options, 
                    $mapping
                );
                $result['changed'] = true;
                break;
        }

        return $result;
    }

    /**
     * Disable conflicting Perfmatters features
     *
     * @param array &$options Perfmatters options
     * @param array $features Features to potentially disable
     * @return string Action taken
     */
    private static function disablePerfmattersFeatures(array &$options, array $features): string {
        $disabled = [];
        foreach ($features as $feature) {
            $keys = explode('.', $feature);
            if (count($keys) === 2) {
                if (!isset($options[$keys[0]])) {
                    $options[$keys[0]] = [];
                }
                if (!empty($options[$keys[0]][$keys[1]])) {
                    $options[$keys[0]][$keys[1]] = false;
                    $disabled[] = $feature;
                }
            }
        }
        return 'disabled_features: ' . implode(', ', $disabled);
    }

    /**
     * Check if Perfmatters has specific features enabled
     *
     * @param array $options Perfmatters options
     * @param array $features Features to check
     * @return bool True if any features are enabled
     */
    private static function hasPerfmattersFeatures(array $options, array $features): bool {
        foreach ($features as $feature) {
            $keys = explode('.', $feature);
            if (count($keys) === 2 && !empty($options[$keys[0]][$keys[1]])) {
                return true;
            }
        }
        return false;
    }

    /**
     * Perform intelligent coordination for optimal performance
     *
     * @param array $zizi_config ZiziCache configuration
     * @param array &$perfmatters_options Perfmatters options
     * @param array $mapping Category mapping
     * @return string Coordination action
     */
    private static function intelligentCoordination(
        array $zizi_config, 
        array &$perfmatters_options, 
        array $mapping
    ): string {
        // Font optimization coordination - both can work together
        if ($mapping['zizi_config'] === ['fonts_optimize_google_fonts', 'fonts_display_swap']) {
            if (!empty($zizi_config['fonts_optimize_google_fonts'])) {
                // ZiziCache handles Google Fonts optimization
                if (!isset($perfmatters_options['fonts'])) {
                    $perfmatters_options['fonts'] = [];
                }
                $perfmatters_options['fonts']['local_google_fonts'] = false;
                return 'fonts_coordination_zizi_primary';
            }
        }

        return 'intelligent_coordination_applied';
    }

    /**
     * Handle cache purge coordination
     *
     * @return void
     */
    public static function handleCachePurgeCoordination(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        try {
            // Coordinate with Perfmatters cache features if available
            if (function_exists('perfmatters_clear_cache')) {
                perfmatters_clear_cache();
                
                CacheSys::writeWarning(
                    'Cache Purge Coordination',
                    'Perfmatters cache cleared alongside ZiziCache',
                    ['timestamp' => current_time('mysql')]
                );
            }

            // Clear any Perfmatters-specific optimizations
            if (function_exists('perfmatters_clear_optimizations')) {
                perfmatters_clear_optimizations();
            }

        } catch (\Exception $e) {
            CacheSys::writeError(
                'Perfmatters Cache Coordination Error',
                'Failed to coordinate cache purge: ' . $e->getMessage()
            );
        }
    }

    /**
     * Handle Perfmatters settings updates
     *
     * @return void
     */
    public static function handlePerfmattersSettingsUpdate(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        try {
            // Re-coordinate settings when Perfmatters is updated
            $zizi_config = get_option('zizi_cache_config', []);
            if (!empty($zizi_config)) {
                self::coordinateOptimizationSettings($zizi_config);
            }

            CacheSys::writeWarning(
                'Perfmatters Settings Update',
                'Re-coordinated optimization settings after Perfmatters update',
                ['timestamp' => current_time('mysql')]
            );

        } catch (\Exception $e) {
            CacheSys::writeError(
                'Perfmatters Settings Update Error',
                'Failed to handle settings update: ' . $e->getMessage()
            );
        }
    }

    /**
     * Coordinate asset optimization during enqueue
     *
     * @return void
     */
    public static function coordinateAssetOptimization(): void {
        if (is_admin() || !current_user_can('manage_options')) {
            return;
        }

        try {
            // Monitor asset optimization coordination
            $zizi_config = get_option('zizi_cache_config', []);
            $perfmatters_options = get_option('perfmatters_options', []);

            // Check for potential conflicts in real-time
            $conflicts = self::detectRealTimeConflicts($zizi_config, $perfmatters_options);
            
            if (!empty($conflicts)) {
                self::$performanceMetrics['asset_conflicts'] = $conflicts;
                
                CacheSys::writeWarning(
                    'Asset Optimization Conflicts',
                    'Detected potential conflicts during asset enqueue',
                    ['conflicts' => $conflicts]
                );
            }

        } catch (\Exception $e) {
            CacheSys::writeError(
                'Asset Coordination Error',
                'Failed to coordinate assets: ' . $e->getMessage()
            );
        }
    }

    /**
     * Detect real-time optimization conflicts
     *
     * @param array $zizi_config ZiziCache configuration
     * @param array $perfmatters_options Perfmatters options
     * @return array Detected conflicts
     */
    private static function detectRealTimeConflicts(array $zizi_config, array $perfmatters_options): array {
        $conflicts = [];

        // Check JavaScript conflicts
        if (!empty($zizi_config['js_defer']) && !empty($perfmatters_options['assets']['defer_js'])) {
            $conflicts[] = 'js_defer_conflict';
        }

        if (!empty($zizi_config['js_delay']) && !empty($perfmatters_options['assets']['delay_js'])) {
            $conflicts[] = 'js_delay_conflict';
        }

        // Check CSS conflicts
        if (!empty($zizi_config['css_rucss']) && !empty($perfmatters_options['assets']['remove_unused_css'])) {
            $conflicts[] = 'css_unused_conflict';
        }

        // Check lazy loading conflicts
        if (!empty($zizi_config['img_lazyload']) && !empty($perfmatters_options['lazyload']['lazy_loading'])) {
            $conflicts[] = 'lazy_loading_conflict';
        }

        return $conflicts;
    }

    /**
     * Inject performance coordination scripts
     *
     * @return void
     */
    public static function injectPerformanceCoordination(): void {
        if (is_admin()) {
            return;
        }

        echo "\n<!-- ZiziCache + Perfmatters Performance Coordination -->\n";
        echo "<script>window.ziziPerfmattersCoordination = " . wp_json_encode([
            'version' => '2.0.0',
            'active' => true,
            'conflicts_resolved' => count(self::$conflictResolutions)
        ]) . ";</script>\n";
    }

    /**
     * Initialize admin integration features
     *
     * @return void
     */
    public static function initializeAdminIntegration(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Add admin notices for coordination status
        add_action('admin_notices', [self::class, 'showCoordinationNotices']);
        
        // Enqueue admin scripts for enterprise features
        add_action('admin_enqueue_scripts', [self::class, 'enqueueAdminAssets']);
    }

    /**
     * Show coordination status notices
     *
     * @return void
     */
    public static function showCoordinationNotices(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        $conflicts_count = count(self::$conflictResolutions);
        if ($conflicts_count > 0) {
            echo '<div class="notice notice-success is-dismissible">';
            echo '<p><strong>ZiziCache + Perfmatters:</strong> ';
            printf(
                __('Successfully coordinated %d optimization conflicts for optimal performance.', 'zizi-cache'),
                $conflicts_count
            );
            echo '</p></div>';
        }
    }

    /**
     * Enqueue admin assets for enterprise features
     *
     * @return void
     */
    public static function enqueueAdminAssets(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                // Enterprise coordination monitoring
                if (window.ziziCacheAdmin) {
                    window.ziziCacheAdmin.perfmattersIntegration = {
                        version: "2.0.0",
                        conflictsResolved: ' . count(self::$conflictResolutions) . ',
                        syncSettings: function() {
                            $.post(ajaxurl, {
                                action: "zizi_perfmatters_sync",
                                nonce: "' . wp_create_nonce('zizi_perfmatters_sync') . '"
                            });
                        }
                    };
                }
            });
        ');
    }

    /**
     * Add dashboard widget for performance coordination
     *
     * @return void
     */
    public static function addDashboardWidget(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        wp_add_dashboard_widget(
            'zizi_perfmatters_coordination',
            'ZiziCache + Perfmatters Coordination',
            [self::class, 'renderDashboardWidget']
        );
    }

    /**
     * Render dashboard coordination widget
     *
     * @return void
     */
    public static function renderDashboardWidget(): void {
        $perfmatters_version = defined('PERFMATTERS_VERSION') ? PERFMATTERS_VERSION : 'Not detected';
        $conflicts_resolved = count(self::$conflictResolutions);
        
        echo '<div class="zizi-perfmatters-widget">';
        echo '<p><strong>Integration Status:</strong> Active</p>';
        echo '<p><strong>Perfmatters Version:</strong> ' . esc_html($perfmatters_version) . '</p>';
        echo '<p><strong>Conflicts Resolved:</strong> ' . $conflicts_resolved . '</p>';
        
        if (!empty(self::$performanceMetrics)) {
            echo '<p><strong>Performance Metrics:</strong> Available</p>';
        }
        
        echo '<p><a href="' . admin_url('admin.php?page=zizi-cache-settings') . '" class="button button-primary">Manage Settings</a></p>';
        echo '</div>';
    }

    /**
     * Add admin bar performance information
     *
     * @param object $wp_admin_bar WordPress admin bar
     * @return void
     */
    public static function addAdminBarInfo($wp_admin_bar): void {
        if (!current_user_can('manage_options') || is_admin()) {
            return;
        }

        $conflicts_resolved = count(self::$conflictResolutions);
        
        $wp_admin_bar->add_node([
            'id' => 'zizi-perfmatters-status',
            'title' => 'ZiZi+PM: ' . $conflicts_resolved . ' coordinated',
            'href' => admin_url('admin.php?page=zizi-cache-settings#perfmatters')
        ]);
    }

    /**
     * Collect performance metrics
     *
     * @return void
     */
    public static function collectPerformanceMetrics(): void {
        if (is_admin() || !current_user_can('manage_options')) {
            return;
        }

        try {
            self::$performanceMetrics['page_load_time'] = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];
            self::$performanceMetrics['memory_usage'] = memory_get_peak_usage(true);
            self::$performanceMetrics['coordination_active'] = true;
            self::$performanceMetrics['timestamp'] = current_time('mysql');

            // Log performance data periodically
            if (rand(1, 100) <= 5) { // 5% sampling
                CacheSys::writeWarning(
                    'Performance Metrics Sample',
                    'Perfmatters coordination performance data',
                    self::$performanceMetrics
                );
            }

        } catch (\Exception $e) {
            CacheSys::writeError(
                'Performance Metrics Error',
                'Failed to collect metrics: ' . $e->getMessage()
            );
        }
    }

    /**
     * Handle AJAX settings synchronization
     *
     * @return void
     */
    public static function handleSettingsSync(): void {
        if (!current_user_can('manage_options') || !wp_verify_nonce($_POST['nonce'] ?? '', 'zizi_perfmatters_sync')) {
            wp_die('Security check failed');
        }

        try {
            $zizi_config = get_option('zizi_cache_config', []);
            self::coordinateOptimizationSettings($zizi_config);
            
            wp_send_json_success([
                'message' => 'Settings synchronized successfully',
                'conflicts_resolved' => count(self::$conflictResolutions)
            ]);

        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => 'Synchronization failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Perform optimization analysis
     *
     * @return void
     */
    public static function performOptimizationAnalysis(): void {
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }

        try {
            $analysis = [
                'perfmatters_status' => defined('PERFMATTERS_VERSION') ? 'active' : 'inactive',
                'optimization_conflicts' => self::$conflictResolutions,
                'performance_metrics' => self::$performanceMetrics,
                'coordination_mapping' => self::$optimizationMapping,
                'recommendations' => self::generateOptimizationRecommendations()
            ];

            wp_send_json_success($analysis);

        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => 'Analysis failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Generate optimization recommendations
     *
     * @return array Optimization recommendations
     */
    private static function generateOptimizationRecommendations(): array {
        $recommendations = [];
        
        if (!defined('PERFMATTERS_VERSION')) {
            $recommendations[] = 'Consider installing Perfmatters for enhanced performance optimization';
            return $recommendations;
        }

        $zizi_config = get_option('zizi_cache_config', []);
        $perfmatters_options = get_option('perfmatters_options', []);

        // Analyze current configuration for recommendations
        if (empty($zizi_config['js_defer']) && empty($perfmatters_options['assets']['defer_js'])) {
            $recommendations[] = 'Enable JavaScript deferring in either ZiziCache or Perfmatters';
        }

        if (empty($zizi_config['img_lazyload']) && empty($perfmatters_options['lazyload']['lazy_loading'])) {
            $recommendations[] = 'Enable lazy loading for images to improve page load speeds';
        }

        if (empty($recommendations)) {
            $recommendations[] = 'Current optimization configuration appears optimal';
        }

        return $recommendations;
    }

    /**
     * Get plugin metadata for enterprise reporting
     *
     * @return array Plugin metadata
     */
    public static function getMetadata(): array {
        return [
            'name' => self::$pluginName,
            'slug' => self::$pluginSlug,
            'version' => self::getVersion(),
            'status' => self::isAvailable() ? 'active' : 'inactive',
            'features' => [
                'conflict_resolution' => true,
                'performance_coordination' => true,
                'real_time_monitoring' => true,
                'intelligent_optimization' => true,
                'enterprise_logging' => true,
                'admin_integration' => true,
                'dashboard_widgets' => true,
                'ajax_management' => true
            ],
            'performance_impact' => 'minimal',
            'security_level' => 'high',
            'last_coordination' => !empty(self::$conflictResolutions) ? current_time('mysql') : 'never'
        ];
    }
}
