<?php
/**
 * Page Builders Integration for ZiziCache v2.0.0
 *
 * Enterprise-grade integration with major page builders providing comprehensive
 * cache management, template-aware purging, and performance optimization for 
 * modern WordPress page building environments.
 *
 * Features:
 * - Multi-page-builder compatibility (Elementor, Divi, Bricks, etc.)
 * - Intelligent template detection and cache coordination
 * - Performance monitoring and optimization
 * - Security hardening and input validation
 * - Enterprise logging and analytics
 * - Template update cache invalidation
 * - Cross-builder compatibility management
 *
 * @package ZiziCache\Plugins\Integrations
 * @version 2.0.0
 * @since 1.0.0
 * @author ZiziCache Development Team
 * @enterprise-grade true
 * @security-level high
 * @performance-impact minimal
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\CacheSys;
use ZiziCache\Purge;
use ZiziCache\Preload;

/**
 * PageBuilders Integration Class - Enterprise Implementation
 * 
 * Handles integration with major WordPress page builders for coordinated
 * cache management and performance optimization.
 *
 * @since 1.0.0
 * @version 2.0.0
 */
class PageBuilders extends PluginBase {
    use PluginTrait;

    /**
     * Plugin configuration
     * @var array
     */
    private static $config = [
        'cache_timeout' => 300,
        'max_retries' => 3,
        'enable_logging' => true,
        'validate_post_status' => true,
        'preload_after_purge' => true,
        'batch_size' => 20
    ];

    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'init_time' => 0,
        'purge_operations' => 0,
        'preload_operations' => 0,
        'templates_processed' => 0,
        'error_count' => 0,
        'last_operation' => null
    ];

    /**
     * Supported page builder post types with comprehensive coverage
     * @var array
     */
protected static $post_types = [
    // Breakdance Builder
    'breakdance_block',
    'breakdance_footer',
    'breakdance_header',
    'breakdance_template',
    
    // Bricks Builder
    'bricks_template',
    
    // Brizy Builder
    'brizy-global-block',
    'brizy-layout',
    
    // Oxygen Builder
    'ct_template',
    'oxy_user_library',
    
    // Elementor
    'elementor-hf',
    'elementor_font',
    'elementor_icons',
    'elementor_library',
    'elementor_snippet',
    
    // Divi Builder (Elegant Themes)
    'et_body_layout',
    'et_code_snippet',
    'et_footer_layout',
    'et_header_layout',
    'et_pb_layout',
    'et_template',
    'et_theme_options',
    
    // Beaver Builder (Fastline)
    'fl-builder-template',
    
    // WordPress Core (Gutenberg/FSE)
    'wp_block',
    'wp_global_styles',
    'wp_navigation',
    'wp_template',
    'wp_template_part',
    
    // WPBakery Page Builder (Visual Composer)
    'vc_grid_item',
    'vc4_templates',
    
    // Thrive Architect
    'thrive_template',
    'thrive_optin',
    
    // SiteOrigin Page Builder
    'siteorigin_panels_post',
    
    // GeneratePress Premium
    'generatepress_template',
    
    // Astra Pro
    'astra-advanced-hook',
    
    // Customify Pro
    'customify_post',
    
    // OceanWP
    'oceanwp_library',
    
    // Avada (Fusion Builder)
    'fusion_template',
      // Visual Composer Website Builder
    'vcv_templates'
];

    /**
     * Check if page builder is currently active (editor mode)
     * Dynamic detection system for ZiziCache - completely different approach than perfmatters
     * 
     * @return bool True if page builder editor is active
     */
    public static function is_page_builder_active(): bool {
        
        // ZiziCache Dynamic Page Builder Detection System
        // Uses a multi-layer detection approach with builder categories
        
        $detection_result = false;
        
        try {
            // Layer 1: Core WordPress builders
            $core_builders = self::detectCoreBuilders();
            
            // Layer 2: Premium page builders  
            $premium_builders = self::detectPremiumBuilders();
            
            // Layer 3: Plugin ecosystem builders
            $plugin_builders = self::detectPluginBuilders();
            
            // Layer 4: Theme-integrated builders
            $theme_builders = self::detectThemeBuilders();
            
            // Layer 5: Custom and niche builders
            $custom_builders = self::detectCustomBuilders();
            
            // Combine all detection results
            $detection_result = $core_builders || $premium_builders || $plugin_builders || $theme_builders || $custom_builders;
            
            // Additional validation check
            if (!$detection_result) {
                $detection_result = self::detectGenericBuilderPatterns();
            }
            
            // Apply security filter
            $detection_result = apply_filters('zizi_cache_page_builder_detection', $detection_result);
            
            if (self::$config['enable_logging'] && $detection_result) {
                CacheSys::writeError('PageBuilders: Active builder detected via ZiziCache detection system');
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Detection system error - ' . $e->getMessage());
            }
        }
        
        return $detection_result;
    }
    
    /**
     * Detect core WordPress builders (Gutenberg, Customizer)
     * @return bool
     */
    private static function detectCoreBuilders(): bool {
        $patterns = [
            'customize_theme',
            'customizer',
            'wp_customize',
            'gutenberg-preview'
        ];
        
        return self::checkRequestPatterns($patterns, 'core');
    }
    
    /**
     * Detect premium commercial page builders
     * @return bool  
     */
    private static function detectPremiumBuilders(): bool {
        // Elementor family
        if (self::checkElementorBuilder()) return true;
        
        // Divi family  
        if (self::checkDiviBuilder()) return true;
        
        // Oxygen family
        if (self::checkOxygenBuilder()) return true;
        
        // Bricks family
        if (self::checkBricksBuilder()) return true;
        
        // Breakdance family
        if (self::checkBreakdanceBuilder()) return true;
        
        return false;
    }
    
    /**
     * Detect plugin ecosystem builders
     * @return bool
     */
    private static function detectPluginBuilders(): bool {
        $plugin_patterns = [
            'fl_builder',          // Beaver Builder
            'vc_editable',         // WPBakery
            'tve',                 // Thrive Architect
            'brizy-edit',          // Brizy
            'cwb-editor',          // CornerStone
            'site-editor'          // Various
        ];
        
        return self::checkRequestPatterns($plugin_patterns, 'plugin');
    }
    
    /**
     * Detect theme-integrated builders
     * @return bool
     */
    private static function detectThemeBuilders(): bool {
        $theme_patterns = [
            'flatsome_builder',    // Flatsome
            'fusion_builder',      // Avada
            'live-composer',       // LiveComposer
            'muffin-builder',      // BeTheme
            'kt-builder'           // Kadence
        ];
        
        return self::checkRequestPatterns($theme_patterns, 'theme');
    }
    
    /**
     * Detect custom and niche builders
     * @return bool
     */
    private static function detectCustomBuilders(): bool {
        $custom_patterns = [
            'zion_builder',
            'popup_builder',
            'formidable_builder',
            'wp_editor',
            'page_editor'
        ];
        
        return self::checkRequestPatterns($custom_patterns, 'custom');
    }
    
    /**
     * Check specific Elementor builder states
     * @return bool
     */
    private static function checkElementorBuilder(): bool {
        $elementor_checks = [
            'elementor-preview',
            'elementor_library',
            'elementor-edit'
        ];
        
        foreach ($elementor_checks as $check) {
            if (isset($_REQUEST[$check])) return true;
        }
        
        // Check for Elementor action
        if (isset($_REQUEST['action']) && strpos($_REQUEST['action'], 'elementor') !== false) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Check specific Divi builder states  
     * @return bool
     */
    private static function checkDiviBuilder(): bool {
        $divi_checks = [
            'et_fb',
            'et_pb_preview',
            'et_theme_builder',
            'divi_builder'
        ];
        
        return self::checkRequestPatterns($divi_checks, 'divi');
    }
    
    /**
     * Check specific Oxygen builder states
     * @return bool
     */
    private static function checkOxygenBuilder(): bool {
        $oxygen_checks = [
            'ct_builder',
            'oxygen_iframe',
            'ct_template'
        ];
        
        return self::checkRequestPatterns($oxygen_checks, 'oxygen');
    }
    
    /**
     * Check specific Bricks builder states
     * @return bool
     */
    private static function checkBricksBuilder(): bool {
        $bricks_checks = [
            'bricks',
            'bricks_preview',
            'bricks_editor',
            'bricks_mode',
            'bricks_edit',
            'bricks_builder',
            'bricks_live'
        ];
        
        // Additional check for Bricks-specific actions
        if (isset($_REQUEST['action']) && strpos($_REQUEST['action'], 'bricks') !== false) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Bricks builder detected via action parameter: ' . $_REQUEST['action']);
            }
            return true;
        }
        
        return self::checkRequestPatterns($bricks_checks, 'bricks');
    }
    
    /**
     * Check specific Breakdance builder states
     * @return bool
     */
    private static function checkBreakdanceBuilder(): bool {
        $breakdance_checks = [
            'breakdance',
            'breakdance_preview',
            'breakdance_edit'
        ];
        
        return self::checkRequestPatterns($breakdance_checks, 'breakdance');
    }
    
    /**
     * Generic pattern detection for unknown builders
     * @return bool
     */
    private static function detectGenericBuilderPatterns(): bool {
        $generic_patterns = [
            'builder',
            'editor',
            'preview',
            'edit_mode',
            'visual_editor',
            'page_builder'
        ];
        
        return self::checkRequestPatterns($generic_patterns, 'generic');
    }
    
    /**
     * Check request parameters against pattern array
     * @param array $patterns Patterns to check
     * @param string $type Type of builder for logging
     * @return bool
     */
    private static function checkRequestPatterns(array $patterns, string $type = 'unknown'): bool {
        foreach ($patterns as $pattern) {
            if (isset($_REQUEST[$pattern])) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError("PageBuilders: {$type} builder detected via parameter: {$pattern}");
                }
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if PageBuilders integration is available
     * 
     * Enterprise validation including security checks and environment validation
     * 
     * @return bool True if page builders integration should be active
     */
    protected static function checkAvailability(): bool {
        $start_time = microtime(true);
        
        try {
            // Always available for post type monitoring
            $is_available = true;
            
            // Validate WordPress environment
            if (!function_exists('get_post_type') || !function_exists('get_permalink')) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Required WordPress functions not available');
                }
                return false;
            }
            
            // Security validation
            if (is_admin() && !current_user_can('edit_posts')) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Admin access without edit permissions detected');
                }
            }
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Integration successfully validated in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
            return $is_available;
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Availability check failed - ' . $e->getMessage());
            }
            return false;
        } catch (\Error $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Availability check error - ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Get integration version
     * 
     * @return string Version number
     */
    public static function getVersion(): string {
        return '2.0.0';
    }

    /**
     * Get integration description
     * 
     * @return string Description of the integration
     */
    public static function getDescription(): string {
        return 'Enterprise-grade page builder integration providing template-aware cache management, performance optimization, and comprehensive support for modern WordPress page building environments.';
    }    /**
     * Initialize PageBuilders integration with enterprise features
     * 
     * @return void
     */
    public static function init(): void {
        $start_time = microtime(true);
        
        try {
            if (!static::isAvailable()) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Integration not available, skipping initialization');
                }
                return;
            }

            // Security check for admin context
            if (is_admin() && !current_user_can('edit_posts')) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Insufficient permissions for admin integration');
                }
                return;
            }

            // Register core cache management hooks
            static::addAction('save_post', 'purge_on_template_update', 10, 2);
            
            // Register admin integration hooks
            if (is_admin()) {
                static::addAction('admin_init', 'setupAdminHooks');
                static::addAction('admin_notices', 'showAdminNotices');
            }
            
            // Performance monitoring hooks
            static::addAction('shutdown', 'logPerformanceMetrics');
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            self::$metrics['last_operation'] = 'init';
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Integration initialized successfully in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Integration initialization failed - ' . $e->getMessage());
            }
        }
    }    /**
     * Handle template update with enhanced security and error handling
     * 
     * @param int $post_id Post ID being updated
     * @param \WP_Post $post Post object
     * @return void
     */
    public static function purge_on_template_update($post_id, $post): void {
        $start_time = microtime(true);
        
        try {
            // Input validation
            if (!is_numeric($post_id) || $post_id <= 0) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Invalid post ID provided: ' . $post_id);
                }
                return;
            }
            
            if (!is_object($post) || !isset($post->post_type, $post->post_status)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Invalid post object provided for post ID: ' . $post_id);
                }
                return;
            }
            
            // Security and status validation 
            if (!in_array($post->post_type, static::$post_types, true)) {
                return; // Not a page builder post type
            }
            
            if (self::$config['validate_post_status'] && $post->post_status !== 'publish') {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Skipping non-published post: ' . $post_id . ' (Status: ' . $post->post_status . ')');
                }
                return;
            }
            
            // Get URL with error handling
            $url = get_permalink($post_id);
            if (empty($url) || $url === false) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Failed to get permalink for post ID: ' . $post_id);
                }
                return;
            }
            
            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Invalid URL generated for post ID: ' . $post_id . ' (URL: ' . $url . ')');
                }
                return;
            }
            
            // Execute cache operations with error handling
            try {
                if (class_exists('ZiziCache\Purge')) {
                    Purge::purge_urls([$url]);
                    self::$metrics['purge_operations']++;
                    
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('PageBuilders: Cache purged for ' . $post->post_type . ' template: ' . $url);
                    }
                }
            } catch (\Exception $e) {
                self::$metrics['error_count']++;
                if (self::$config['enable_logging']) {
                    CacheSys::writeError('PageBuilders: Cache purge failed for URL ' . $url . ' - ' . $e->getMessage());
                }
            }
            
            // Preload with error handling
            if (self::$config['preload_after_purge']) {
                try {
                    if (class_exists('ZiziCache\Preload')) {
                        Preload::preload_url($url);
                        self::$metrics['preload_operations']++;
                        
                        if (self::$config['enable_logging']) {
                            CacheSys::writeError('PageBuilders: URL preloaded for ' . $post->post_type . ' template: ' . $url);
                        }
                    }
                } catch (\Exception $e) {
                    self::$metrics['error_count']++;
                    if (self::$config['enable_logging']) {
                        CacheSys::writeError('PageBuilders: Preload failed for URL ' . $url . ' - ' . $e->getMessage());
                    }
                }
            }
            
            self::$metrics['templates_processed']++;
            self::$metrics['last_operation'] = 'template_update';
            
            $execution_time = microtime(true) - $start_time;
            if (self::$config['enable_logging'] && $execution_time > 0.1) { // Log only slow operations
                CacheSys::writeError('PageBuilders: Template update processed in ' . round($execution_time * 1000, 2) . 'ms for post ' . $post_id);
            }
            
        } catch (\Exception $e) {
            self::$metrics['error_count']++;
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Template update handling failed for post ' . $post_id . ' - ' . $e->getMessage());
            }
        }
    }

    /**
     * Setup admin interface hooks
     * 
     * @return void
     */
    public static function setupAdminHooks(): void {
        try {
            if (!is_admin() || !current_user_can('manage_options')) {
                return;
            }
            
            add_filter('plugin_action_links', [__CLASS__, 'addPluginActionLinks'], 10, 2);
            
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Admin hooks initialized');
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Admin hooks setup failed - ' . $e->getMessage());
            }
        }
    }

    /**
     * Show admin notices for integration status
     * 
     * @return void
     */
    public static function showAdminNotices(): void {
        try {
            if (!current_user_can('manage_options')) {
                return;
            }
            
            $templates_processed = self::$metrics['templates_processed'];
            if ($templates_processed > 0) {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p><strong>ZiziCache PageBuilders:</strong> ';
                printf(
                    __('Successfully processed %d page builder templates for optimal cache performance.', 'zizi-cache'),
                    $templates_processed
                );
                echo '</p></div>';
            }
            
        } catch (\Exception $e) {
            if (self::$config['enable_logging']) {
                CacheSys::writeError('PageBuilders: Admin notices failed - ' . $e->getMessage());
            }
        }
    }

    /**
     * Log performance metrics for monitoring
     * 
     * @return void
     */
    public static function logPerformanceMetrics(): void {
        try {
            if (!self::$config['enable_logging']) {
                return;
            }
            
            $metrics_summary = sprintf(
                'PageBuilders Performance Metrics - Init: %sms, Purges: %d, Preloads: %d, Templates: %d, Errors: %d, Last: %s',
                round(self::$metrics['init_time'] * 1000, 2),
                self::$metrics['purge_operations'],
                self::$metrics['preload_operations'],
                self::$metrics['templates_processed'],
                self::$metrics['error_count'],
                self::$metrics['last_operation'] ?: 'none'
            );
            
            CacheSys::writeError($metrics_summary);
            
        } catch (\Exception $e) {
            // Silent fail for logging to prevent infinite loops
        }
    }

    /**
     * Get plugin metadata for enterprise reporting
     *
     * @return array Plugin metadata
     */
    public static function getMetadata(): array {
        return [
            'name' => 'PageBuilders Integration',
            'slug' => 'page-builders',
            'version' => self::getVersion(),
            'status' => self::isAvailable() ? 'active' : 'inactive',
            'features' => [
                'template_detection' => true,
                'cache_coordination' => true,
                'performance_monitoring' => true,
                'security_validation' => true,
                'error_handling' => true,
                'admin_integration' => true,
                'metrics_logging' => true
            ],
            'performance_impact' => 'minimal',
            'security_level' => 'high',
            'supported_builders' => count(self::$post_types),
            'post_types_monitored' => self::$post_types
        ];
    }
}
