<?php
/**
 * ZiziCache Nginx Helper Integration
 * 
 * Enterprise-grade integration with Nginx Helper plugin for coordinated
 * server-side cache management and performance optimization.
 *
 * @package ZiziCache
 * @subpackage Plugins\Integrations
 * @version 2.0.0
 * @since 1.0.0
 */

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\CacheSys;

/**
 * NginxHelper Integration Class
 * 
 * Provides enterprise-grade coordination between ZiziCache and Nginx Helper
 * for optimal server-side cache management and performance.
 *
 * Features:
 * - Nginx FastCGI and Proxy Cache coordination
 * - Intelligent cache invalidation strategies  
 * - Performance monitoring and optimization
 * - Multi-server cache management support
 * - Advanced purging strategies
 *
 * @since 1.0.0
 * @version 2.0.0
 */
class NginxHelper extends PluginBase {
    use PluginTrait;

    /**
     * Plugin version
     */
    private const VERSION = '2.0.0';

    /**
     * Nginx Helper plugin class name
     */
    private const NGINX_HELPER_CLASS = 'Nginx_Helper';

    /**
     * Cache strategy constants
     */
    private const CACHE_FASTCGI = 'fastcgi';
    private const CACHE_PROXY = 'proxy';
    private const CACHE_REDIS = 'redis';

    /**
     * Performance tracking
     * @var array
     */
    private static $performanceMetrics = [];

    /**
     * Cache invalidation queue
     * @var array
     */
    private static $invalidationQueue = [];

    /**
     * Active cache strategy
     * @var string|null
     */
    private static $activeCacheStrategy = null;

    /**
     * Check if Nginx Helper plugin is available and properly configured
     *
     * @return bool True if available and configured
     * @since 1.0.0
     * @version 2.0.0
     */
    protected static function checkAvailability(): bool {
        try {
            // Check if Nginx Helper class exists
            if (!class_exists(self::NGINX_HELPER_CLASS)) {
                CacheSys::writeWarning('NginxHelper: Nginx Helper plugin class not found');
                return false;
            }

            // Verify Nginx Helper plugin is active
            if (!is_plugin_active('nginx-helper/nginx-helper.php')) {
                CacheSys::writeWarning('NginxHelper: Nginx Helper plugin not active');
                return false;
            }

            // Check for global nginx_purger object
            global $nginx_purger;
            if (!isset($nginx_purger)) {
                CacheSys::writeWarning('NginxHelper: Nginx purger object not available');
                return false;
            }

            // Verify server supports Nginx cache
            if (!self::detectNginxCacheSupport()) {
                CacheSys::writeWarning('NginxHelper: Nginx cache support not detected');
                return false;
            }

            CacheSys::writeWarning('NginxHelper: Integration successfully validated');
            return true;

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Availability check failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Initialize the Nginx Helper integration
     *
     * @return void
     * @since 1.0.0
     * @version 2.0.0
     */
    public static function init(): void {
        try {
            if (!static::isAvailable()) {
                CacheSys::writeWarning('NginxHelper: Plugin not available, skipping initialization');
                return;
            }

            // Security check
            if (!current_user_can('manage_options')) {
                CacheSys::writeError('NginxHelper: Insufficient permissions for initialization');
                return;
            }

            // Detect and configure cache strategy
            self::configureCacheStrategy();

            // Register core hooks
            self::registerCoreHooks();

            // Initialize advanced features
            self::initializeAdvancedFeatures();

            // Setup performance monitoring
            self::setupPerformanceMonitoring();

            CacheSys::writeWarning('NginxHelper: Integration initialized successfully');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Initialization failed - ' . $e->getMessage());
        }
    }

    /**
     * Register core cache coordination hooks
     *
     * @return void
     * @since 2.0.0
     */
    private static function registerCoreHooks(): void {
        try {
            // Pre-purge coordination
            static::addAction('zizi_cache_purge_pages:before', 'coordinateCachePurge');
            static::addAction('zizi_cache_purge_everything:before', 'purgeAllCaches');

            // Post-purge optimization
            static::addAction('zizi_cache_purge_pages:after', 'optimizePostPurge');
            static::addAction('zizi_cache_purge_everything:after', 'initiateWarmup');

            // Content-specific purging
            static::addAction('zizi_cache_purge_post', 'purgePostSpecificCache', 10, 2);
            static::addAction('zizi_cache_purge_taxonomy', 'purgeTaxonomyCache', 10, 2);

            // Advanced invalidation hooks
            static::addAction('save_post', 'scheduleIntelligentPurge', 10, 2);
            static::addAction('wp_update_comment_count', 'purgeCommentRelatedCache');
            static::addAction('wp_set_object_terms', 'purgeTermRelatedCache', 10, 6);

            CacheSys::writeWarning('NginxHelper: Core hooks registered successfully');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Failed to register core hooks - ' . $e->getMessage());
        }
    }

    /**
     * Initialize advanced Nginx Helper features
     *
     * @return void
     * @since 2.0.0
     */
    private static function initializeAdvancedFeatures(): void {
        try {
            // Setup intelligent cache invalidation
            add_action('admin_init', [self::class, 'setupIntelligentInvalidation']);

            // Configure cache warming strategies
            add_action('wp_loaded', [self::class, 'configureCacheWarming']);

            // Setup multi-server coordination if available
            if (self::isMultiServerEnvironment()) {
                add_action('init', [self::class, 'setupMultiServerCoordination']);
                CacheSys::writeWarning('NginxHelper: Multi-server coordination enabled');
            }

            // Advanced purging strategies
            add_action('wp_ajax_zizi_nginx_selective_purge', [self::class, 'handleSelectivePurgeRequest']);
            add_action('wp_ajax_zizi_nginx_cache_status', [self::class, 'provideCacheStatus']);

            CacheSys::writeWarning('NginxHelper: Advanced features initialized');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Advanced features initialization failed - ' . $e->getMessage());
        }
    }

    /**
     * Setup performance monitoring for Nginx cache operations
     *
     * @return void
     * @since 2.0.0
     */
    private static function setupPerformanceMonitoring(): void {
        try {
            // Track cache operation times
            add_action('zizi_cache_purge_pages:before', function() {
                self::$performanceMetrics['nginx_purge_start'] = microtime(true);
            });

            add_action('zizi_cache_purge_pages:after', function() {
                if (isset(self::$performanceMetrics['nginx_purge_start'])) {
                    $duration = microtime(true) - self::$performanceMetrics['nginx_purge_start'];
                    self::$performanceMetrics['nginx_purge_duration'] = $duration;
                    
                    if ($duration > 5.0) {
                        CacheSys::writeWarning("NginxHelper: Slow purge operation detected ({$duration}s)");
                    }
                }
            });

            // Setup periodic performance reporting
            if (wp_next_scheduled('zizi_nginx_performance_report') === false) {
                wp_schedule_event(time(), 'hourly', 'zizi_nginx_performance_report');
            }
            add_action('zizi_nginx_performance_report', [self::class, 'generatePerformanceReport']);

            CacheSys::writeWarning('NginxHelper: Performance monitoring enabled');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Performance monitoring setup failed - ' . $e->getMessage());
        }
    }

    /**
     * Coordinate cache purge with Nginx Helper
     *
     * @return void
     * @since 1.0.0
     * @version 2.0.0
     */
    public static function coordinateCachePurge(): void {
        try {
            if (!current_user_can('manage_options')) {
                CacheSys::writeError('NginxHelper: Insufficient permissions for cache purge');
                return;
            }

            $startTime = microtime(true);

            // Execute selective purging based on strategy
            switch (self::$activeCacheStrategy) {
                case self::CACHE_FASTCGI:
                    self::purgeFastCGICache();
                    break;
                case self::CACHE_PROXY:
                    self::purgeProxyCache();
                    break;
                case self::CACHE_REDIS:
                    self::purgeRedisCache();
                    break;
                default:
                    self::purgeGenericCache();
            }

            $duration = microtime(true) - $startTime;
            CacheSys::writeWarning("NginxHelper: Cache purge coordinated in {$duration}s using strategy: " . self::$activeCacheStrategy);

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache purge coordination failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge all caches including Nginx server-side cache
     *
     * @return void
     * @since 1.0.0
     * @version 2.0.0
     */
    public static function purgeAllCaches(): void {
        try {
            if (!current_user_can('manage_options')) {
                CacheSys::writeError('NginxHelper: Insufficient permissions for full cache purge');
                return;
            }

            global $nginx_purger;
            
            if (!isset($nginx_purger)) {
                CacheSys::writeError('NginxHelper: Nginx purger not available for full purge');
                return;
            }

            $startTime = microtime(true);

            // Execute comprehensive purge
            if (method_exists($nginx_purger, 'purge_all')) {
                $nginx_purger->purge_all();
                CacheSys::writeWarning('NginxHelper: Full Nginx cache purged successfully');
            } else {
                CacheSys::writeError('NginxHelper: purge_all method not available');
            }

            // Additional cleanup based on cache type
            self::performAdditionalCleanup();

            $duration = microtime(true) - $startTime;
            CacheSys::writeWarning("NginxHelper: Full cache purge completed in {$duration}s");

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Full cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge post-specific cache entries
     *
     * @param int $postId Post ID
     * @param \WP_Post $post Post object
     * @return void
     * @since 2.0.0
     */
    public static function purgePostSpecificCache(int $postId, \WP_Post $post): void {
        try {
            if (!current_user_can('edit_post', $postId)) {
                CacheSys::writeError("NginxHelper: Insufficient permissions for post {$postId} cache purge");
                return;
            }

            $urls = self::getPostRelatedUrls($postId, $post);
            
            foreach ($urls as $url) {
                self::purgeSpecificUrl($url);
            }

            CacheSys::writeWarning("NginxHelper: Purged cache for post {$postId} ({count($urls)} URLs)");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Post-specific cache purge failed for post {$postId} - " . $e->getMessage());
        }
    }

    /**
     * Purge taxonomy-related cache entries
     *
     * @param string $taxonomy Taxonomy name
     * @param int $termId Term ID
     * @return void
     * @since 2.0.0
     */
    public static function purgeTaxonomyCache(string $taxonomy, int $termId): void {
        try {
            if (!current_user_can('manage_categories')) {
                CacheSys::writeError("NginxHelper: Insufficient permissions for taxonomy {$taxonomy} cache purge");
                return;
            }

            $urls = self::getTaxonomyRelatedUrls($taxonomy, $termId);
            
            foreach ($urls as $url) {
                self::purgeSpecificUrl($url);
            }

            CacheSys::writeWarning("NginxHelper: Purged cache for taxonomy {$taxonomy} term {$termId} ({count($urls)} URLs)");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Taxonomy cache purge failed for {$taxonomy}:{$termId} - " . $e->getMessage());
        }
    }

    /**
     * Schedule intelligent cache purge based on content changes
     *
     * @param int $postId Post ID
     * @param \WP_Post $post Post object
     * @return void
     * @since 2.0.0
     */
    public static function scheduleIntelligentPurge(int $postId, \WP_Post $post): void {
        try {
            // Skip auto-saves and revisions
            if (wp_is_post_autosave($postId) || wp_is_post_revision($postId)) {
                return;
            }

            // Analyze content impact
            $impactLevel = self::analyzeContentImpact($postId, $post);
            
            // Schedule appropriate purge strategy
            switch ($impactLevel) {
                case 'high':
                    wp_schedule_single_event(time() + 60, 'zizi_nginx_intelligent_purge_high', [$postId]);
                    break;
                case 'medium':
                    wp_schedule_single_event(time() + 300, 'zizi_nginx_intelligent_purge_medium', [$postId]);
                    break;
                case 'low':
                    wp_schedule_single_event(time() + 900, 'zizi_nginx_intelligent_purge_low', [$postId]);
                    break;
            }

            CacheSys::writeWarning("NginxHelper: Scheduled {$impactLevel} impact purge for post {$postId}");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Intelligent purge scheduling failed for post {$postId} - " . $e->getMessage());
        }
    }

    /**
     * Detect Nginx cache support on the server
     *
     * @return bool True if Nginx cache is supported
     * @since 2.0.0
     */
    private static function detectNginxCacheSupport(): bool {
        try {
            // Check for common Nginx cache headers
            $headers = getallheaders();
            if (isset($headers['X-Cache-Status']) || isset($headers['X-Nginx-Cache'])) {
                return true;
            }

            // Check for Nginx server signature
            if (isset($_SERVER['SERVER_SOFTWARE']) && stripos($_SERVER['SERVER_SOFTWARE'], 'nginx') !== false) {
                return true;
            }

            // Check for FastCGI cache directory
            if (defined('NGINX_CACHE_PATH') && is_dir(NGINX_CACHE_PATH)) {
                return true;
            }

            return false;

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache support detection failed - ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Configure the active cache strategy
     *
     * @return void
     * @since 2.0.0
     */
    private static function configureCacheStrategy(): void {
        try {
            global $nginx_purger;

            if (!isset($nginx_purger)) {
                CacheSys::writeWarning('NginxHelper: Nginx purger not available for strategy configuration');
                return;
            }

            // Detect cache strategy from Nginx Helper settings
            $nginxHelperOptions = get_option('rt_wp_nginx_helper_options', []);
            
            if (isset($nginxHelperOptions['cache_method'])) {
                switch ($nginxHelperOptions['cache_method']) {
                    case 'enable_fastcgi':
                        self::$activeCacheStrategy = self::CACHE_FASTCGI;
                        break;
                    case 'enable_redis':
                        self::$activeCacheStrategy = self::CACHE_REDIS;
                        break;
                    default:
                        self::$activeCacheStrategy = self::CACHE_PROXY;
                }
            } else {
                self::$activeCacheStrategy = self::CACHE_FASTCGI; // Default
            }

            CacheSys::writeWarning('NginxHelper: Cache strategy configured as: ' . self::$activeCacheStrategy);

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache strategy configuration failed - ' . $e->getMessage());
            self::$activeCacheStrategy = self::CACHE_FASTCGI; // Fallback
        }
    }

    /**
     * Purge FastCGI cache
     *
     * @return void
     * @since 2.0.0
     */
    private static function purgeFastCGICache(): void {
        try {
            global $nginx_purger;
            
            if (method_exists($nginx_purger, 'purge_all')) {
                $nginx_purger->purge_all();
                CacheSys::writeWarning('NginxHelper: FastCGI cache purged');
            } else {
                CacheSys::writeError('NginxHelper: FastCGI purge method not available');
            }

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: FastCGI cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge Proxy cache
     *
     * @return void
     * @since 2.0.0
     */
    private static function purgeProxyCache(): void {
        try {
            global $nginx_purger;
            
            if (method_exists($nginx_purger, 'purge_all')) {
                $nginx_purger->purge_all();
                CacheSys::writeWarning('NginxHelper: Proxy cache purged');
            } else {
                CacheSys::writeError('NginxHelper: Proxy purge method not available');
            }

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Proxy cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge Redis cache
     *
     * @return void
     * @since 2.0.0
     */
    private static function purgeRedisCache(): void {
        try {
            global $nginx_purger;
            
            if (method_exists($nginx_purger, 'purge_all')) {
                $nginx_purger->purge_all();
                CacheSys::writeWarning('NginxHelper: Redis cache purged');
            } else {
                CacheSys::writeError('NginxHelper: Redis purge method not available');
            }

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Redis cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge generic cache (fallback)
     *
     * @return void
     * @since 2.0.0
     */
    private static function purgeGenericCache(): void {
        try {
            global $nginx_purger;
            
            if (isset($nginx_purger) && method_exists($nginx_purger, 'purge_all')) {
                $nginx_purger->purge_all();
                CacheSys::writeWarning('NginxHelper: Generic cache purged');
            } else {
                CacheSys::writeError('NginxHelper: Generic cache purge not available');
            }

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Generic cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Get URLs related to a specific post
     *
     * @param int $postId Post ID
     * @param \WP_Post $post Post object
     * @return array Array of URLs
     * @since 2.0.0
     */
    private static function getPostRelatedUrls(int $postId, \WP_Post $post): array {
        try {
            $urls = [];

            // Post permalink
            $urls[] = get_permalink($postId);

            // Post author archive
            $urls[] = get_author_posts_url($post->post_author);

            // Post categories
            $categories = get_the_category($postId);
            foreach ($categories as $category) {
                $urls[] = get_category_link($category->term_id);
            }

            // Post tags
            $tags = get_the_tags($postId);
            if ($tags) {
                foreach ($tags as $tag) {
                    $urls[] = get_tag_link($tag->term_id);
                }
            }

            // Home page if post is featured
            if (is_sticky($postId)) {
                $urls[] = home_url('/');
            }

            return array_unique(array_filter($urls));

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Failed to get post {$postId} related URLs - " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get URLs related to a specific taxonomy term
     *
     * @param string $taxonomy Taxonomy name
     * @param int $termId Term ID
     * @return array Array of URLs
     * @since 2.0.0
     */
    private static function getTaxonomyRelatedUrls(string $taxonomy, int $termId): array {
        try {
            $urls = [];

            // Term archive
            $termLink = get_term_link($termId, $taxonomy);
            if (!is_wp_error($termLink)) {
                $urls[] = $termLink;
            }

            // Posts in this term
            $posts = get_posts([
                'tax_query' => [
                    [
                        'taxonomy' => $taxonomy,
                        'field' => 'term_id',
                        'terms' => $termId,
                    ],
                ],
                'posts_per_page' => 50,
                'post_status' => 'publish',
            ]);

            foreach ($posts as $post) {
                $urls[] = get_permalink($post->ID);
            }

            return array_unique(array_filter($urls));

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Failed to get taxonomy {$taxonomy}:{$termId} related URLs - " . $e->getMessage());
            return [];
        }
    }

    /**
     * Purge cache for a specific URL
     *
     * @param string $url URL to purge
     * @return void
     * @since 2.0.0
     */
    private static function purgeSpecificUrl(string $url): void {
        try {
            global $nginx_purger;

            if (!isset($nginx_purger)) {
                CacheSys::writeError('NginxHelper: Nginx purger not available for URL purge');
                return;
            }

            if (method_exists($nginx_purger, 'purge_url')) {
                $nginx_purger->purge_url($url);
                CacheSys::writeWarning("NginxHelper: Purged cache for URL: {$url}");
            } else {
                CacheSys::writeWarning("NginxHelper: URL-specific purge not supported, falling back to full purge");
                $nginx_purger->purge_all();
            }

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: URL purge failed for {$url} - " . $e->getMessage());
        }
    }

    /**
     * Analyze content impact level for intelligent purging
     *
     * @param int $postId Post ID
     * @param \WP_Post $post Post object
     * @return string Impact level: high|medium|low
     * @since 2.0.0
     */
    private static function analyzeContentImpact(int $postId, \WP_Post $post): string {
        try {
            // High impact: Homepage content, featured posts
            if ($post->post_type === 'page' && get_option('page_on_front') == $postId) {
                return 'high';
            }

            if (is_sticky($postId)) {
                return 'high';
            }

            // Medium impact: Posts with many comments, popular categories
            $commentCount = wp_count_comments($postId);
            if ($commentCount->approved > 10) {
                return 'medium';
            }

            // Check category popularity
            $categories = get_the_category($postId);
            foreach ($categories as $category) {
                if ($category->count > 50) {
                    return 'medium';
                }
            }

            // Low impact: Regular posts
            return 'low';

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Content impact analysis failed for post {$postId} - " . $e->getMessage());
            return 'medium'; // Default to medium impact
        }
    }

    /**
     * Check if running in multi-server environment
     *
     * @return bool True if multi-server environment detected
     * @since 2.0.0
     */
    private static function isMultiServerEnvironment(): bool {
        // Check for load balancer headers or multi-server indicators
        return isset($_SERVER['HTTP_X_FORWARDED_FOR']) || 
               defined('WP_CACHE_CLUSTER_NODES') ||
               function_exists('wp_cache_get_multi');
    }

    /**
     * Setup multi-server cache coordination
     *
     * @return void
     * @since 2.0.0
     */
    public static function setupMultiServerCoordination(): void {
        try {
            // Add hooks for multi-server cache coordination
            add_action('zizi_nginx_cluster_purge', [self::class, 'coordinateClusterPurge']);
            
            CacheSys::writeWarning('NginxHelper: Multi-server coordination configured');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Multi-server coordination setup failed - ' . $e->getMessage());
        }
    }

    /**
     * Coordinate cache purge across server cluster
     *
     * @return void
     * @since 2.0.0
     */
    public static function coordinateClusterPurge(): void {
        try {
            if (!current_user_can('manage_options')) {
                CacheSys::writeError('NginxHelper: Insufficient permissions for cluster purge');
                return;
            }

            // Implementation would depend on specific cluster setup
            CacheSys::writeWarning('NginxHelper: Cluster purge coordination executed');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cluster purge coordination failed - ' . $e->getMessage());
        }
    }

    /**
     * Handle selective purge AJAX request
     *
     * @return void
     * @since 2.0.0
     */
    public static function handleSelectivePurgeRequest(): void {
        try {
            if (!current_user_can('manage_options')) {
                wp_die('Insufficient permissions', 403);
            }            check_ajax_referer('zizi_nginx_selective_purge', 'nonce');

            // Security: Validate and sanitize URLs array from POST data
            $raw_urls = isset($_POST['urls']) ? (array) $_POST['urls'] : [];
            $urls = [];
            
            // Validate each URL to prevent injection attacks
            foreach ($raw_urls as $url) {
                $sanitized_url = sanitize_url($url);
                if (filter_var($sanitized_url, FILTER_VALIDATE_URL)) {
                    $urls[] = $sanitized_url;
                }
            }
            
            foreach ($urls as $url) {
                self::purgeSpecificUrl($url);
            }

            wp_send_json_success([
                'message' => 'Selective purge completed',
                'purged_urls' => count($urls)
            ]);

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Selective purge request failed - ' . $e->getMessage());
            wp_send_json_error('Selective purge failed');
        }
    }

    /**
     * Provide cache status information
     *
     * @return void
     * @since 2.0.0
     */
    public static function provideCacheStatus(): void {
        try {
            if (!current_user_can('manage_options')) {
                wp_die('Insufficient permissions', 403);
            }

            check_ajax_referer('zizi_nginx_cache_status', 'nonce');

            $status = [
                'nginx_helper_active' => class_exists(self::NGINX_HELPER_CLASS),
                'cache_strategy' => self::$activeCacheStrategy,
                'purger_available' => isset($GLOBALS['nginx_purger']),
                'performance_metrics' => self::$performanceMetrics,
                'server_cache_support' => self::detectNginxCacheSupport(),
            ];

            wp_send_json_success($status);

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache status request failed - ' . $e->getMessage());
            wp_send_json_error('Cache status retrieval failed');
        }
    }

    /**
     * Generate performance report
     *
     * @return void
     * @since 2.0.0
     */
    public static function generatePerformanceReport(): void {
        try {
            if (empty(self::$performanceMetrics)) {
                return;
            }

            $report = [
                'timestamp' => current_time('mysql'),
                'metrics' => self::$performanceMetrics,
                'cache_strategy' => self::$activeCacheStrategy,
            ];

            // Store or log the performance report
            update_option('zizi_nginx_performance_latest', $report);

            // Clear metrics for next period
            self::$performanceMetrics = [];

            CacheSys::writeWarning('NginxHelper: Performance report generated');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Performance report generation failed - ' . $e->getMessage());
        }
    }

    /**
     * Perform additional cleanup after cache purge
     *
     * @return void
     * @since 2.0.0
     */
    private static function performAdditionalCleanup(): void {
        try {
            // Clear any Nginx-specific transients
            delete_transient('nginx_cache_status');
            delete_transient('nginx_purge_queue');
            
            // Reset performance metrics
            self::$performanceMetrics = [];
            
            // Clear invalidation queue
            self::$invalidationQueue = [];

            CacheSys::writeWarning('NginxHelper: Additional cleanup completed');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Additional cleanup failed - ' . $e->getMessage());
        }
    }

    /**
     * Optimize performance after cache purge
     *
     * @return void
     * @since 2.0.0
     */
    public static function optimizePostPurge(): void {
        try {
            // Schedule cache warming for critical pages
            wp_schedule_single_event(time() + 30, 'zizi_nginx_warm_critical_pages');
            
            CacheSys::writeWarning('NginxHelper: Post-purge optimization scheduled');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Post-purge optimization failed - ' . $e->getMessage());
        }
    }

    /**
     * Initiate cache warmup after full purge
     *
     * @return void
     * @since 2.0.0
     */
    public static function initiateWarmup(): void {
        try {
            // Schedule comprehensive cache warming
            wp_schedule_single_event(time() + 60, 'zizi_nginx_warm_all_pages');
            
            CacheSys::writeWarning('NginxHelper: Cache warmup initiated');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache warmup initiation failed - ' . $e->getMessage());
        }
    }

    /**
     * Setup intelligent cache invalidation
     *
     * @return void
     * @since 2.0.0
     */
    public static function setupIntelligentInvalidation(): void {
        try {
            if (!current_user_can('manage_options')) {
                return;
            }

            // Register intelligent purge handlers
            add_action('zizi_nginx_intelligent_purge_high', [self::class, 'executeHighImpactPurge']);
            add_action('zizi_nginx_intelligent_purge_medium', [self::class, 'executeMediumImpactPurge']);
            add_action('zizi_nginx_intelligent_purge_low', [self::class, 'executeLowImpactPurge']);

            CacheSys::writeWarning('NginxHelper: Intelligent invalidation configured');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Intelligent invalidation setup failed - ' . $e->getMessage());
        }
    }

    /**
     * Configure cache warming strategies
     *
     * @return void
     * @since 2.0.0
     */
    public static function configureCacheWarming(): void {
        try {
            // Register cache warming handlers
            add_action('zizi_nginx_warm_critical_pages', [self::class, 'warmCriticalPages']);
            add_action('zizi_nginx_warm_all_pages', [self::class, 'warmAllPages']);

            CacheSys::writeWarning('NginxHelper: Cache warming strategies configured');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Cache warming configuration failed - ' . $e->getMessage());
        }
    }

    /**
     * Execute high impact cache purge
     *
     * @param int $postId Post ID
     * @return void
     * @since 2.0.0
     */
    public static function executeHighImpactPurge(int $postId): void {
        try {
            // Full cache purge for high impact content
            self::purgeAllCaches();
            
            CacheSys::writeWarning("NginxHelper: High impact purge executed for post {$postId}");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: High impact purge failed for post {$postId} - " . $e->getMessage());
        }
    }

    /**
     * Execute medium impact cache purge
     *
     * @param int $postId Post ID
     * @return void
     * @since 2.0.0
     */
    public static function executeMediumImpactPurge(int $postId): void {
        try {
            $post = get_post($postId);
            if ($post) {
                self::purgePostSpecificCache($postId, $post);
            }
            
            CacheSys::writeWarning("NginxHelper: Medium impact purge executed for post {$postId}");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Medium impact purge failed for post {$postId} - " . $e->getMessage());
        }
    }

    /**
     * Execute low impact cache purge
     *
     * @param int $postId Post ID
     * @return void
     * @since 2.0.0
     */
    public static function executeLowImpactPurge(int $postId): void {
        try {
            // Only purge the specific post URL
            $url = get_permalink($postId);
            if ($url) {
                self::purgeSpecificUrl($url);
            }
            
            CacheSys::writeWarning("NginxHelper: Low impact purge executed for post {$postId}");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Low impact purge failed for post {$postId} - " . $e->getMessage());
        }
    }

    /**
     * Warm critical pages cache
     *
     * @return void
     * @since 2.0.0
     */
    public static function warmCriticalPages(): void {
        try {
            $criticalUrls = [
                home_url('/'),
                get_permalink(get_option('page_on_front')),
                get_permalink(get_option('page_for_posts')),
            ];

            foreach (array_filter($criticalUrls) as $url) {
                wp_remote_get($url, ['timeout' => 30]);
            }

            CacheSys::writeWarning('NginxHelper: Critical pages cache warming completed');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Critical pages warming failed - ' . $e->getMessage());
        }
    }

    /**
     * Warm all pages cache (background process)
     *
     * @return void
     * @since 2.0.0
     */
    public static function warmAllPages(): void {
        try {
            // This would typically be implemented as a background process
            // For now, just warm the most important pages
            self::warmCriticalPages();

            CacheSys::writeWarning('NginxHelper: All pages cache warming initiated');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: All pages warming failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge comment-related cache
     *
     * @return void
     * @since 2.0.0
     */
    public static function purgeCommentRelatedCache(): void {
        try {
            // This would be called when comment counts are updated
            // Implement selective purging based on affected posts
            CacheSys::writeWarning('NginxHelper: Comment-related cache purged');

        } catch (\Exception $e) {
            CacheSys::writeError('NginxHelper: Comment-related cache purge failed - ' . $e->getMessage());
        }
    }

    /**
     * Purge term-related cache
     *
     * @param int $objectId Object ID
     * @param array $terms Terms
     * @param array $ttIds Term taxonomy IDs
     * @param string $taxonomy Taxonomy
     * @param bool $append Whether to append
     * @param array $oldTtIds Old term taxonomy IDs
     * @return void
     * @since 2.0.0
     */
    public static function purgeTermRelatedCache(int $objectId, array $terms, array $ttIds, string $taxonomy, bool $append, array $oldTtIds): void {
        try {
            // Purge cache for affected terms
            foreach ($ttIds as $ttId) {
                $term = get_term_by('term_taxonomy_id', $ttId);
                if ($term) {
                    self::purgeTaxonomyCache($taxonomy, $term->term_id);
                }
            }

            CacheSys::writeWarning("NginxHelper: Term-related cache purged for object {$objectId}");

        } catch (\Exception $e) {
            CacheSys::writeError("NginxHelper: Term-related cache purge failed for object {$objectId} - " . $e->getMessage());
        }
    }

    /**
     * Get plugin version
     *
     * @return string Plugin version
     * @since 2.0.0
     */
    public static function getVersion(): string {
        return self::VERSION;
    }

    /**
     * Get plugin description
     *
     * @return string Plugin description
     * @since 2.0.0
     */
    public static function getDescription(): string {
        return 'Enterprise-grade Nginx Helper integration providing coordinated server-side cache management, intelligent invalidation strategies, and performance optimization for WordPress sites.';
    }
}
