<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\CacheSys;

/**
 * Gutenberg Block Editor Integration - Enterprise Grade
 * 
 * Comprehensive integration with WordPress Gutenberg Block Editor providing:
 * - Advanced lazy rendering for blocks
 * - Performance optimization for block editor
 * - Custom ZiziCache block controls
 * - Block-level cache management
 * - Editor asset optimization
 * - Intersection observer integration
 * 
 * @version 2.0.0
 * @author ZiziCache Team
 */
class Gutenberg extends PluginBase
{
    /**
     * ZiziCache block attributes
     */
    private const ZIZI_BLOCK_ATTRIBUTES = [
        'ziziLazyRender' => [
            'type' => 'boolean',
            'default' => false,
        ],
        'ziziCacheSettings' => [
            'type' => 'object',
            'default' => [
                'enabled' => true,
                'duration' => 3600,
                'priority' => 'normal'
            ],
        ],
        'ziziPerformance' => [
            'type' => 'object',
            'default' => [
                'intersectionObserver' => true,
                'deferLoading' => false,
                'preloadCritical' => true
            ],
        ],
    ];

    /**
     * Block types that support lazy rendering
     */
    private const LAZY_RENDER_BLOCKS = [
        'core/image',
        'core/gallery',
        'core/video',
        'core/embed',
        'core/media-text',
        'core/cover',
        'core/latest-posts',
        'core/query',
        'core/post-content'
    ];

    /**
     * Critical blocks that should not be lazy loaded
     */
    private const CRITICAL_BLOCKS = [
        'core/heading',
        'core/paragraph',
        'core/navigation',
        'core/site-title',
        'core/site-logo'
    ];

    /**
     * Integration status flags
     */
    private static $assets_enqueued = false;
    private static $editor_enhanced = false;

    /**
     * Check if Gutenberg Block Editor is available
     * 
     * @return bool True if Gutenberg is available
     */
    protected static function checkAvailability(): bool
    {
        try {
            // Gutenberg is part of WordPress core since 5.0
            $is_available = (
                function_exists('register_block_type') &&
                function_exists('wp_enqueue_block_editor_assets') &&
                function_exists('get_block_wrapper_attributes')
            );

            if ($is_available) {
                global $wp_version;
                CacheSys::writeError("Gutenberg Block Editor integration activated - WordPress version: {$wp_version}", 'Gutenberg');
                
                // Check for block editor context
                if (is_admin() && function_exists('get_current_screen')) {
                    $screen = get_current_screen();
                    if ($screen && $screen->is_block_editor()) {
                        CacheSys::writeError("Block editor context detected", 'Gutenberg');
                    }
                }
            }

            return $is_available;
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to check Gutenberg availability: " . $e->getMessage(), 'Gutenberg');
            return false;
        }
    }

    /**
     * Get integration version
     */
    public function getVersion(): string
    {
        return '2.0.0';
    }

    /**
     * Get integration description
     */
    public function getDescription(): string
    {
        return 'Enterprise-grade Gutenberg Block Editor integration providing advanced lazy rendering, performance optimization, and block-level cache management';
    }

    /**
     * Initialize Gutenberg integration with comprehensive enhancements
     */
    public function init(): void
    {
        if (!self::isAvailable()) {
            return;
        }

        if (!current_user_can('edit_posts') && is_admin()) {
            CacheSys::writeWarning("Gutenberg integration accessed by user without edit permissions", 'Gutenberg');
        }

        try {
            // Core editor enhancements
            add_action('enqueue_block_editor_assets', [$this, 'enqueue_editor_assets'], 5);
            add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_assets'], 5);
            
            // Block registration and attribute management
            add_filter('register_block_type_args', [$this, 'enhance_block_attributes'], 10, 2);
            add_filter('block_type_metadata', [$this, 'optimize_block_metadata'], 10, 2);
            
            // Block rendering optimization
            add_filter('render_block', [$this, 'optimize_block_rendering'], 10, 2);
            add_filter('render_block_data', [$this, 'preprocess_block_data'], 10, 2);
            
            // Editor experience enhancements
            add_action('admin_init', [$this, 'setup_editor_enhancements'], 5);
            add_filter('block_editor_settings', [$this, 'enhance_editor_settings'], 10, 2);
            
            // Performance optimization
            add_action('wp_footer', [$this, 'add_intersection_observer'], 99);
            add_action('wp_head', [$this, 'add_critical_css'], 1);
            
            CacheSys::writeError("Gutenberg Block Editor integration successfully initialized", 'Gutenberg');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to initialize Gutenberg integration: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Enqueue enhanced block editor assets
     */
    public function enqueue_editor_assets(): void
    {
        if (!current_user_can('edit_posts')) {
            return;
        }

        try {
            if (self::$assets_enqueued) {
                return;
            }

            $asset_url = defined('ZIZI_CACHE_PLUGIN_URL') ? ZIZI_CACHE_PLUGIN_URL : '';
            $version = defined('ZIZI_CACHE_VERSION') ? ZIZI_CACHE_VERSION : '2.0.0';

            // Enhanced block editor script
            wp_enqueue_script(
                'zizi-cache-block-editor',
                $asset_url . 'assets/js/block-editor.js',
                [
                    'wp-blocks', 
                    'wp-i18n', 
                    'wp-element', 
                    'wp-editor', 
                    'wp-hooks',
                    'wp-components',
                    'wp-block-editor',
                    'wp-data'
                ],
                $version,
                true
            );

            // Editor styles
            wp_enqueue_style(
                'zizi-cache-block-editor',
                $asset_url . 'assets/css/block-editor.css',
                ['wp-edit-blocks'],
                $version
            );

            // Localization data
            wp_localize_script('zizi-cache-block-editor', 'ziziCacheBlockEditor', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('zizi_cache_block_editor'),
                'settings' => $this->get_editor_settings(),
                'lazy_blocks' => self::LAZY_RENDER_BLOCKS,
                'critical_blocks' => self::CRITICAL_BLOCKS
            ]);

            self::$assets_enqueued = true;
            CacheSys::writeError("Block editor assets enqueued successfully", 'Gutenberg');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to enqueue editor assets: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Enqueue frontend assets for block optimization
     */
    public function enqueue_frontend_assets(): void
    {
        if (is_admin()) {
            return;
        }

        try {
            $asset_url = defined('ZIZI_CACHE_PLUGIN_URL') ? ZIZI_CACHE_PLUGIN_URL : '';
            $version = defined('ZIZI_CACHE_VERSION') ? ZIZI_CACHE_VERSION : '2.0.0';

            // Frontend block optimization script - temporarily disabled
            // wp_enqueue_script(
            //     'zizi-cache-blocks-frontend',
            //     $asset_url . 'assets/js/blocks-frontend.js',
            //     ['jquery'],
            //     $version,
            //     true
            // );

            // Frontend optimization data
            // wp_localize_script('zizi-cache-blocks-frontend', 'ziziCacheBlocks', [
            //     'lazy_loading' => true,
            //     'intersection_observer' => true,
            //     'performance_mode' => get_option('zizi_cache_performance_mode', 'balanced')
            // ]);

            CacheSys::writeError("Frontend block assets enqueued", 'Gutenberg');
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to enqueue frontend assets: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Enhance block type registration with ZiziCache attributes
     */
    public function enhance_block_attributes($args, $block_type): array
    {
        try {
            if (empty($args['attributes'])) {
                $args['attributes'] = [];
            }

            // Add ZiziCache attributes to all blocks
            foreach (self::ZIZI_BLOCK_ATTRIBUTES as $attr_name => $attr_config) {
                $args['attributes'][$attr_name] = $attr_config;
            }

            // Specific enhancements for lazy-renderable blocks
            if (in_array($block_type, self::LAZY_RENDER_BLOCKS)) {
                $args['attributes']['ziziLazyRender']['default'] = true;
                CacheSys::writeError("Enhanced block type '{$block_type}' with lazy rendering support", 'Gutenberg');
            }

            // Prevent lazy loading for critical blocks
            if (in_array($block_type, self::CRITICAL_BLOCKS)) {
                $args['attributes']['ziziLazyRender']['default'] = false;
                $args['attributes']['ziziPerformance']['default']['preloadCritical'] = true;
                CacheSys::writeError("Enhanced critical block type '{$block_type}' with performance optimization", 'Gutenberg');
            }

            return $args;
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to enhance block attributes for '{$block_type}': " . $e->getMessage(), 'Gutenberg');
            return $args;
        }
    }

    /**
     * Optimize block metadata during registration
     */
    public function optimize_block_metadata($metadata, $file): array
    {
        try {
            if (!empty($metadata['name'])) {
                // Add performance hints
                $metadata['ziziCache'] = [
                    'version' => '2.0.0',
                    'optimized' => true,
                    'lazy_eligible' => in_array($metadata['name'], self::LAZY_RENDER_BLOCKS)
                ];

                CacheSys::writeError("Optimized metadata for block: " . $metadata['name'], 'Gutenberg');
            }

            return $metadata;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to optimize block metadata: " . $e->getMessage(), 'Gutenberg');
            return $metadata;
        }
    }

    /**
     * Optimize block rendering with lazy loading and caching
     */
    public function optimize_block_rendering($block_content, $block): string
    {
        try {
            if (empty($block['blockName'])) {
                return $block_content;
            }

            $block_name = $block['blockName'];
            $attributes = $block['attrs'] ?? [];

            // Apply lazy rendering if enabled
            if ($this->should_lazy_render($block_name, $attributes)) {
                $block_content = $this->apply_lazy_rendering($block_content, $block_name, $attributes);
                CacheSys::writeError("Applied lazy rendering to block: {$block_name}", 'Gutenberg');
            }

            // Add intersection observer for performance
            if ($this->should_use_intersection_observer($block_name, $attributes)) {
                $block_content = $this->add_intersection_observer_attributes($block_content, $block_name);
            }

            // Add performance monitoring
            $block_content = $this->add_performance_monitoring($block_content, $block_name);

            return $block_content;
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to optimize block rendering: " . $e->getMessage(), 'Gutenberg');
            return $block_content;
        }
    }

    /**
     * Preprocess block data before rendering
     */
    public function preprocess_block_data($parsed_block, $source_block): array
    {
        try {
            if (!empty($parsed_block['blockName'])) {
                // Add ZiziCache processing flags
                $parsed_block['ziziProcessed'] = true;
                $parsed_block['ziziTimestamp'] = time();
                
                // Optimize attributes
                if (!empty($parsed_block['attrs'])) {
                    $parsed_block['attrs'] = $this->optimize_block_attributes($parsed_block['attrs'], $parsed_block['blockName']);
                }

                CacheSys::writeError("Preprocessed block data: " . $parsed_block['blockName'], 'Gutenberg');
            }

            return $parsed_block;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to preprocess block data: " . $e->getMessage(), 'Gutenberg');
            return $parsed_block;
        }
    }

    /**
     * Setup enhanced editor functionality
     */
    public function setup_editor_enhancements(): void
    {
        if (!current_user_can('edit_posts')) {
            return;
        }

        try {
            if (self::$editor_enhanced) {
                return;
            }

            // Add editor-specific hooks
            add_action('admin_footer', [$this, 'add_editor_enhancements_script']);
            add_filter('block_editor_rest_api_preload', [$this, 'optimize_editor_preload']);
            
            // Add ZiziCache panel to block editor
            add_action('enqueue_block_editor_assets', [$this, 'add_zizi_cache_panel'], 20);

            self::$editor_enhanced = true;
            CacheSys::writeError("Editor enhancements setup completed", 'Gutenberg');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to setup editor enhancements: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Enhance block editor settings
     */
    public function enhance_editor_settings($settings, $context): array
    {
        try {
            if (!is_array($settings)) {
                $settings = [];
            }

            // Add ZiziCache-specific settings
            $settings['ziziCache'] = [
                'enabled' => true,
                'version' => '2.0.0',
                'lazy_loading' => true,
                'performance_mode' => get_option('zizi_cache_performance_mode', 'balanced'),
                'supported_blocks' => self::LAZY_RENDER_BLOCKS,
                'critical_blocks' => self::CRITICAL_BLOCKS
            ];

            // Optimize existing settings
            if (isset($settings['__experimentalFeatures'])) {
                $settings['__experimentalFeatures']['ziziCache'] = true;
            }

            CacheSys::writeError("Enhanced editor settings with ZiziCache configuration", 'Gutenberg');
            return $settings;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to enhance editor settings: " . $e->getMessage(), 'Gutenberg');
            return $settings;
        }
    }

    /**
     * Add intersection observer for lazy loading
     */
    public function add_intersection_observer(): void
    {
        if (is_admin()) {
            return;
        }

        try {
            echo '<script>
                if ("IntersectionObserver" in window) {
                    document.addEventListener("DOMContentLoaded", function() {
                        const observer = new IntersectionObserver((entries) => {
                            entries.forEach(entry => {
                                if (entry.isIntersecting) {
                                    const element = entry.target;
                                    if (element.dataset.ziziLazy) {
                                        element.classList.add("zizi-lazy-loaded");
                                        element.dispatchEvent(new CustomEvent("ziziLazyLoad"));
                                    }
                                    observer.unobserve(element);
                                }
                            });
                        }, { rootMargin: "50px" });
                        
                        document.querySelectorAll("[data-zizi-lazy]").forEach(el => {
                            observer.observe(el);
                        });
                    });
                }
            </script>';
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add intersection observer: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Add critical CSS for block rendering
     */
    public function add_critical_css(): void
    {
        if (is_admin()) {
            return;
        }

        try {
            echo '<style>
                .zizi-lazy-block {
                    min-height: 100px;
                    background: #f0f0f0;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: opacity 0.3s ease;
                }
                .zizi-lazy-block::before {
                    content: "Loading...";
                    color: #666;
                }
                .zizi-lazy-loaded {
                    opacity: 1;
                }
                .zizi-performance-optimized {
                    contain: layout;
                }
            </style>';
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add critical CSS: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Add ZiziCache panel to block editor
     */
    public function add_zizi_cache_panel(): void
    {
        try {
            // Editor panel script - temporarily disabled until file is created
            // wp_enqueue_script(
            //     'zizi-cache-editor-panel',
            //     defined('ZIZI_CACHE_PLUGIN_URL') ? ZIZI_CACHE_PLUGIN_URL . 'assets/js/editor-panel.js' : '',
            //     ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components'],
            //     defined('ZIZI_CACHE_VERSION') ? ZIZI_CACHE_VERSION : '2.0.0',
            //     true
            // );

            CacheSys::writeError("ZiziCache editor panel skipped (file not found)", 'Gutenberg');
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add editor panel: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Add editor enhancements script
     */
    public function add_editor_enhancements_script(): void
    {
        $screen = get_current_screen();
        if (!$screen || !$screen->is_block_editor()) {
            return;
        }

        try {
            echo '<script>
                document.addEventListener("DOMContentLoaded", function() {
                    // Enhanced block editor functionality
                    if (window.wp && window.wp.hooks) {
                        window.wp.hooks.addFilter(
                            "blocks.registerBlockType",
                            "zizi-cache/enhance-blocks",
                            function(settings, name) {
                                if (window.ziziCacheBlockEditor && window.ziziCacheBlockEditor.lazy_blocks.includes(name)) {
                                    settings.supports = settings.supports || {};
                                    settings.supports.ziziCache = true;
                                }
                                return settings;
                            }
                        );
                    }
                });
            </script>';
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add editor enhancements script: " . $e->getMessage(), 'Gutenberg');
        }
    }

    /**
     * Optimize editor preload data
     */
    public function optimize_editor_preload($preload_data): array
    {
        try {
            // Add ZiziCache-specific preload data
            $preload_data['zizi_cache_settings'] = [
                'lazy_loading' => true,
                'performance_mode' => get_option('zizi_cache_performance_mode', 'balanced'),
                'version' => '2.0.0'
            ];

            CacheSys::writeError("Optimized editor preload data", 'Gutenberg');
            return $preload_data;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to optimize editor preload: " . $e->getMessage(), 'Gutenberg');
            return $preload_data;
        }
    }

    /**
     * Determine if block should use lazy rendering
     */
    private function should_lazy_render(string $block_name, array $attributes): bool
    {
        // Never lazy render critical blocks
        if (in_array($block_name, self::CRITICAL_BLOCKS)) {
            return false;
        }

        // Check if block supports lazy rendering
        if (!in_array($block_name, self::LAZY_RENDER_BLOCKS)) {
            return false;
        }

        // Check block-specific setting
        return $attributes['ziziLazyRender'] ?? true;
    }

    /**
     * Apply lazy rendering to block content
     */
    private function apply_lazy_rendering(string $content, string $block_name, array $attributes): string
    {
        if (empty($content)) {
            return $content;
        }

        // Add lazy loading attributes
        $lazy_attributes = 'data-zizi-lazy="true" data-block-name="' . esc_attr($block_name) . '"';
        
        // Find the first opening tag and add attributes
        if (preg_match('/<([a-zA-Z0-9-]+)([^>]*)>/', $content, $matches, PREG_OFFSET_CAPTURE)) {
            $tag = $matches[1][0];
            $existing_attrs = $matches[2][0];
            $position = $matches[0][1] + strlen($matches[1][0]) + 1;
            
            $enhanced_content = substr_replace(
                $content,
                $existing_attrs . ' ' . $lazy_attributes,
                $position,
                strlen($existing_attrs)
            );
            
            return $enhanced_content;
        }

        return $content;
    }

    /**
     * Check if block should use intersection observer
     */
    private function should_use_intersection_observer(string $block_name, array $attributes): bool
    {
        $performance_settings = $attributes['ziziPerformance'] ?? [];
        return $performance_settings['intersectionObserver'] ?? true;
    }

    /**
     * Add intersection observer attributes to block
     */
    private function add_intersection_observer_attributes(string $content, string $block_name): string
    {
        if (empty($content)) {
            return $content;
        }

        $observer_attrs = 'data-zizi-observer="true"';
        
        if (preg_match('/<([a-zA-Z0-9-]+)([^>]*)>/', $content, $matches, PREG_OFFSET_CAPTURE)) {
            $existing_attrs = $matches[2][0];
            $position = $matches[0][1] + strlen($matches[1][0]) + 1;
            
            return substr_replace(
                $content,
                $existing_attrs . ' ' . $observer_attrs,
                $position,
                strlen($existing_attrs)
            );
        }

        return $content;
    }

    /**
     * Add performance monitoring to block
     */
    private function add_performance_monitoring(string $content, string $block_name): string
    {
        if (is_admin() || empty($content)) {
            return $content;
        }

        // Add performance monitoring comment
        return $content . '<!-- ZiziCache: ' . esc_html($block_name) . ' optimized -->';
    }

    /**
     * Optimize block attributes
     */
    private function optimize_block_attributes(array $attributes, string $block_name): array
    {
        // Set default ZiziCache attributes if not present
        foreach (self::ZIZI_BLOCK_ATTRIBUTES as $attr_name => $attr_config) {
            if (!isset($attributes[$attr_name])) {
                $attributes[$attr_name] = $attr_config['default'];
            }
        }

        return $attributes;
    }

    /**
     * Get editor settings for localization
     */
    private function get_editor_settings(): array
    {
        return [
            'lazy_loading' => true,
            'performance_mode' => get_option('zizi_cache_performance_mode', 'balanced'),
            'intersection_observer' => true,
            'critical_css' => true,
            'version' => '2.0.0'
        ];
    }
}
