<?php

namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\CacheSys;

/**
 * EWWW Image Optimizer Integration - Enterprise Grade
 * 
 * Comprehensive integration with EWWW Image Optimizer providing:
 * - Coordinated lazy loading management
 * - WebP delivery optimization
 * - Cache invalidation coordination
 * - Image optimization conflict resolution
 * - Performance monitoring and reporting
 * 
 * @version 2.0.0
 * @author ZiziCache Team
 */
class EWWW extends PluginBase
{
    /**
     * EWWW settings that conflict with ZiziCache
     */
    private const CONFLICTING_SETTINGS = [
        'ewww_image_optimizer_lazy_load',
        'ewww_image_optimizer_picture_webp', 
        'ewww_image_optimizer_js_webp',
        'ewww_image_optimizer_webp_for_cdn',
        'ewww_image_optimizer_auto',
        'ewww_image_optimizer_delay_js'
    ];

    /**
     * EWWW WebP settings to coordinate
     */
    private const WEBP_SETTINGS = [
        'ewww_image_optimizer_webp',
        'ewww_image_optimizer_webp_force',
        'ewww_image_optimizer_picture_webp'
    ];

    /**
     * Integration status flags
     */
    private static $coordination_applied = false;
    private static $webp_optimization_enabled = false;

    /**
     * Check if EWWW Image Optimizer is available and properly configured
     * 
     * @return bool True if EWWW is available and compatible
     */
    protected static function checkAvailability(): bool
    {
        try {
            $is_available = (
                defined('EWWW_IMAGE_OPTIMIZER_VERSION') &&
                class_exists('EWWW_Image_Optimizer') &&
                function_exists('ewww_image_optimizer_get_option')
            );

            if ($is_available) {
                $version = defined('EWWW_IMAGE_OPTIMIZER_VERSION') ? EWWW_IMAGE_OPTIMIZER_VERSION : 'unknown';
                CacheSys::writeError("EWWW Image Optimizer integration activated - version: {$version}", 'EWWW');
                
                // Check for potential conflicts
                self::checkAndLogConflicts();
            }

            return $is_available;
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to check EWWW availability: " . $e->getMessage(), 'EWWW');
            return false;
        }
    }

    /**
     * Get integration version
     */
    public function getVersion(): string
    {
        return '2.0.0';
    }

    /**
     * Get integration description
     */
    public function getDescription(): string
    {
        return 'Enterprise-grade EWWW Image Optimizer integration providing coordinated lazy loading, WebP optimization, and cache management';
    }

    /**
     * Initialize EWWW integration with comprehensive coordination
     */
    public function init(): void
    {
        if (!self::isAvailable()) {
            return;
        }

        if (!current_user_can('manage_options') && is_admin()) {
            CacheSys::writeWarning("EWWW integration accessed by non-admin user", 'EWWW');
        }

        try {
            // Core integration hooks
            add_action('zizi_cache_update_config:after', [$this, 'coordinate_settings']);
            add_action('admin_init', [$this, 'setup_admin_coordination'], 5);
            
            // Image optimization coordination
            add_action('ewww_image_optimizer_post_optimization', [$this, 'handle_image_optimization'], 10, 2);
            add_filter('ewww_image_optimizer_bypass_optimization', [$this, 'coordinate_optimization_bypass'], 10, 2);
            
            // WebP delivery coordination
            add_filter('ewww_image_optimizer_webp_available', [$this, 'coordinate_webp_delivery']);
            add_action('ewww_image_optimizer_webp_cleanup', [$this, 'handle_webp_cleanup']);
            
            // Cache invalidation coordination
            add_action('ewww_image_optimizer_optimize_attachment', [$this, 'invalidate_cache_on_optimization'], 10, 1);
            add_action('ewww_image_optimizer_bulk_loop_post_optimize', [$this, 'handle_bulk_optimization_cache']);
            
            // Performance monitoring
            add_action('wp_footer', [$this, 'add_performance_monitoring'], 99);
            
            CacheSys::writeError("EWWW Image Optimizer integration successfully initialized", 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to initialize EWWW integration: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Coordinate settings between EWWW and ZiziCache
     */
    public function coordinate_settings(): void
    {
        if (!current_user_can('manage_options')) {
            CacheSys::writeWarning("Settings coordination attempted by non-admin user", 'EWWW');
            return;
        }

        try {
            if (self::$coordination_applied) {
                return;
            }

            $conflicts_resolved = 0;
            $webp_coordinated = false;

            // Disable conflicting EWWW settings
            foreach (self::CONFLICTING_SETTINGS as $setting) {
                $current_value = get_option($setting);
                
                if ($current_value && $current_value !== false) {
                    update_option($setting, false);
                    update_site_option($setting, false);
                    $conflicts_resolved++;
                    
                    CacheSys::writeError("Disabled conflicting EWWW setting: {$setting}", 'EWWW');
                }
            }

            // Coordinate WebP settings for optimal performance
            $webp_coordinated = $this->coordinate_webp_settings();

            // Enable beneficial EWWW settings that work well with ZiziCache
            $this->enable_compatible_features();

            self::$coordination_applied = true;
            
            CacheSys::writeError("Settings coordination completed - conflicts resolved: {$conflicts_resolved}, WebP coordinated: " . ($webp_coordinated ? 'yes' : 'no'), 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to coordinate settings: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Setup admin interface coordination
     */
    public function setup_admin_coordination(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        try {
            // Add admin notices for coordination status
            add_action('admin_notices', [$this, 'show_coordination_notices']);
            
            // Add settings validation
            add_filter('pre_update_option_ewww_image_optimizer_lazy_load', [$this, 'prevent_lazy_load_conflicts']);
            
            // Add ZiziCache section to EWWW settings
            add_action('ewww_image_optimizer_settings_page', [$this, 'add_zizi_cache_settings_section']);
            
            CacheSys::writeError("Admin coordination setup completed", 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to setup admin coordination: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Handle image optimization events
     */
    public function handle_image_optimization($optimized_data, $original_file): void
    {
        try {
            if (empty($optimized_data) || empty($original_file)) {
                return;
            }

            // Invalidate related caches
            $this->invalidate_image_caches($original_file);
            
            // Update optimization statistics
            $this->update_optimization_stats($optimized_data);
            
            CacheSys::writeError("Processed image optimization for: " . basename($original_file), 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to handle image optimization: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Coordinate optimization bypass decisions
     */
    public function coordinate_optimization_bypass($bypass, $file): bool
    {
        try {
            // Don't bypass optimization for critical images
            if ($this->is_critical_image($file)) {
                CacheSys::writeError("Allowing optimization for critical image: " . basename($file), 'EWWW');
                return false;
            }

            // Bypass for temporary cache files
            if (strpos($file, 'zizi-cache') !== false) {
                CacheSys::writeWarning("Bypassing optimization for ZiziCache temporary file: " . basename($file), 'EWWW');
                return true;
            }

            return $bypass;
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to coordinate optimization bypass: " . $e->getMessage(), 'EWWW');
            return $bypass;
        }
    }

    /**
     * Coordinate WebP delivery with ZiziCache
     */
    public function coordinate_webp_delivery($webp_available): bool
    {
        try {
            // Enhance WebP delivery with ZiziCache coordination
            if ($webp_available && !self::$webp_optimization_enabled) {
                $this->setup_webp_coordination();
                self::$webp_optimization_enabled = true;
                
                CacheSys::writeError("WebP delivery coordination enabled", 'EWWW');
            }

            return $webp_available;
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to coordinate WebP delivery: " . $e->getMessage(), 'EWWW');
            return $webp_available;
        }
    }

    /**
     * Handle WebP cleanup events
     */
    public function handle_webp_cleanup(): void
    {
        try {
            // Clear related ZiziCache entries when WebP files are cleaned
            if (function_exists('wp_cache_flush')) {
                wp_cache_flush();
            }
            
            CacheSys::writeError("WebP cleanup processed - cache invalidated", 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to handle WebP cleanup: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Invalidate cache when images are optimized
     */
    public function invalidate_cache_on_optimization($attachment_id): void
    {
        try {
            if (empty($attachment_id)) {
                return;
            }

            // Clear page caches that contain this image
            $this->invalidate_pages_with_image($attachment_id);
            
            // Clear image-specific caches
            $this->clear_image_cache($attachment_id);
            
            CacheSys::writeError("Cache invalidated for optimized attachment ID: {$attachment_id}", 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to invalidate cache on optimization: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Handle bulk optimization cache management
     */
    public function handle_bulk_optimization_cache(): void
    {
        try {
            // During bulk optimization, defer cache invalidation to prevent performance issues
            if (!wp_next_scheduled('zizi_cache_bulk_optimization_cleanup')) {
                wp_schedule_single_event(time() + 300, 'zizi_cache_bulk_optimization_cleanup');
                
                CacheSys::writeError("Scheduled bulk optimization cache cleanup", 'EWWW');
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to handle bulk optimization cache: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Add performance monitoring
     */
    public function add_performance_monitoring(): void
    {
        if (!current_user_can('manage_options') || is_admin()) {
            return;
        }

        try {
            $stats = $this->get_optimization_stats();
            
            if (!empty($stats['images_optimized']) && $stats['images_optimized'] > 0) {
                echo '<!-- ZiziCache + EWWW: ' . $stats['images_optimized'] . ' images optimized, ' . 
                     number_format($stats['bytes_saved'] / 1024, 2) . 'KB saved -->';
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add performance monitoring: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Show coordination status notices in admin
     */
    public function show_coordination_notices(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        try {
            $coordination_status = get_option('zizi_cache_ewww_coordination_status', []);
            
            if (!empty($coordination_status['conflicts_resolved'])) {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p><strong>ZiziCache + EWWW:</strong> Successfully resolved ' . 
                     $coordination_status['conflicts_resolved'] . ' setting conflicts for optimal performance.</p>';
                echo '</div>';
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to show coordination notices: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Prevent lazy loading conflicts
     */
    public function prevent_lazy_load_conflicts($value): bool
    {
        if ($value && function_exists('is_plugin_active') && is_plugin_active('zizi-cache/zizi-cache.php')) {
            CacheSys::writeWarning("Prevented EWWW lazy loading activation - conflicts with ZiziCache", 'EWWW');
            
            add_action('admin_notices', function() {
                echo '<div class="notice notice-warning"><p>';
                echo '<strong>ZiziCache:</strong> EWWW lazy loading was disabled to prevent conflicts. ';
                echo 'ZiziCache provides superior lazy loading functionality.';
                echo '</p></div>';
            });
            
            return false;
        }
        
        return $value;
    }

    /**
     * Add ZiziCache settings section to EWWW page
     */
    public function add_zizi_cache_settings_section(): void
    {
        try {
            echo '<div class="ewww-zizi-cache-coordination" style="margin: 20px 0; padding: 15px; border: 1px solid #ddd; background: #f9f9f9;">';
            echo '<h3>🚀 ZiziCache Coordination</h3>';
            echo '<p>ZiziCache is actively coordinating with EWWW Image Optimizer for optimal performance:</p>';
            echo '<ul style="margin-left: 20px;">';
            echo '<li>✅ Lazy loading managed by ZiziCache (EWWW lazy loading disabled)</li>';
            echo '<li>✅ WebP delivery optimized and coordinated</li>';
            echo '<li>✅ Cache invalidation synchronized with image optimization</li>';
            echo '<li>✅ Performance monitoring enabled</li>';
            echo '</ul>';
            
            $stats = $this->get_optimization_stats();
            if (!empty($stats['images_optimized'])) {
                echo '<p><strong>Optimization Statistics:</strong> ';
                echo $stats['images_optimized'] . ' images optimized, ';
                echo number_format($stats['bytes_saved'] / 1024, 2) . 'KB saved</p>';
            }
            
            echo '</div>';
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to add ZiziCache settings section: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Check and log potential conflicts during initialization
     */
    private static function checkAndLogConflicts(): void
    {
        try {
            $conflicts_found = 0;
            
            foreach (self::CONFLICTING_SETTINGS as $setting) {
                if (get_option($setting)) {
                    $conflicts_found++;
                }
            }
            
            if ($conflicts_found > 0) {
                CacheSys::writeWarning("Found {$conflicts_found} potential EWWW setting conflicts that will be resolved", 'EWWW');
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to check conflicts: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Coordinate WebP settings for optimal performance
     */
    private function coordinate_webp_settings(): bool
    {
        try {
            $webp_coordinated = false;
            
            // Enable WebP where beneficial but coordinate delivery
            foreach (self::WEBP_SETTINGS as $setting) {
                $current_value = get_option($setting);
                
                // Coordinate WebP generation but let ZiziCache handle delivery
                if ($setting === 'ewww_image_optimizer_webp' && !$current_value) {
                    update_option($setting, true);
                    $webp_coordinated = true;
                    CacheSys::writeError("Enabled coordinated WebP generation: {$setting}", 'EWWW');
                }
            }
            
            return $webp_coordinated;
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to coordinate WebP settings: " . $e->getMessage(), 'EWWW');
            return false;
        }
    }

    /**
     * Enable EWWW features that are compatible with ZiziCache
     */
    private function enable_compatible_features(): void
    {
        try {
            $compatible_settings = [
                'ewww_image_optimizer_metadata_remove' => true,
                'ewww_image_optimizer_jpg_quality' => 90,
                'ewww_image_optimizer_png_level' => 6,
                'ewww_image_optimizer_backup_files' => true
            ];
            
            foreach ($compatible_settings as $setting => $value) {
                if (get_option($setting) === false) {
                    update_option($setting, $value);
                    CacheSys::writeError("Enabled compatible EWWW feature: {$setting}", 'EWWW');
                }
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to enable compatible features: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Setup WebP coordination mechanisms
     */
    private function setup_webp_coordination(): void
    {
        try {
            // Add WebP delivery hooks that work with ZiziCache
            add_filter('wp_get_attachment_image_src', [$this, 'coordinate_webp_urls'], 10, 4);
            add_filter('wp_calculate_image_srcset', [$this, 'coordinate_webp_srcset'], 10, 5);
            
            CacheSys::writeError("WebP coordination mechanisms setup completed", 'EWWW');
            
        } catch (\Throwable $e) {
            CacheSys::writeError("Failed to setup WebP coordination: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Coordinate WebP URLs with ZiziCache
     */
    public function coordinate_webp_urls($image, $attachment_id, $size, $icon): array
    {
        try {
            if (!empty($image[0]) && $this->should_use_webp($image[0])) {
                // Let ZiziCache handle WebP delivery coordination
                $image['zizi_webp_eligible'] = true;
            }
            
            return $image;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to coordinate WebP URLs: " . $e->getMessage(), 'EWWW');
            return $image;
        }
    }

    /**
     * Coordinate WebP srcset with ZiziCache
     */
    public function coordinate_webp_srcset($sources, $size_array, $image_src, $image_meta, $attachment_id): array
    {
        try {
            if (!empty($sources) && is_array($sources)) {
                foreach ($sources as &$source) {
                    if ($this->should_use_webp($source['url'])) {
                        $source['zizi_webp_eligible'] = true;
                    }
                }
            }
            
            return $sources;
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to coordinate WebP srcset: " . $e->getMessage(), 'EWWW');
            return $sources;
        }
    }

    /**
     * Check if image is critical for performance
     */
    private function is_critical_image(string $file): bool
    {
        $critical_patterns = [
            '/logo/',
            '/header/',
            '/hero/',
            '/banner/',
            '/favicon/',
            '/above-fold/'
        ];
        
        foreach ($critical_patterns as $pattern) {
            if (preg_match($pattern, $file)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if WebP should be used for given URL
     */
    private function should_use_webp(string $url): bool
    {
        return (
            strpos($url, '.jpg') !== false || 
            strpos($url, '.jpeg') !== false || 
            strpos($url, '.png') !== false
        ) && function_exists('ewww_image_optimizer_webp_available') && 
        ewww_image_optimizer_webp_available();
    }

    /**
     * Invalidate caches for images
     */
    private function invalidate_image_caches(string $file): void
    {
        try {
            // Clear object cache
            if (function_exists('wp_cache_delete')) {
                wp_cache_delete('zizi_image_' . md5($file), 'zizi_cache');
            }
            
            // Clear page cache for pages containing this image
            $this->clear_pages_with_image($file);
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to invalidate image caches: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Clear page caches containing specific image
     */
    private function clear_pages_with_image(string $file): void
    {
        // Implementation would depend on ZiziCache's cache storage system
        // This is a placeholder for the actual cache clearing logic
    }

    /**
     * Invalidate pages containing specific attachment
     */
    private function invalidate_pages_with_image(int $attachment_id): void
    {
        try {
            // Find posts using this attachment
            $posts = get_posts([
                'post_type' => 'any',
                'meta_query' => [
                    [
                        'key' => '_thumbnail_id',
                        'value' => $attachment_id,
                        'compare' => '='
                    ]
                ],
                'fields' => 'ids'
            ]);
            
            foreach ($posts as $post_id) {
                // Clear cache for each post
                if (function_exists('wp_cache_delete')) {
                    wp_cache_delete('zizi_page_' . $post_id, 'zizi_cache');
                }
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to invalidate pages with image: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Clear image-specific cache
     */
    private function clear_image_cache(int $attachment_id): void
    {
        try {
            if (function_exists('wp_cache_delete')) {
                wp_cache_delete('zizi_attachment_' . $attachment_id, 'zizi_cache');
                wp_cache_delete('zizi_image_meta_' . $attachment_id, 'zizi_cache');
            }
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to clear image cache: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Update optimization statistics
     */
    private function update_optimization_stats(array $optimized_data): void
    {
        try {
            $stats = get_option('zizi_cache_ewww_stats', [
                'images_optimized' => 0,
                'bytes_saved' => 0,
                'last_optimization' => 0
            ]);
            
            $stats['images_optimized']++;
            if (!empty($optimized_data['bytes_saved'])) {
                $stats['bytes_saved'] += (int)$optimized_data['bytes_saved'];
            }
            $stats['last_optimization'] = time();
            
            update_option('zizi_cache_ewww_stats', $stats);
            
        } catch (\Throwable $e) {
            CacheSys::writeWarning("Failed to update optimization stats: " . $e->getMessage(), 'EWWW');
        }
    }

    /**
     * Get optimization statistics
     */
    private function get_optimization_stats(): array
    {
        return get_option('zizi_cache_ewww_stats', [
            'images_optimized' => 0,
            'bytes_saved' => 0,
            'last_optimization' => 0
        ]);
    }
}
