<?php
namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use Exception;

/**
 * Integration for BuddyBoss theme and platform.
 *
 * Handles endpoint exclusions and disables conflicting options for BuddyBoss.
 *
 * @package ZiziCache\Plugins\Integrations
 */
class BuddyBoss extends PluginBase {
    use PluginTrait;

    /**
     * Check if BuddyBoss integration is available (plugin is active).
     *
     * @return bool True if BuddyBoss is available, false otherwise
     */
    protected static function checkAvailability(): bool {        return class_exists('buddyboss_theme_Redux_Framework_config');
    }

    /**
     * Initialize the integration by registering filters and actions for BuddyBoss.
     *
     * @return void
     */
    public static function init(): void {
        // Enhanced security check - only administrators can manage cache settings
        if (!current_user_can('manage_options')) {
            return;
        }

        // Verify BuddyBoss availability before proceeding
        if (!static::isAvailable()) {
            return;
        }        try {
            add_filter('bb_exclude_endpoints_from_restriction', [__CLASS__, 'excludedEndpoints']);
            static::addAction('zizi_cache_update_config:after', 'disableConflictingOptions');
            
            \ZiziCache\CacheSys::writeLog('Integration initialized successfully', '', 'BuddyBoss');
        } catch (Exception $e) {
            \ZiziCache\CacheSys::writeError('Integration initialization failed - ' . $e->getMessage(), 'BuddyBoss');        }
    }

    /**
     * Add ZiziCache endpoints to BuddyBoss exclusion list.
     * These endpoints need to be excluded from BuddyBoss restrictions to ensure proper functionality.
     *
     * @param array $endpoints Existing endpoints
     * @return array Modified endpoints with ZiziCache endpoints added
     */
    public static function excludedEndpoints($endpoints) {
        if (!is_array($endpoints)) {
            $endpoints = [];
        }

        try {
            $zizi_endpoints = [
                '/zizi-cache/config',
                '/zizi-cache/cached-pages-count',
                '/zizi-cache/purge-current-page',
                '/zizi-cache/preload-cache',
                '/zizi-cache/purge-pages',
                '/zizi-cache/purge-pages-and-preload',
                '/zizi-cache/purge-everything-and-preload',
                '/zizi-cache/activate-license',
            ];
              $merged_endpoints = array_merge($endpoints, $zizi_endpoints);
            \ZiziCache\CacheSys::writeLog('Successfully added ' . count($zizi_endpoints) . ' endpoints to exclusion list', '', 'BuddyBoss');
            
            return $merged_endpoints;
        } catch (Exception $e) {
            \ZiziCache\CacheSys::writeError('Error adding endpoints to exclusion list - ' . $e->getMessage(), 'BuddyBoss');
            return $endpoints;        }
    }

    /**
     * Disable BuddyBoss options that conflict with ZiziCache optimization.
     * 
     * BuddyBoss theme has built-in CSS/JS minification that can conflict with 
     * ZiziCache's advanced optimization algorithms. This method ensures optimal
     * performance by disabling conflicting BuddyBoss optimizations.
     *
     * @return void
     */
    public static function disableConflictingOptions(): void {        // Double security validation - enterprise grade security
        if (!current_user_can('manage_options')) {
            \ZiziCache\CacheSys::writeWarning('Conflict resolution blocked - insufficient permissions', 'BuddyBoss');
            return;
        }

        // Verify BuddyBoss is available before attempting to modify options
        if (!class_exists('buddyboss_theme_Redux_Framework_config')) {
            \ZiziCache\CacheSys::writeWarning('Conflict resolution skipped - theme not available', 'BuddyBoss');
            return;
        }

        try {
            // Get current BuddyBoss theme options
            $options = get_option('buddyboss_theme_options');
            
            if (!is_array($options)) {
                \ZiziCache\CacheSys::writeWarning('Options not found or invalid format', 'BuddyBoss');
                return;
            }

            $changes_made = false;
            $disabled_options = [];            // Disable CSS minification if enabled (conflicts with ZiziCache CSS optimization)
            if (isset($options['boss_minified_css']) && $options['boss_minified_css']) {
                $options['boss_minified_css'] = 0;
                $changes_made = true;
                $disabled_options[] = 'CSS Minification';
            }

            // Disable JS minification if enabled (conflicts with ZiziCache JS optimization)
            if (isset($options['boss_minified_js']) && $options['boss_minified_js']) {
                $options['boss_minified_js'] = 0;
                $changes_made = true;
                $disabled_options[] = 'JavaScript Minification';
            }

            // Apply changes if any conflicts were found and resolved
            if ($changes_made) {
                $update_result = update_option('buddyboss_theme_options', $options);
                
                if ($update_result) {
                    $disabled_count = count($disabled_options);
                    $disabled_list = implode(', ', $disabled_options);
                    \ZiziCache\CacheSys::writeLog("Successfully disabled {$disabled_count} conflicting options: {$disabled_list}", '', 'BuddyBoss');
                } else {
                    \ZiziCache\CacheSys::writeError('Failed to update theme options', 'BuddyBoss');
                }
            }        } catch (Exception $e) {
            \ZiziCache\CacheSys::writeError('Critical error in conflict resolution - ' . $e->getMessage(), 'BuddyBoss');
        }
    }    /**
     * Get the integration version.
     *
     * @return string Version number
     */
    public static function getVersion(): string {
        return '2.0.0';
    }

    /**
     * Get the integration description.
     *
     * @return string Description
     */
    public static function getDescription(): string {
        return 'Enterprise-grade BuddyBoss theme integration with advanced conflict resolution and endpoint management for optimal ZiziCache performance.';
    }
}
